//
// NAME: CIDKernel_Environment.Cpp
//
// DESCRIPTION:
//
//	This module implements the TKrnlEnvironment class.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 08/15/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"


// ----------------------------------------------------------------------------
//  Local const data
//
//  __c4Modulus
//      The modulus used in the hashing of the key strings. This is passed
//      along to the hash map object.
// ----------------------------------------------------------------------------
static const tCIDLib::TCard4    __c4Modulus = 23;


// ----------------------------------------------------------------------------
//  Local static data
//
//  __pkcrsAccess
//      This is a critical section used to sync access to the environment
//      data across threads.
//
//  __pkhshmEnv
//      The kernel hash map structure used to hold the environmental
//      variables. Its allocated during module init.
// ----------------------------------------------------------------------------
static TKrnlCritSec*            __pkcrsAccess = 0;
static TKrnlHashMap*            __pkhshmEnv = 0;



// ----------------------------------------------------------------------------
//  Intra-facility functions
// ----------------------------------------------------------------------------

tCIDLib::TVoid
_InitTermEnvironment(   const   tCIDLib::EInitTerm      eInitTerm
                        , const tCIDLib::EGlobalStates  eGlobals
                        , const tCIDLib::TModHandle     hmodThis
                        , const tCIDLib::TCard4         c4MaxChars
                        ,       tCIDLib::Tch* const     pszFailReason)
{
    const tCIDLib::Tch* pszReason = kMessages::pszGen_Unknown;

    if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
    &&  (eGlobals == tCIDLib::EGlobalState_Before))
    {
        tCIDLib::Tch* pszRawEnv = 0;
        try
        {
            // Allocate our critical section and our hash map objects.
            pszReason = kMessages::pszEnv_CreateCritSec;
            __pkcrsAccess = new TKrnlCritSec;

            pszReason = kMessages::pszEnv_CreateHashMap;
            __pkhshmEnv = new TKrnlHashMap(__c4Modulus);

            //
            //  Get a pointer to the raw environment buffer from the OS.
            //  We will run through it ourself.
            //
            pszRawEnv = (tCIDLib::Tch*)GetEnvironmentStringsW();

            //
            //  Get a temp pointer to it that we can run through the buffer.
            //  Bump it past the first char, which seems to be junk.
            //
            const tCIDLib::Tch* pszCur = pszRawEnv;
            pszCur++;

            //
            //  Now load up the environment with all of the environment
            //  strings that are set.
            //
            pszReason = kMessages::pszEnv_LoadEnv;
            tCIDLib::TZStr512   szTmp;

            tCIDLib::TCard4 c4Count = 0;
            while (*pszCur)
            {
                // Make a copy of the current element
                TRawStr::CopyStr(szTmp, pszCur, c4MaxBufChars(szTmp));

                //
                //  We need to find the two parts of the string and pull
                //  it into its key and value sections.
                //
                tCIDLib::Tch* pszSplit = TRawStr::pszFindLastChar
                (
                    szTmp
                    , kCIDLib::chEquals
                );

                // This is bad, and should not be able to happen
                if (!pszSplit)
                    TKrnlError::ThrowKrnlError(kKrnlErrors::errcBadEnvironment);

                // Put a null at the split to create two strings
                *pszSplit = kCIDLib::chNull;

                // And add them now to the hash map
                __pkhshmEnv->Add(szTmp, pszSplit+1);

                // And bump up the pointer to next string
                pszCur += TRawStr::c4StrLen(pszCur)+1;

                c4Count++;
            }

            // Free the raw env buffer
            FreeEnvironmentStringsW(pszRawEnv);
        }

        catch(...)
        {
            // Make sure we get the environment block freed
            FreeEnvironmentStrings(pszRawEnv);

            TRawStr::CopyStr(pszFailReason, pszReason, c4MaxChars);
            throw;
        }
    }
     else if ((eInitTerm == tCIDLib::EInitTerm_Terminate)
          &&  (eGlobals == tCIDLib::EGlobalState_After))
    {
        try
        {
            // Allocate our critical section and our hash map objects.
            pszReason = kMessages::pszEnv_DeleteCritSec;
            delete __pkcrsAccess;

            pszReason = kMessages::pszEnv_DeleteHashMap;
            delete __pkhshmEnv;
        }

        catch(...)
        {
            TRawStr::CopyStr(pszFailReason, pszReason, c4MaxChars);
            throw;
        }
    }
}



// ----------------------------------------------------------------------------
//   CLASS: TKrnlEnvironment
//  PREFIX: kenv
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlEnvironment: Public, static methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TKrnlEnvironment::Add(  const   tCIDLib::Tch* const pszKey
                        , const tCIDLib::Tch* const pszNewValue)
{
    TKrnlCritSecLocker crslAccess(__pkcrsAccess);
    __pkhshmEnv->Add(pszKey, pszNewValue);
}


tCIDLib::TBoolean
TKrnlEnvironment::bAddOrUpdate( const   tCIDLib::Tch* const pszKey
                                , const tCIDLib::Tch* const pszNewValue)
{
    TKrnlCritSecLocker crslAccess(__pkcrsAccess);

    // See if this element is in the map
    try
    {
        __pkhshmEnv->UpdateKey(pszKey, pszNewValue);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        // If anything other than not found, then rethrow
        if (kerrToCatch.errcId() != kKrnlErrors::errcNotFound)
            throw;

        // Add the new value
        __pkhshmEnv->Add(pszKey, pszNewValue);

        // Return true to say it was added
        return kCIDLib::True;
    }

    // Return false since it was not added
    return kCIDLib::False;
}

tCIDLib::TBoolean
TKrnlEnvironment::bFind(const   tCIDLib::Tch* const pszKey
                        ,       tCIDLib::Tch* const pszToFill
                        , const tCIDLib::TCard4     c4MaxChars)
{
    TKrnlCritSecLocker crslAccess(__pkcrsAccess);

    // See if this element is in the map
    try
    {
        const tCIDLib::Tch* pszValue = __pkhshmEnv->pszValueForKey(pszKey);
        TRawStr::CopyStr(pszToFill, pszValue, c4MaxChars);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        if (kerrToCatch.errcId() != kKrnlErrors::errcNotFound)
            throw;

        return kCIDLib::False;
    }
    return kCIDLib::True;
}

tCIDLib::TCard4 TKrnlEnvironment::c4Entries()
{
    tCIDLib::TCard4 c4Count;
    {
        TKrnlCritSecLocker crslAccess(__pkcrsAccess);
        c4Count = __pkhshmEnv->c4ElemCount();
    }
    return c4Count;
}

tCIDLib::TCard4 TKrnlEnvironment::c4QueryState(TElem*& pelemToFill)
{
    TKrnlCritSecLocker crslAccess(__pkcrsAccess);

    //
    //  Allocate the array of environment elements, large enough to hold
    //  all our entries.
    //
    tCIDLib::TCard4 c4Count = __pkhshmEnv->c4ElemCount();
    if (!c4Count)
        return 0;

    pelemToFill = new TElem[c4Count];

    //
    //  We have at least one element so iterate the map and fill in
    //  a record for each one.
    //
    __pkhshmEnv->bResetIter();
    tCIDLib::TCard4 c4ActualCount = 0;
    do
    {
        pelemToFill[c4ActualCount].Set
        (
            __pkhshmEnv->pszCurKey()
            , __pkhshmEnv->pszCurValue()
        );
        c4ActualCount++;
    }   while (__pkhshmEnv->bNext());

    if (c4ActualCount != c4Count)
    {
        delete [] pelemToFill;
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcBadEnvironment);
    }

    return c4Count;
}

tCIDLib::TCard4
TKrnlEnvironment::chCharsInValue(const tCIDLib::Tch* const pszKey)
{
    tCIDLib::TCard4 c4Chars;
    {
        TKrnlCritSecLocker crslAccess(__pkcrsAccess);

        //
        //  See if this element is in the map. If not, then a not found
        //  exception is thrown.
        //
        const tCIDLib::Tch* pszValue = __pkhshmEnv->pszValueForKey(pszKey);
        c4Chars = TRawStr::c4StrLen(pszValue);
    }
    return c4Chars;
}

tCIDLib::TVoid TKrnlEnvironment::Lock()
{
    __pkcrsAccess->Enter();
}

tCIDLib::Tch* TKrnlEnvironment::pszFind(const tCIDLib::Tch* const pszKey)
{
    TKrnlCritSecLocker crslAccess(__pkcrsAccess);

    // See if this element is in the map
    tCIDLib::Tch* pszRet;
    try
    {
        const tCIDLib::Tch* pszValue = __pkhshmEnv->pszValueForKey(pszKey);
        pszRet = TRawStr::pszReplicate(pszValue);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        // If not found, return 0
        if (kerrToCatch.errcId() == kKrnlErrors::errcNotFound)
            return 0;

        // It was some other error, so pass it on
        throw;
    }
    return pszRet;
}

tCIDLib::TVoid TKrnlEnvironment::Unlock()
{
    __pkcrsAccess->Exit();
}

tCIDLib::TVoid
TKrnlEnvironment::Update(   const   tCIDLib::Tch* const pszKey
                            , const tCIDLib::Tch* const pszNewValue)
{
    TKrnlCritSecLocker crslAccess(__pkcrsAccess);

    //
    //  See if this element is in the map. If not, a not found exception
    //  is thrown.
    //
    __pkhshmEnv->UpdateKey(pszKey, pszNewValue);
}
