//
// NAME: CIDKernel_Console.Cpp
//
// DESCRIPTION:
//
//	This module implements the TKrnlConsole class, which represents the
//  physical console input device for the host OS. This hides the ins and
//  outs of how input gets and to and from the raw console. It supports
//  command line editing and recall on line input.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 11/10/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"
#include    "CIDKernel_ScanCodes.Hpp"


// ----------------------------------------------------------------------------
//   CLASS: TKrnlConsole
//  PREFIX: kcon
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TKrnlConsole: Constructors and destructors
// ----------------------------------------------------------------------------

TKrnlConsole::TKrnlConsole() :

    __apszRecallBuf(0)
    , __bFirstRecall(kCIDLib::True)
    , __bInsertMode(kCIDLib::False)
    , __bRedirIn(kCIDLib::False)
    , __bRedirOut(kCIDLib::False)
    , __c4CharCount(0)
    , __c4CursorPos(0)
    , __c4MaxRecall(0)
    , __c4RecallIndex(0)
    , __eTextFormat(tCIDLib::ETextFmt_UNICode)
    , __fposCurPos(0)
    , __hflIn(kCIDLib::hflInvalid)
    , __hflOut(kCIDLib::hflInvalid)
{
}

TKrnlConsole::TKrnlConsole(const tCIDLib::ETextFormats eTextFormat) :

    __apszRecallBuf(0)
    , __bFirstRecall(kCIDLib::True)
    , __bInsertMode(kCIDLib::False)
    , __bRedirIn(kCIDLib::False)
    , __bRedirOut(kCIDLib::False)
    , __c4CharCount(0)
    , __c4CursorPos(0)
    , __c4MaxRecall(0)
    , __c4RecallIndex(0)
    , __eTextFormat(eTextFormat)
    , __fposCurPos(0)
    , __hflIn(kCIDLib::hflInvalid)
    , __hflOut(kCIDLib::hflInvalid)
{
}

TKrnlConsole::TKrnlConsole( const   tCIDLib::TBoolean       bInsertMode
                            , const tCIDLib::TCard4         c4MaxRecall
                            , const tCIDLib::ETextFormats   eTextFormat
                            , const   tCIDLib::ERedir       eRedirect) :
    __apszRecallBuf(0)
    , __bFirstRecall(kCIDLib::True)
    , __bInsertMode(bInsertMode)
    , __bRedirIn(eRedirect == tCIDLib::ERedir_Allow)
    , __bRedirOut(eRedirect == tCIDLib::ERedir_Allow)
    , __c4CharCount(0)
    , __c4CursorPos(0)
    , __c4MaxRecall(c4MaxRecall)
    , __c4RecallIndex(0)
    , __eTextFormat(eTextFormat)
    , __fposCurPos(0)
    , __hflIn(kCIDLib::hflInvalid)
    , __hflOut(kCIDLib::hflInvalid)

{
    __OpenFiles(eRedirect);

    if (__c4MaxRecall)
    {
        __apszRecallBuf = new tCIDLib::Tch*[__c4MaxRecall];

        //
        //  Note the fill must be typed because there are multiple versions
        //  that will fill with different sized values.
        //
        TRawMem::SetMemBuf(__apszRecallBuf, tCIDLib::TCard4(0), __c4MaxRecall);
    }
}

TKrnlConsole::~TKrnlConsole()
{
    // Close the handles
    if (__hflIn != kCIDLib::hflInvalid)
        CloseHandle(__hflIn);

    if (__hflOut != kCIDLib::hflInvalid)
        CloseHandle(__hflOut);

    if (__apszRecallBuf)
    {
        // Delete any string points we might have
        for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4MaxRecall; c4Ind++)
            delete __apszRecallBuf[c4Ind];

        // Then delete the buffer itself
        delete __apszRecallBuf;
    }
}


// ----------------------------------------------------------------------------
//  TKrnlConsole: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TBoolean TKrnlConsole::bAtEnd() const
{
    // If not redirected, then we need hit the end
    if (!__bRedirIn)
        kCIDLib::False;

    //
    //  Get the current position of the input file and its current size and
    //  see if we are at the end.
    //
    tCIDLib::TInt8      i8Tmp;
    tCIDLib::TCard4*    pc4Low  = (tCIDLib::TCard4*)&i8Tmp;
    tCIDLib::TCard4*    pc4High = (tCIDLib::TCard4*)&i8Tmp;
    pc4High++;

    // Do a 0 seek from the current to get the current position
    i8Tmp = 0;
    tCIDLib::TCard4 c4Low = SetFilePointer
    (
        __hflIn
        , *pc4Low
        , (long*)pc4High
        , tCIDLib::ESeekFrom_Current
    );

    if (c4Low == kCIDLib::c4MaxCard)
    {
        if (TKrnlThread::errcGetLast())
            TKrnlError::ThrowKrnlError();
    }

    i8Tmp |= c4Low;
    tCIDLib::TFilePos fposCur = i8Tmp;

    //
    //  Now query the size of the file, this tells us what the end of the
    //  available data is.
    //
    i8Tmp = 0;
    c4Low = GetFileSize(__hflIn, pc4High);
    if (c4Low == kCIDLib::c4MaxCard)
    {
        if (TKrnlThread::errcGetLast())
            TKrnlError::ThrowKrnlError();
    }
    i8Tmp |= c4Low;

    // Return true if current is at or beyond size
    return (fposCur >= i8Tmp);
}


tCIDLib::TCard4
TKrnlConsole::c4GetLine(        tCIDLib::Tch* const pszBufToFill
                        , const tCIDLib::TCard4     c4MaxChars
                        , const tCIDLib::TBoolean   bStripWhitespace)
{
    //
    //  Initialize our character count and cursor position flags for an
    //  empty string.
    //
    __c4CharCount = 0;
    __c4CursorPos = 0;

    // And make sure the passed buffer starts empty
    pszBufToFill[0] = 0;

    tCIDLib::Tch    chCur;
    tCIDLib::TCard4 c4TmpInd;
    while (1)
    {
        // Get a character from the console
        chCur = chRead();

        //
        //  If its one of the prefix chars, then this is an extended
        //  key and we have to do more work. Otherwise, its just a normal
        //  character (but could be one of the non-extended editing keys.)
        //
        if ((chCur == Key_PadPrefix) || (chCur == Key_ExtPrefix))
        {
            // Get the real character
            chCur = chRead();

            //
            //  Output the current recall buffer, copy it to the user input
            //  buffer, and set the current input index to the end of the
            //  string.
            //
            if (((chCur == Key_Down) || (chCur == Key_Up)) && __apszRecallBuf)
            {
                //
                //  Look for the next/prev recall string from the current
                //  index. If none, then done. Handle wrapping back around.
                //
                if (chCur == Key_Down)
                {
                    if (__bFirstRecall)
                    {
                        __c4RecallIndex = 0;
                        __bFirstRecall = kCIDLib::False;
                    }

                    c4TmpInd = __c4RecallIndex;
                    do
                    {
                        // Find the previous index
                        if (!__c4RecallIndex)
                            __c4RecallIndex = __c4MaxRecall - 1;
                        else
                            __c4RecallIndex--;

                        // If there is something here, then break out
                        if (__apszRecallBuf[__c4RecallIndex])
                            break;
                    }   while (__c4RecallIndex != c4TmpInd);
                }
                 else if (chCur == Key_Up)
                {
                    if (__bFirstRecall)
                    {
                        __c4RecallIndex = __c4MaxRecall-1;
                        __bFirstRecall = kCIDLib::False;
                    }

                    c4TmpInd = __c4RecallIndex;
                    do
                    {
                        if (__c4RecallIndex == __c4MaxRecall-1)
                            __c4RecallIndex = 0;
                        else
                            __c4RecallIndex++;

                        if (__apszRecallBuf[__c4RecallIndex])
                            break;
                    }   while (__c4RecallIndex != c4TmpInd);
                }

                //
                //  If we went all the way around, its empty and nothing
                //  to do. Otherwise, we need to make the newly selected
                //  buffer the current input.
                //
                if (__c4RecallIndex != c4TmpInd)
                {
                    __ClearInput(__c4CursorPos, __c4CharCount);
                    __c4CharCount = 0;
                    __c4CursorPos = 0;

                    //
                    //  Output the selected recall and make it the input
                    //  string. Put the cursor at the end of it.
                    //
                    TRawStr::CopyStr
                    (
                        pszBufToFill
                        , __apszRecallBuf[__c4RecallIndex]
                        , c4MaxChars
                    );
                    PutLine(pszBufToFill);
                    __c4CharCount =
                            TRawStr::c4StrLen(__apszRecallBuf[__c4RecallIndex]);
                    __c4CursorPos = __c4CharCount;
                }
            }
             else if ((chCur == Key_CtrlRight) || (chCur == Key_CtrlLeft))
            {
                __NextPrevWord
                (
                    pszBufToFill
                    , __c4CursorPos
                    , __c4CharCount
                    , chCur
                );
            }
             else if (chCur == Key_Delete)
            {
                // If not at the end, then delete
                if (__c4CursorPos != __c4CharCount)
                {
                    __DelInpChar
                    (
                        pszBufToFill
                        , __c4CharCount
                        , __c4CursorPos
                        , kCIDLib::True
                    );
                    __c4CharCount--;
                }
            }
             else if (chCur == Key_End)
            {
                for (; __c4CursorPos < __c4CharCount; __c4CursorPos++)
                    PutCh(pszBufToFill[__c4CursorPos]);
            }
             else if (chCur == Key_Home)
            {
                for (; __c4CursorPos > 0; __c4CursorPos--)
                    PutCh(Key_BS);
            }
             else if (chCur == Key_Insert)
            {
                if (__bInsertMode)
                    __bInsertMode = kCIDLib::False;
                else
                    __bInsertMode = kCIDLib::True;
            }
             else if (chCur == Key_Left)
            {
                //
                //  Move the current position to the left if we are
                //  not already at the start.
                //
                if (__c4CursorPos)
                {
                    __c4CursorPos--;
                    PutCh(Key_BS);
                }
            }
             else if (chCur == Key_Right)
            {
                //
                //  Move the current position to the right if we are
                //  not already at the end. Output the character at
                //  this position. This is kind of redundant since we
                //  are overwriting the same char, but we have to move
                //  the cursor forward somehow.
                //
                if (__c4CharCount && (__c4CursorPos < __c4CharCount))
                {
                    PutCh(pszBufToFill[__c4CursorPos]);
                    __c4CursorPos++;
                }
            }
        }
         else
        {
            //
            //  Its either a normal character to input, or its one of
            //  non-extended editing keys.
            //
            if (chCur == Key_BS)
            {
                if (__c4CursorPos)
                {
                    if (__c4CursorPos == __c4CharCount)
                    {
                        PutCh(chCur);
                        PutCh(Key_Space);
                        PutCh(chCur);
                    }
                     else
                    {
                        __DelInpChar
                        (
                            pszBufToFill
                            , __c4CharCount
                            , __c4CursorPos
                            , kCIDLib::False
                        );
                    }
                    __c4CharCount--;
                    __c4CursorPos--;
                }
            }
             else if (chCur == Key_Enter)
            {
                // Null terminate the input
                pszBufToFill[__c4CharCount] = 0;

                //
                //  Strip whitespace. We use an internal stripper for
                //  raw strings.
                //
                if (bStripWhitespace)
                {
                    TRawStr::StripStr
                    (
                        pszBufToFill
                        , kCIDLib::szWhitespace
                        , tCIDLib::EStripMode_LeadTrail
                    );
                    __c4CharCount = TRawStr::c4StrLen(pszBufToFill);
                }

                //
                //  Add it to the recall list, if there is one and the
                //  string was not just whitespace or empty.
                //
                if (__apszRecallBuf && __c4CharCount)
                    __NewEntry(pszBufToFill);

                // Output a new line
                PutCh(kCIDLib::chCR);
                PutCh(kCIDLib::chLF);

                //
                //  Return the chars in the string. Add it to current pos.
                //
                __fposCurPos += __c4CharCount;
                return __c4CharCount;
            }
             else if (chCur == Key_CtrlC)
            {
                pszBufToFill[0] = kCIDLib::chNull;
                return 0;
            }
             else if (chCur == Key_Escape)
            {
                // Clear out the input and put the cursor back at home
                __ClearInput(__c4CursorPos, __c4CharCount);
                __c4CharCount = 0;
                __c4CursorPos = 0;
            }
             else if ((chCur == Key_Tab) || (chCur == Key_BackTab))
            {
                TKrnlThread::BeepSpeaker(440, 50);
            }
             else
            {
                //
                //  If we are full, then beep at the user. Otherwise, output
                //  the char and add it to the user buffer.
                //
                if ((__bInsertMode && (__c4CharCount == c4MaxChars))
                ||  (!__bInsertMode && (__c4CursorPos == c4MaxChars)))
                {
                    TKrnlThread::BeepSpeaker(440, 50);
                    continue;
                }

                //
                //  We either need to insert or overwrite here. If at
                //  the end, then its not an issue so do that first.
                //
                if (__c4CursorPos == __c4CharCount)
                {
                    pszBufToFill[__c4CharCount++] = chCur;

                    // Echo to the screen
                    PutCh(chCur);
                }
                 else
                {
                    if (__bInsertMode)
                    {
                        //
                        //  Move the characters all up in the string, then
                        //  display them, then backspace again to where we
                        //  need to be.
                        //
                        for (c4TmpInd = __c4CharCount; c4TmpInd > __c4CursorPos; c4TmpInd--)
                            pszBufToFill[c4TmpInd] = pszBufToFill[c4TmpInd-1];

                        pszBufToFill[__c4CursorPos] = chCur;

                        // Bump up the chart count now
                        __c4CharCount++;

                        // Display all of the new chars
                        for (c4TmpInd = __c4CursorPos; c4TmpInd < __c4CharCount; c4TmpInd++)
                            PutCh(pszBufToFill[c4TmpInd]);

                        // Put out a space to kill the last char
                        PutCh(Key_Space);

                        // And backspace again
                        for (c4TmpInd = __c4CharCount; c4TmpInd > __c4CursorPos; c4TmpInd--)
                            PutCh(Key_BS);
                    }
                     else
                    {
                        pszBufToFill[__c4CursorPos] = chCur;
                        PutCh(chCur);
                    }
                }

                // Move the cursor position up 1
                __c4CursorPos++;
            }
        }
    }

    // Make the compiler happy, but we won't really get here
    return 0;
}

tCIDLib::Tch TKrnlConsole::chRead()
{
    tCIDLib::Tch chRet = __chGet();
    __fposCurPos++;
    return chRet;
}


tCIDLib::TFilePos TKrnlConsole::fposCurPos() const
{
    if (__bRedirIn)
    {
        tCIDLib::TInt8      i8Tmp = 0;
        tCIDLib::TCard4*    pc4Low  = (tCIDLib::TCard4*)&i8Tmp;
        tCIDLib::TInt4*     pi4High = (tCIDLib::TInt4*)&i8Tmp;
        pi4High++;

        tCIDLib::TCard4 c4Low = SetFilePointer
        (
            __hflIn
            , *pc4Low
            , pi4High
            , tCIDLib::ESeekFrom_Current
        );

        // See if it there was an error or just a position over 32 bits
        if (c4Low == kCIDLib::c4MaxCard)
        {
            if (TKrnlThread::errcGetLast())
                TKrnlError::ThrowKrnlError();
        }

        i8Tmp |= c4Low;
        return i8Tmp;
    }
     else
    {
        return __fposCurPos;
    }
}


tCIDLib::TFilePos TKrnlConsole::fposLogicalEnd() const
{
    if (__bRedirIn)
    {
        // Query the size of the input file
        tCIDLib::TInt8      i8Tmp;
        tCIDLib::TCard4*    pc4Low  = (tCIDLib::TCard4*)&i8Tmp;
        tCIDLib::TCard4*    pc4High = (tCIDLib::TCard4*)&i8Tmp;
        pc4High++;

        tCIDLib::TCard4 c4Low = GetFileSize(__hflIn, pc4High);
        if (c4Low == kCIDLib::c4MaxCard)
        {
            if (TKrnlThread::errcGetLast())
                TKrnlError::ThrowKrnlError();
        }
        i8Tmp |= c4Low;

        // If UNICode then divide size by bytes per char
        if (__eTextFormat == tCIDLib::ETextFmt_UNICode)
            i8Tmp /= kCIDLib::c4CharBytes;

        return i8Tmp;
    }
    return kCIDLib::i8MaxInt;
}


tCIDLib::TFilePos TKrnlConsole::fposPhysicalEnd() const
{
    if (__bRedirOut)
    {
        // Query the size of the output file
        tCIDLib::TInt8      i8Tmp;
        tCIDLib::TCard4*    pc4Low  = (tCIDLib::TCard4*)&i8Tmp;
        tCIDLib::TCard4*    pc4High = (tCIDLib::TCard4*)&i8Tmp;
        pc4High++;

        tCIDLib::TCard4 c4Low = GetFileSize(__hflOut, pc4High);
        if (c4Low == kCIDLib::c4MaxCard)
        {
            if (TKrnlThread::errcGetLast())
                TKrnlError::ThrowKrnlError();
        }
        i8Tmp |= c4Low;

        // If UNICode then divide size by bytes per char
        if (__eTextFormat == tCIDLib::ETextFmt_UNICode)
            i8Tmp /= kCIDLib::c4CharBytes;

        return i8Tmp;
    }
    return kCIDLib::i8MaxInt;
}


tCIDLib::TFilePos
TKrnlConsole::fposSkipForwardBy(const tCIDLib::TFilePos fposSkipBy)
{
    tCIDLib::TCard4     c4Low;
    tCIDLib::TInt8      i8Tmp;
    tCIDLib::TCard4*    pc4Low  = (tCIDLib::TCard4*)&i8Tmp;
    tCIDLib::TInt4*     pi4High = (tCIDLib::TInt4*)&i8Tmp;
    pi4High++;

    //
    //  If the input file has been redirected, then we need to move
    //  foward the indicated number of bytes.
    //
    if (__bRedirIn)
    {
        // Get the offset into a temp
        i8Tmp = fposSkipBy;

        c4Low = SetFilePointer
        (
            __hflIn
            , *pc4Low
            , pi4High
            , tCIDLib::ESeekFrom_Current
        );

        // See if it there was an error or just a position over 32 bits
        if (c4Low == kCIDLib::c4MaxCard)
        {
            if (TKrnlThread::errcGetLast())
                TKrnlError::ThrowKrnlError();
        }

        i8Tmp |= c4Low;
        __fposCurPos = i8Tmp;
    }

    // Now do the out file if its been redirected
    if (__bRedirOut)
    {
        // Get the offset into a temp
        i8Tmp = fposSkipBy;

        c4Low = SetFilePointer
        (
            __hflOut
            , *pc4Low
            , pi4High
            , tCIDLib::ESeekFrom_Current
        );

        // See if it there was an error or just a position over 32 bits
        if (c4Low == kCIDLib::c4MaxCard)
        {
            if (TKrnlThread::errcGetLast())
                TKrnlError::ThrowKrnlError();
        }

        i8Tmp |= c4Low;
        __fposCurPos = i8Tmp;
    }
    return __fposCurPos;
}


tCIDLib::TFilePos TKrnlConsole::fposSeekToEnd()
{
    tCIDLib::TCard4     c4Low;
    tCIDLib::TInt8      i8Tmp;
    tCIDLib::TCard4*    pc4Low  = (tCIDLib::TCard4*)&i8Tmp;
    tCIDLib::TInt4*     pi4High = (tCIDLib::TInt4*)&i8Tmp;
    pi4High++;

    // See the output handle if its been redirected
    if (__bRedirOut)
    {
        i8Tmp = 0;
        c4Low = SetFilePointer
        (
            __hflOut
            , *pc4Low
            , pi4High
            , tCIDLib::ESeekFrom_End
        );

        // See if it there was an error or just a position over 32 bits
        if (c4Low == kCIDLib::c4MaxCard)
        {
            if (TKrnlThread::errcGetLast())
                TKrnlError::ThrowKrnlError();
        }
        i8Tmp |= c4Low;
        __fposCurPos = i8Tmp;
    }

    //
    //  See the input handle if its been redirected. This one's position
    //  is left as the value we return.
    if (__bRedirIn)
    {
        i8Tmp = 0;
        c4Low = SetFilePointer
        (
            __hflIn
            , *pc4Low
            , pi4High
            , tCIDLib::ESeekFrom_End
        );

        // See if it there was an error or just a position over 32 bits
        if (c4Low == kCIDLib::c4MaxCard)
        {
            if (TKrnlThread::errcGetLast())
                TKrnlError::ThrowKrnlError();
        }

        i8Tmp |= c4Low;
        __fposCurPos = i8Tmp;
    }
    return __fposCurPos;
}


tCIDLib::TVoid TKrnlConsole::PutCh(const tCIDLib::Tch chToWrite)
{
    __PutCh(chToWrite);
    __fposCurPos++;
}

tCIDLib::TVoid TKrnlConsole::PutCh(const tCIDLib::Tsch schToWrite)
{
    __PutCh(schToWrite);
    __fposCurPos++;
}


tCIDLib::TVoid TKrnlConsole::PutLine(const tCIDLib::Tch* const pszToWrite)
{
    __PutStr(pszToWrite);
    __fposCurPos += TRawStr::c4StrLen(pszToWrite);
}

tCIDLib::TVoid TKrnlConsole::PutLine(const tCIDLib::Tsch* const pszToWrite)
{
    __PutStr(pszToWrite);
    __fposCurPos += TRawStr::c4StrLen(pszToWrite);
}


tCIDLib::TVoid TKrnlConsole::Reset()
{
    tCIDLib::TCard4     c4Low;
    tCIDLib::TInt8      i8Tmp;
    tCIDLib::TCard4*    pc4Low  = (tCIDLib::TCard4*)&i8Tmp;
    tCIDLib::TInt4*     pi4High = (tCIDLib::TInt4*)&i8Tmp;
    pi4High++;

    if (__bRedirIn)
    {
        i8Tmp = 0;
        c4Low = SetFilePointer
        (
            __hflIn
            , *pc4Low
            , pi4High
            , tCIDLib::ESeekFrom_Start
        );

        // See if it there was an error or just a position over 32 bits
        if (c4Low == kCIDLib::c4MaxCard)
        {
            if (TKrnlThread::errcGetLast())
                TKrnlError::ThrowKrnlError();
        }
    }

    if (__bRedirOut)
    {
        i8Tmp = 0;
        c4Low = SetFilePointer
        (
            __hflOut
            , *pc4Low
            , pi4High
            , tCIDLib::ESeekFrom_Start
        );

        // See if it there was an error or just a position over 32 bits
        if (c4Low == kCIDLib::c4MaxCard)
        {
            if (TKrnlThread::errcGetLast())
                TKrnlError::ThrowKrnlError();
        }
    }
    __fposCurPos = 0;
}


tCIDLib::Tsch TKrnlConsole::schRead()
{
    tCIDLib::Tsch schRet = __schGet();
    __fposCurPos++;
    return schRet;
}



// ----------------------------------------------------------------------------
//  TConsole: Protected, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TKrnlConsole::Open( const   tCIDLib::TBoolean   bInsertMode
                    , const tCIDLib::TCard4     c4MaxRecall
                    , const   tCIDLib::ERedir   eRedirect)
{
    if ((__hflIn != kCIDLib::hflInvalid)
    ||  (__hflOut != kCIDLib::hflInvalid))
    {
        TKrnlError::ThrowKrnlError(kKrnlErrors::errcAlreadyOpen);
    }

    __bInsertMode = bInsertMode;
    __c4MaxRecall = c4MaxRecall;
    __OpenFiles(eRedirect);
}


// ----------------------------------------------------------------------------
//  TConsole: Private, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::Tch TKrnlConsole::__chGet()
{
    tCIDLib::TBoolean   bOk;
    tCIDLib::Tch        chBuf;
    tCIDLib::TCard4     c4Read;
    tCIDLib::TCard4     c4Actual;

    if (__bRedirIn)
    {
        tCIDLib::Tsch schBuf;

        if (__eTextFormat == tCIDLib::ETextFmt_ASCII)
        {
            c4Actual = 1;
            bOk = (ReadFile(__hflIn, &schBuf, c4Actual, &c4Read, 0) != 0);

            if (bOk)
                chBuf = TRawStr::chConvert(schBuf);
        }
         else
        {
            c4Actual = 2;
            bOk = (ReadFile(__hflIn, &chBuf, c4Actual, &c4Read, 0) != 0);
        }
    }
     else
    {
        c4Actual = 1;
        bOk = (ReadConsoleW(__hflIn, &chBuf, c4Actual, &c4Read, 0) != 0);
    }

    if (!bOk)
        TKrnlError::ThrowKrnlError();

    if (c4Read != c4Actual)
    {
        if (!c4Read)
            TKrnlError::ThrowKrnlError(kKrnlErrors::errcEndOfData);
        else
            TKrnlError::ThrowKrnlError(kKrnlErrors::errcNotAllData);
    }
    return chBuf;
}


tCIDLib::TVoid
TKrnlConsole::__ClearInput( const   tCIDLib::TCard4 c4CursorPos
                            , const tCIDLib::TCard4 c4CharCount)
{
    //
    //  Output enough backspaces and overwrites to get us back to the
    //  start. We have to run up to the end first if not there.
    //
    tCIDLib::TCard4 ulTmpInd;
    for (ulTmpInd = c4CursorPos; ulTmpInd < c4CharCount; ulTmpInd++)
        PutCh(Key_Space);

    for (ulTmpInd = 0; ulTmpInd < c4CharCount; ulTmpInd++)
    {
        PutCh(Key_BS);
        PutCh(Key_Space);
        PutCh(Key_BS);
    }
}


tCIDLib::TVoid
TKrnlConsole::__DelInpChar(         tCIDLib::Tch* const pszBuffer
                            , const tCIDLib::TCard4     c4CharCount
                            , const tCIDLib::TCard4     c4CursorPos
                            , const tCIDLib::TBoolean   bUnder)
{
    //
    //  We need to copy all of the chars down. We can assume that we are not
    //  at the last char or we would not have gotten called. We also have a
    //  cursor position that is > 0.
    //
    tCIDLib::TCard4 c4Ind = c4CursorPos;
    if (!bUnder)
    {
        if (!c4CursorPos)
            TKrnlError::ThrowKrnlError(kKrnlErrors::errcNegativeSeek);

        c4Ind--;
        PutCh(Key_BS);
    }

    for (; c4Ind < c4CharCount-1; c4Ind++)
    {
        pszBuffer[c4Ind] = pszBuffer[c4Ind+1];
        PutCh(pszBuffer[c4Ind]);
    }

    // Override the last char
    PutCh(Key_Space);

    // And backspace to the correct position
    tCIDLib::TCard4 c4StartPos = c4CharCount;
    if (!bUnder)
        c4StartPos++;
    for (c4Ind = c4StartPos; c4Ind > c4CursorPos; c4Ind--)
        PutCh(Key_BS);
}


tCIDLib::TVoid
TKrnlConsole::__NextPrevWord(   const   tCIDLib::Tch* const pszBuffer
                                ,       tCIDLib::TCard4&    c4CursorPos
                                , const tCIDLib::TCard4     c4CharCount
                                , const tCIDLib::Tch        chCur)
{
    tCIDLib::TBoolean   bContinue;
    tCIDLib::TCard4     c4Ind = c4CursorPos;

    if (chCur == Key_CtrlLeft)
    {
        if (!c4CursorPos)
            return;

        if (c4CursorPos == 1)
        {
            PutCh(Key_BS);
            c4CursorPos = 0;
        }

        //
        //  If the current char is not a space, and neither is the previous
        //  one, then look backward for a space. When we hit it, go back up
        //  one. If the current char is not a space, but the previous one
        //  is, then back up 1 and fall through to next block
        //
        bContinue = kCIDLib::False;
        if (pszBuffer[c4CursorPos] != kCIDLib::chSpace)
        {
            if (pszBuffer[c4CursorPos-1] == kCIDLib::chSpace)
            {
                c4Ind--;
                bContinue = kCIDLib::True;
            }
             else
            {
                for (; c4Ind > 0; c4Ind--)
                {
                    if (pszBuffer[c4Ind] == kCIDLib::chSpace)
                    {
                        c4Ind++;
                        break;
                    }
                }
            }
        }
         else
        {
            bContinue = kCIDLib::True;
        }

        if (bContinue)
        {
            // Look backward for a non-space
            for (; c4Ind > 0; c4Ind--)
            {
                if (pszBuffer[c4Ind] != kCIDLib::chSpace)
                    break;
            }

            //
            //  Look backwards for a space, then move up one.
            //
            for (; c4Ind > 0; c4Ind--)
            {
                if (pszBuffer[c4Ind] == kCIDLib::chSpace)
                {
                    c4Ind++;
                    break;
                }
            }
        }

        // Output enough backspaces to get the cursor there
        for (; c4CursorPos > c4Ind; c4CursorPos--)
            PutCh(Key_BS);
    }
     else if (chCur == Key_CtrlRight)
    {
        if (c4CursorPos == c4CharCount)
            return;

        //
        //  If the current char is a space, then we just need to look for
        //  the next non-space. If its not a space, then we need to find
        //  the next space, then the next non space.
        //
        if (pszBuffer[c4CursorPos] != kCIDLib::chSpace)
        {
            // Look forward for a space
            for (; c4Ind < c4CharCount; c4Ind++)
            {
                if (pszBuffer[c4Ind] == kCIDLib::chSpace)
                    break;
            }
        }

        // And now look forward for a non-space
        for (; c4Ind < c4CharCount; c4Ind++)
        {
            if (pszBuffer[c4Ind] != kCIDLib::chSpace)
                break;
        }

        // Output enough chars to get the cursor there
        for (; c4CursorPos < c4Ind; c4CursorPos++)
            PutCh(pszBuffer[c4CursorPos]);
    }
}


tCIDLib::TVoid TKrnlConsole::__NewEntry(const tCIDLib::Tch* const pszBuffer)
{
    // If it was empty, then nothing to do
    if (!pszBuffer[0])
        return;

    //
    //  See if it matches an existing entry. If so, then move that
    //  entry to the top.
    //
    for (tCIDLib::TCard4 c4TmpInd = 0; c4TmpInd < __c4MaxRecall-1; c4TmpInd++)
    {
        if (!__apszRecallBuf[c4TmpInd])
            break;

        if (!TRawStr::eCompareStr(__apszRecallBuf[c4TmpInd], pszBuffer))
        {
            //
            //  Save the current one, then move all of the ones before
            //  this index upwards, then put this one in 0.
            //
            tCIDLib::Tch* const pszTmp = __apszRecallBuf[c4TmpInd];

            for (; c4TmpInd > 0; c4TmpInd--)
                __apszRecallBuf[c4TmpInd] = __apszRecallBuf[c4TmpInd-1];
            __apszRecallBuf[0] = pszTmp;
            return;
        }
    }

    // If its full, then delete the last one
    if (__apszRecallBuf[__c4MaxRecall-1])
    {
        delete __apszRecallBuf[__c4MaxRecall-1];
        __apszRecallBuf[__c4MaxRecall-1] = 0;
    }

    //
    //  Rotate the elements all upwards and copy the new
    //  string into the 0th element.
    //
    for (c4TmpInd = __c4MaxRecall-1; c4TmpInd > 0; c4TmpInd--)
        __apszRecallBuf[c4TmpInd] = __apszRecallBuf[c4TmpInd-1];

    __apszRecallBuf[0] = TRawStr::pszReplicate(pszBuffer);
}


tCIDLib::TVoid TKrnlConsole::__OpenFiles(const tCIDLib::ERedir eRedirect)
{
    //
    //  According to the redirectability flag the caller provided, we get
    //  the handles differently.
    //
    tCIDLib::TFileHandle hflTmp;
    if (eRedirect == tCIDLib::ERedir_Disallow)
    {
        // Open the standard console input
        hflTmp = CreateFile
        (
            L"CONIN$"
            , GENERIC_READ | GENERIC_WRITE
            , FILE_SHARE_READ | FILE_SHARE_WRITE
            , 0
            , OPEN_EXISTING
            , FILE_ATTRIBUTE_NORMAL
            , 0
        );
        if (hflTmp == INVALID_HANDLE_VALUE)
            TKrnlError::ThrowKrnlError();

        __hflIn = hflTmp;

        // Open the standard console output
        hflTmp = CreateFile
        (
            L"CONOUT$"
            , GENERIC_READ | GENERIC_WRITE
            , FILE_SHARE_READ | FILE_SHARE_WRITE
            , 0
            , OPEN_EXISTING
            , FILE_ATTRIBUTE_NORMAL
            , 0
        );
        if (hflTmp == INVALID_HANDLE_VALUE)
            TKrnlError::ThrowKrnlError();

        __hflOut = hflTmp;
    }
     else
    {
        // Use GetStdHandle to get the redirectable handles
        hflTmp = GetStdHandle(STD_INPUT_HANDLE);
        if (hflTmp == INVALID_HANDLE_VALUE)
            TKrnlError::ThrowKrnlError();
        __hflIn = hflTmp;

        hflTmp = GetStdHandle(STD_OUTPUT_HANDLE);
        if (hflTmp == INVALID_HANDLE_VALUE)
            TKrnlError::ThrowKrnlError();
        __hflOut = hflTmp;

        //
        //  Just because redirection is allowed, doesn't mean it happened,
        //  so we check each handle and see if its a console 
        //
        tCIDLib::TCard4 c4Dummy;
        if (!GetConsoleMode(__hflIn, &c4Dummy))
            __bRedirIn = kCIDLib::True;

        if (!GetConsoleMode(__hflOut, &c4Dummy))
            __bRedirOut = kCIDLib::True;
   }

    //
    //  Put them both into the modes we want, if they are real console
    //  handles.
    //
    if (!__bRedirIn)
    {
        if (!SetConsoleMode(__hflIn, 0))
            TKrnlError::ThrowKrnlError();
    }

    if (!__bRedirOut)
    {
        if (!SetConsoleMode
        (
            __hflOut
            , ENABLE_PROCESSED_OUTPUT
              | ENABLE_WRAP_AT_EOL_OUTPUT))
        {
            TKrnlError::ThrowKrnlError();
        }
    }
}

tCIDLib::TVoid TKrnlConsole::__PutCh(const tCIDLib::Tch chToPut)
{
    tCIDLib::TBoolean   bOk;
    tCIDLib::TCard4     c4Actual;
    tCIDLib::TCard4     c4Written;

    if (__bRedirOut)
    {
        if (__eTextFormat == tCIDLib::ETextFmt_ASCII)
        {
            tCIDLib::Tsch schTmp;
            schTmp = TRawStr::schConvert(chToPut);
            c4Actual = 1;
            bOk = (WriteFile(__hflOut, &schTmp, c4Actual, &c4Written, 0) != 0);
        }
         else
        {
            c4Actual = 2;
            bOk = (WriteFile(__hflOut, &chToPut, c4Actual, &c4Written, 0) != 0);
        }
    }
     else
    {
        c4Actual = 1;
        bOk = (WriteConsoleW(__hflOut, &chToPut, c4Actual, &c4Written, 0) != 0);
    }

    if (!bOk)
        TKrnlError::ThrowKrnlError();

    if (c4Written != c4Actual)
    {
        if (!c4Written)
            TKrnlError::ThrowKrnlError(kKrnlErrors::errcEndOfData);
        else
            TKrnlError::ThrowKrnlError(kKrnlErrors::errcNotAllData);
    }
}

tCIDLib::TVoid TKrnlConsole::__PutStr(const tCIDLib::Tch* pszToPut)
{
    tCIDLib::TCard4 c4Len = TRawStr::c4StrLen(pszToPut);
    if (!c4Len)
        return;

    tCIDLib::TBoolean   bOk;
    tCIDLib::TCard4     c4Actual;
    tCIDLib::TCard4     c4Written;

    if (__bRedirOut)
    {
        if (__eTextFormat == tCIDLib::ETextFmt_ASCII)
        {
            c4Actual = c4Len;
            tCIDLib::Tsch* pszTmp = TRawStr::pszConvert(pszToPut);
            bOk = (WriteFile
            (
                __hflOut
                , pszTmp
                , c4Actual
                , &c4Written
                , 0
            ) != 0);

            delete [] pszTmp;
        }
         else
        {
            c4Actual = c4Len * kCIDLib::c4CharBytes;
            bOk = (WriteFile
            (
                __hflOut
                , pszToPut
                , c4Actual
                , &c4Written
                , 0
            ) != 0);
        }
    }
     else
    {
        c4Actual = c4Len;
        bOk = (WriteConsoleW(__hflOut, pszToPut, c4Actual, &c4Written, 0) != 0);
    }

    if (!bOk)
        TKrnlError::ThrowKrnlError();

    if (c4Written != c4Actual)
    {
        if (!c4Written)
            TKrnlError::ThrowKrnlError(kKrnlErrors::errcEndOfData);
        else
            TKrnlError::ThrowKrnlError(kKrnlErrors::errcNotAllData);
    }
}

tCIDLib::TVoid TKrnlConsole::__PutStr(const tCIDLib::Tsch* pszToPut)
{
    tCIDLib::TCard4 c4Len = TRawStr::c4StrLen(pszToPut);
    if (!c4Len)
        return;

    tCIDLib::TBoolean   bOk;
    tCIDLib::TCard4     c4Actual;
    tCIDLib::TCard4     c4Written;

    if (__bRedirOut)
    {
        if (__eTextFormat == tCIDLib::ETextFmt_ASCII)
        {
            c4Actual = c4Len;
            bOk = (WriteFile(__hflOut, pszToPut, c4Actual, &c4Written, 0) != 0);
        }
         else
        {
            c4Actual = c4Len * kCIDLib::c4CharBytes;
            tCIDLib::Tch* pszTmp = TRawStr::pszConvert(pszToPut);
            bOk = (WriteFile(__hflOut, pszToPut, c4Len, &c4Written, 0) != 0);
            delete [] pszTmp;
        }
    }
     else
    {
        c4Actual = c4Len;
        bOk = (WriteConsoleA(__hflOut, pszToPut, c4Actual, &c4Written, 0) != 0);
    }

    if (!bOk)
        TKrnlError::ThrowKrnlError();

    if (c4Written != c4Actual)
    {
        if (!c4Written)
            TKrnlError::ThrowKrnlError(kKrnlErrors::errcEndOfData);
        else
            TKrnlError::ThrowKrnlError(kKrnlErrors::errcNotAllData);
    }
}

tCIDLib::Tsch TKrnlConsole::__schGet()
{
    tCIDLib::TBoolean   bOk;
    tCIDLib::Tsch       schBuf;
    tCIDLib::TCard4     c4Actual;
    tCIDLib::TCard4     c4Read;

    if (__bRedirIn)
    {
        if (__eTextFormat == tCIDLib::ETextFmt_ASCII)
        {
            c4Actual = 1;
            bOk = (ReadFile(__hflIn, &schBuf, c4Actual, &c4Read, 0) != 0);
        }
         else
        {
            tCIDLib::Tch chTmp;
            c4Actual = 2;
            bOk = (ReadFile(__hflIn, &chTmp, c4Actual, &c4Read, 0) != 0);

            if (bOk)
                schBuf = TRawStr::schConvert(chTmp);
        }
    }
     else
    {
        c4Actual = 1;
        bOk = (ReadConsoleA(__hflIn, &schBuf, c4Actual, &c4Read, 0) != 0);
    }

    if (!bOk)
        TKrnlError::ThrowKrnlError();

    if (c4Read != c4Actual)
    {
        if (!c4Read)
            TKrnlError::ThrowKrnlError(kKrnlErrors::errcEndOfData);
        else
            TKrnlError::ThrowKrnlError(kKrnlErrors::errcNotAllData);
    }
    return schBuf;
}
