//
//  FILE NAME: CIDKernel.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/06/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module of the facility. This guy provides the DLL
//  init function in order to handle initialization of any modules that
//  require it.
//
//  CAVEATS/GOTCHAS:
//
//  1)  Note that the init done here is platform specific. Its not abstracted
//      like it is for all the higher level facilities.
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDKernel_.Hpp"



// ----------------------------------------------------------------------------
//  Intrafacility data
//
//  _hmodKernel
//      This is the raw module handler for internal use. For public access,
//      we have the kmodCIDKernel object.
// ----------------------------------------------------------------------------
tCIDLib::TModHandle _hmodKernel;


// ----------------------------------------------------------------------------
//  Public data
//
//  kmodCIDKernel
//      This is the kernel's version of a facility object. It allows access
//      to things like the kernel's strings and such, to CIDLib. It uses the
//      _hmodKernel which was set during init before global constructors.
// ----------------------------------------------------------------------------
TKrnlModule         kmodCIDKernel(_hmodKernel);


// ----------------------------------------------------------------------------
//  Local data
//
//  __afnInitTerm
//      This is the list of init term functions for this facility.
//
//  __c4InitCount
//      The count of elements in the init/term list.
// ----------------------------------------------------------------------------
#if 0
static tCIDKernel_::TKModuleInitTermRec __afnInitTerm[] =
{
      { _InitTermStackDump          , kMessages::pszMod_StackDump }
    , { _InitTermMemCheck           , kMessages::pszMod_MemCheck }
    , { _InitTermSysInfo            , kMessages::pszMod_SysInfo }
    , { _InitTermEnvironment        , kMessages::pszMod_Environment }
    , { _InitTermLocale             , kMessages::pszMod_Locale }
    , { _InitTermMetrics            , kMessages::pszMod_Metrics }
    , { _InitTermProcessRegistry    , kMessages::pszMod_ProcReg }
};
#endif

static tCIDKernel_::TKModuleInitTermRec __afnInitTerm[] =
{
      { _InitTermStackDump          , L"Stack Dump" }
    , { _InitTermMemCheck           , L"Heap Check" }
    , { _InitTermSysInfo            , L"Sys Info" }
    , { _InitTermEnvironment        , L"Environment" }
    , { _InitTermLocale             , L"Locale" }
    , { _InitTermMetrics            , L"Metrics" }
    , { _InitTermProcessRegistry    , L"Process Registry" }
};
static tCIDLib::TCard4      __c4InitCount = c4ArrayElems(__afnInitTerm);


// ----------------------------------------------------------------------------
//  External function prototypes
// ----------------------------------------------------------------------------
extern "C" tCIDLib::TCard4 DLLINITAPI _DllMainCRTStartup
(
    tCIDLib::TVoid*
    , tCIDLib::TUInt
    , tCIDLib::TVoid*
);


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __CoreSetup
//
// DESCRIPTION:
//
//  This method handles some very core setup that is done before any
//  of the module init calls are made.
// -------------------------------------
//   INPUT: hmodKernel is the module handle of this module.
//
//  OUTPUT: None
//
//  RETURN: True if successful, else False.
//
static tCIDLib::TBoolean __CoreSetup(const tCIDLib::TModHandle hmodKernel)
{
    // Store away the module handle
    _hmodKernel = hmodKernel;

    // Disable system popups
    SetErrorMode
    (
        SEM_FAILCRITICALERRORS
        | SEM_NOGPFAULTERRORBOX
        | SEM_NOOPENFILEERRORBOX
    );

    return kCIDLib::True;
}


//
// FUNCTION/METHOD NAME: __bDoModuleInitTerm
//
// DESCRIPTION:
//
//  This is called from the DLL entry point below. Its called once before
//  global constructors and once after. It passes on the flag to each of
//  the module inits so that they can do before/after init.
// ---------------------------------------
//   INPUT: eGlobals indicates whether this is for before or after globals
//              are constructed or destructed.
//          eInitTerm is the reason we are getting called, which will be
//              init or term.
//          hmodThis is the module handle, which is passed to any init
//              functions that need it.
//
//  OUTPUT: None
//
//  RETURN: True if successful, else False.
//
static tCIDLib::TBoolean
__bDoModuleInitTerm(    const   tCIDLib::EGlobalStates  eGlobals
                        , const tCIDLib::EInitTerm      eInitTerm
                        , const tCIDLib::TModHandle     hmodThis)
{
    tCIDLib::TBoolean       bSuccess = kCIDLib::False;
    const tCIDLib::TCard4   c4FailBufMax = 1024;
    const tCIDLib::TCard4   c4MsgBufMax = 1024;
    tCIDLib::Tch            szFailBuffer[c4FailBufMax+1];
    tCIDLib::Tch            szMsgBuf[c4MsgBufMax+1];

    TRawStr::CopyStr(szFailBuffer, kMessages::pszGen_Unknown, c4FailBufMax);
    const tCIDLib::Tch* pszPhase = kMessages::pszGen_Unknown;
    try
    {
        //
        //  Do any of the module initialization class that need to
        //  be called. We do them in the correct bootstrapping order
        //  from most primitive to least.
        //
        if (eInitTerm == tCIDLib::EInitTerm_Initialize)
        {
            // Do the core system setup stuff
            __CoreSetup(hmodThis);

            for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4InitCount; c4Index++)
            {
                pszPhase = __afnInitTerm[c4Index].pszText;
                __afnInitTerm[c4Index].pfnInitTermFunc
                (
                    eInitTerm
                    , eGlobals
                    , hmodThis
                    , c4FailBufMax
                    , szFailBuffer
                );
            }
        }
         else
        {
            tCIDLib::TInt4 i4Index = __c4InitCount-1;

            while (i4Index > 0)
            {
                pszPhase = __afnInitTerm[i4Index].pszText;
                __afnInitTerm[i4Index].pfnInitTermFunc
                (
                    eInitTerm
                    , eGlobals
                    , hmodThis
                    , c4FailBufMax
                    , szFailBuffer
                );
                i4Index--;
            }
        }

        //
        //  If we are in initialize phase and before globals, then
        //  do the constructors. If we are in term phase and before
        //  globals, then do the destructors.
        //
        if (((eGlobals == tCIDLib::EGlobalState_Before)
        &&   (eInitTerm == tCIDLib::EInitTerm_Initialize))
        ||  ((eGlobals == tCIDLib::EGlobalState_After)
        &&   (eInitTerm == tCIDLib::EInitTerm_Terminate)))
        {
            if (eInitTerm == tCIDLib::EInitTerm_Initialize)
                pszPhase = kMessages::pszInit_GlobalCtor;
            else
                pszPhase = kMessages::pszInit_GlobalDtor;

            _DllMainCRTStartup(hmodThis, eInitTerm, 0);
        }
        bSuccess = kCIDLib::True;
    }

    catch(const TKrnlError& kerrToCatch)
    {
        tCIDLib::TZStr64    szKrnlErr;
        tCIDLib::TZStr64    szOSErr;

        TRawStr::FormatVal(kerrToCatch.errcId(), szKrnlErr, 64, tCIDLib::ERadix_Dec);
        TRawStr::FormatVal(kerrToCatch.errcHostId(), szOSErr, 64, tCIDLib::ERadix_Dec);

        TRawStr::CopyCatStr
        (
            szMsgBuf
            , c4MsgBufMax
            , kMessages::pszInit_KrnlError
            , pszPhase
            , L"/"
            , szFailBuffer
        );

        TRawStr::CatStr(szMsgBuf, L"/", c4MsgBufMax);
        TRawStr::CatStr(szMsgBuf, szKrnlErr, c4MsgBufMax);
        TRawStr::CatStr(szMsgBuf, L"/", c4MsgBufMax);
        TRawStr::CatStr(szMsgBuf, szOSErr, c4MsgBufMax);
        _KRNLERRPOPUP_(szMsgBuf, kerrToCatch)
    }

    catch(...)
    {
        TRawStr::CopyCatStr
        (
            szMsgBuf
            , c4MsgBufMax
            , kMessages::pszInit_SysError
            , pszPhase
            , L"/"
            , szFailBuffer
        );
        _MSGPOPUP_(szMsgBuf)
    }
    return bSuccess;
}


// ----------------------------------------------------------------------------
//  Program entry point
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: DllInit
//
// DESCRIPTION:
//
//  This method will call all of the individual module init functions. It is
//  called from the system when the .Dll is loaded. We have to kick off the
//  creation of global objects for each process attach. This way we get to
//  do our low level init before globals are created.
// -------------------------------------
//   INPUT: hmodThis is the module handle of this DLL.
//          eInitTerm is the reason we are getting called.
//          pReserved is not used of course.
//
//  OUTPUT: None
//
//  RETURN: 1 if successful, else 0. If a 0 is returned, then the load of
//              the client app will fail.
//
tCIDLib::TCard4 DLLINITAPI
CIDKernelInit(  tCIDLib::TModHandle     hmodThis
                , tCIDLib::EInitTerm    eInitTerm
                , tCIDLib::TVoid*       pDummy)
{
    if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
    ||  (eInitTerm == tCIDLib::EInitTerm_Terminate))
    {
        //
        //  Do the pre-global constructor init. This will kick off the
        //  global constructors after all the inits are done.
        //
        if (!__bDoModuleInitTerm(tCIDLib::EGlobalState_Before, eInitTerm, hmodThis))
            return 0;

        // Now do the post globals init
        if (!__bDoModuleInitTerm(tCIDLib::EGlobalState_After, eInitTerm, hmodThis))
            return 0;
    }
     else
    {
        return _DllMainCRTStartup(hmodThis, eInitTerm, pDummy);
    }
    return 1;
}
