//
// NAME: CIDFrac_MultiCalcEngine.Cpp
//
// DESCRIPTION: 
//
//  This module implements the TFracMultiEngine class, which provides a
//  multithreaded derivative of the base calculation engine class.
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 06/15/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  Facility includes
// -----------------------------------------------------------------------------
#include    "CIDFractal_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macro
// -----------------------------------------------------------------------------
RTTIData2(TFracMTEngine,TFracCalcEngine)



// -----------------------------------------------------------------------------
//   CLASS: TFracMTEngine
//  PREFIX: fceng
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TFracMTEngine: Constructors and Destructors
// -----------------------------------------------------------------------------

TFracMTEngine::TFracMTEngine() :

    __c4ThreadCount(0)
    , __pthrWorkers(0)
{
    // This requires a call to Setup() before use
}

TFracMTEngine::TFracMTEngine(   const   TSize&              szPelImage
                                , const TFArea&             fareaSpace
                                ,       TBaseFractal* const pfracToAdopt
                                , const tCIDLib::TCard4     c4ThreadCount
                                , TStatusController* const  pstatcToAdopt) :
    TFracCalcEngine
    (
        szPelImage
        , fareaSpace
        , pfracToAdopt
        , pstatcToAdopt
    )
    , __c4ThreadCount(0)
    , __pthrWorkers(0)
{
    Setup(szPelImage, fareaSpace, pfracToAdopt, c4ThreadCount, pstatcToAdopt);
}

TFracMTEngine::TFracMTEngine(const TFracMTEngine& fcengToCopy) :

    TFracCalcEngine(fcengToCopy)
    , __c4ThreadCount(fcengToCopy.__c4ThreadCount)
    , __pthrWorkers(0)
{
    //
    //  Create our bag of worker threads. Then fill it with standard threads,
    //  the number that we were told to create.
    //
    __pthrWorkers = new TStdFracThread[__c4ThreadCount];

    for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4ThreadCount; c4Index++)
        __pthrWorkers[c4Index].SetCalcEngine(this);
}

TFracMTEngine::~TFracMTEngine()
{
    // Delete the array of threads
    delete [] __pthrWorkers;
}


// -----------------------------------------------------------------------------
//  TFracMTEngine: Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TFracMTEngine::StartProcessing()
{
    if (!__c4ThreadCount)
    {
        facCIDFractal.LogErr
        (
            __FILE__
            , __LINE__
            , kFractErrs::errcCEng_NotSetUp
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
        );
    }

    for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4ThreadCount; c4Index++)
        __pthrWorkers[c4Index].Start();
}

tCIDLib::TVoid TFracMTEngine::StopProcessing()
{
    if (!__c4ThreadCount)
    {
        facCIDFractal.LogErr
        (
            __FILE__
            , __LINE__
            , kFractErrs::errcCEng_NotSetUp
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
        );
    }

    for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4ThreadCount; c4Index++)
        __pthrWorkers[c4Index].RequestShutdown(30 Seconds);
}


// -----------------------------------------------------------------------------
//  TFracMTEngine: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TFracMTEngine::Setup(   const   TSize&                  szPelImage
                        , const TFArea&                 fareaImage
                        ,       TBaseFractal* const     pfracToAdopt
                        , const tCIDLib::TCard4         c4ThreadCount
                        ,       TStatusController* const pstatcToAdopt)
{
    // Pass on the basic information to our parent
    _SetImageInfo(szPelImage, fareaImage, pfracToAdopt, pstatcToAdopt);

    // The thread count cannot be 0 or > 64
    if (!c4ThreadCount || (c4ThreadCount > 64))
    {
        facCIDFractal.LogErr
        (
            __FILE__
            , __LINE__
            , kFractErrs::errcMTEng_BadThreadCount
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
            , TCardinal(c4ThreadCount)
        );
    }

    //
    //  Create our bag of worker threads. Then fill it with standard threads,
    //  the number that we were told to create.
    //
    __pthrWorkers = new TStdFracThread[c4ThreadCount];

    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4ThreadCount; c4Index++)
        __pthrWorkers[c4Index].SetCalcEngine(this);
}



// -----------------------------------------------------------------------------
//  TFracMTEngine: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TFracMTEngine::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    TParent::_StreamFrom(strmToReadFrom);
    strmToReadFrom >> __c4ThreadCount;
}

tCIDLib::TVoid
TFracMTEngine::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    TParent::_StreamTo(strmToWriteTo);
    strmToWriteTo << __c4ThreadCount;
}
