//
// NAME: CIDFractal_Mandelbrot.Cpp
//
// DESCRIPTION: 
//
//  This module implements the TMandelbrot class and a few smaller,
//  more specialized derivatives.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 08/22/94
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  Facility includes
// -----------------------------------------------------------------------------
#include    "CIDFractal_.Hpp"


// -----------------------------------------------------------------------------
//  Local constant data
// -----------------------------------------------------------------------------
const tCIDLib::TFloat8  f8SmallVal   = 0.000000001;


// -----------------------------------------------------------------------------
//  Do our standard members macro
// -----------------------------------------------------------------------------
RTTIData2(TMandelbrot,TEscapeFractal)



// -----------------------------------------------------------------------------
//   CLASS: TMandelbrot
//  PREFIX: frac
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TMandelbrot: Constructors and Destructors
// -----------------------------------------------------------------------------

TMandelbrot::TMandelbrot() :

    TEscapeFractal(tCIDLib::ESymmetry_VerticalFlip, 16)
{
}

TMandelbrot::TMandelbrot(const TMandelbrot& fracToCopy) :

    TEscapeFractal(fracToCopy)
{
}

TMandelbrot::TMandelbrot(const tCIDLib::TCard4 c4MaxIterations) :

    TEscapeFractal
    (
        tCIDLib::ESymmetry_VerticalFlip
        , c4MaxIterations
    )
{
}


// -----------------------------------------------------------------------------
//  TMandelbrot: Public operators
// -----------------------------------------------------------------------------

TMandelbrot& TMandelbrot::operator=(const TMandelbrot& fracToAssign)
{
    if (this == &fracToAssign)
        return *this;

    TParent::operator=(fracToAssign);
    return *this;
}


// -----------------------------------------------------------------------------
//  TMandelbrot: Public, static methods
// -----------------------------------------------------------------------------

tCIDLib::TCard4 ASMCALL
TMandelbrot::c4MandelLoop(  const   tCIDLib::TFloat8    f8ZReal
                            , const tCIDLib::TFloat8    f8ZImag
                            , const tCIDLib::TFloat8    f8CReal
                            , const tCIDLib::TFloat8    f8CImag
                            , const tCIDLib::TCard4     c4MaxIter)
{
    tCIDLib::TCard4     c4Ret;

    #if defined(CIDLIB_CPU_X86)
    tCIDLib::TCard4     c4Tmp;
    tCIDLib::TFloat8    f8LastReal, f8LastImag;
    _asm
    {
        xor     eax, eax            ; Set up 0 return if 0 count
        mov     ecx, [c4MaxIter]    ; Set loop counter to maximum
        jecxz   NoOpExit            ; If c4MaxIter is 0, then exit
        jmp     SHORT OpCont
    NoOpExit:
        jmp     NoOp
    OpCont:
        FLDZ                        ; Load a 0
        FST     [f8LastReal]        ; Init the two periodicity checking save values
        FSTP    [f8LastImag]        ; Stack is empty again

        FLD     [f8CReal]           ; Load up the real/imag of c component
        FLD     [f8CImag]           ; which will remain the same
        FLD     [f8ZReal]           ; Load up the real/imag of z component
        FLD     [f8ZImag]           ; which will change
        FLD1
        FLD1
        FADD
        ; -------------
        ; ST0=2.0
        ; ST1=f8ZImag
        ; ST2=f8ZReal
        ; ST3=fCImag
        ; ST4=fCReal
        ; -------------

        ; -------------
        ;  Init the periodicity checking registers
        ;
        ;  ebx=c4CheckEvery
        ;  edi=c4LastPeriod
        ;  esi=c4Periodicity
        ;  edx=PeriodicityThreshold
        ;--------------
        mov     ebx, 3
        mov     edi, 1
        mov     esi, 1
        xor     edx, edx

    IterLoop:                       ; Enter the iteration loop
        dec     ecx                 ; Bump down the counter
        jecxz   ExitLoop            ; Fall out if we hit 0
        jmp     SHORT ContLoop      ; Handle the non-short jump
    ExitLoop:
        jmp     Exit

    ContLoop:
        FLD     ST(1)               ; Load another copy of f8ZImag
        ; -------------
        ; ST0=f8ZImag
        ; ST1=2.0
        ; ST2=f8ZImag
        ; ST3=f8ZReal
        ; ST4=fCImag
        ; ST5=fCReal
        ; -------------


        ; --------------------------------------------------------------------------
        ; Do 2*f8ZReal*f8ZImag+fCImag
        ; --------------------------------------------------------------------------
        FMUL    ST, ST(3)
        FMUL    ST, ST(1)
        ; -------------
        ; ST0=2*f8ZImag*f8ZReal
        ; ST1=2.0
        ; ST2=f8ZImag
        ; ST3=f8ZReal
        ; ST4=fCImag
        ; ST5=fCReal
        ; -------------


        FADD    ST, ST(4)
        ; -------------
        ; ST0=2*f8ZReal*f8ZImag+fCImag
        ; ST1=2.0
        ; ST2=f8ZImag
        ; ST3=f8ZReal
        ; ST4=fCImag
        ; ST5=fCReal
        ; -------------


        FXCH    ST(2)               ; Stow away that value in ST(2) and bring in f8ZImag
        ; -------------
        ; ST0=f8ZImag
        ; ST1=2.0
        ; ST2=2*f8ZReal*f8ZImag+fCImag
        ; ST3=f8ZReal
        ; ST4=fCImag
        ; ST5=fCReal
        ; -------------


        ; --------------------------------------------------------------------------
        ; Do (f8ZReal*f8ZReal)-(f8ZImag*f8ZImag) + cReal
        ; --------------------------------------------------------------------------
        FLD     ST(0)               ; Load another copy of f8ZImag and square it
        FMUL
        ; -------------
        ; ST0=f8ZImag^2
        ; ST1=2.0
        ; ST2=2*f8ZReal*f8ZImag+fCImag
        ; ST3=f8ZReal
        ; ST4=fCImag
        ; ST5=fCReal
        ; -------------


        FLD     ST(3)               ; Load another copy of f8ZReal
        ; -------------
        ; ST0=f8ZReal
        ; ST1=f8ZImag^2
        ; ST2=2.0
        ; ST3=2*f8ZReal*f8ZImag+fCImag
        ; ST4=f8ZReal
        ; ST5=fCImag
        ; ST6=fCReal
        ; -------------


        FMULP   ST(4), ST
        ; -------------
        ; ST0=f8ZImag^2
        ; ST1=2.0
        ; ST2=2*f8ZReal*f8ZImag+fCImag
        ; ST3=f8ZReal^2
        ; ST4=fCImag
        ; ST5=fCReal
        ; -------------


        FSUBP   ST(3), ST
        ; -------------
        ; ST0=2.0
        ; ST1=2*f8ZReal*f8ZImag+fCImag
        ; ST2=f8ZReal^2-f8ZImag^2
        ; ST3=fCImag
        ; ST4=fCReal
        ; -------------


        FLD     ST(4)               ; Load another copy of fCReal
        ; -------------
        ; ST0=fCReal
        ; ST1=2.0
        ; ST2=2*f8ZReal*f8ZImag+fCImag
        ; ST3=f8ZReal^2-f8ZImag^2
        ; ST4=fCImag
        ; ST5=fCReal
        ; -------------


        FADDP   ST(3), ST
        ; -------------
        ; ST0=2.0
        ; ST1=2*f8ZReal*f8ZImag+fCImag
        ; ST2=f8ZReal^2-f8ZImag^2+fCReal
        ; ST3=fCImag
        ; ST4=fCReal
        ; -------------


        FLD     ST(1)               ; Load 2 extra copies of new f8ZReal & square
        FLD     ST(0)
        FMUL
        ; -------------
        ; ST0=sqr(2*f8ZReal*f8ZImag+fCImag)
        ; ST1=2.0
        ; ST2=2*f8ZReal*f8ZImag+fCImag
        ; ST3=(f8ZReal^2-f8ZImag^2)+fCReal
        ; ST4=fCImag
        ; ST5=fCReal
        ; -------------


        FLD     ST(3)               ; Load 2 extra copies of new f8ZImag & square
        FLD     ST(0)
        FMUL
        ; -------------
        ; ST0=sqr((f8ZReal^2-f8ZImag^2)+fCReal)
        ; ST1=sqr(2*f8ZReal*f8ZImag+fCImag)
        ; ST2=2.0
        ; ST3=2*f8ZReal*f8ZImag+fCImag
        ; ST4=(f8ZReal^2-f8ZImag^2)+fCReal
        ; ST5=fCImag
        ; ST6=fCReal
        ; -------------


        FADD                        ; Add them and pop, leaving modulus in ST(0)
        ; -------------
        ; ST0=Modulus of z component squared
        ; ST1=2.0
        ; ST2=2*f8ZReal*f8ZImag+fCImag
        ; ST3=(f8ZReal^2-f8ZImag^2)+fCReal
        ; ST4=fCImag
        ; ST5=fCReal
        ; -------------


        mov     [c4Tmp], 4          ; Load up a constant 4.0
        FILD    [c4Tmp]
        FXCH
        ; -------------
        ; ST0=Modulus of z component squared
        ; ST1=4.0
        ; ST2=2.0
        ; ST3=2*f8ZReal*f8ZImag+fCImag
        ; ST4=(f8ZReal^2-f8ZImag^2)+fCReal
        ; ST5=fCImag
        ; ST6=fCReal
        ; -------------


        FCOMPP                      ; Compare Modulus to 4.0 and pop them both
        ; -------------
        ; ST0=2.0
        ; ST1=New f8ZImag           ; Stack is ready for next round
        ; ST2=New f8ZReal
        ; ST3=fCImag
        ; ST4=fCReal
        ; -------------
        FSTSW   ax                  ; Store the status word
        test    ax, 0400H           ; Check for unordered
        jnz     BadVal

        test    ax, 04500H          ; Test comparison flags
        jz      Exit                ; If none on, then > 4.0

        ; -------------
        ;  Do periodicity check before we go back
        ;  to the top of the loop
        ; -------------
        FLD     ST(1)               ; Load a copy of f8ZImag for compare
        FSUB    [f8LastImag]
        FABS
        FLD     [f8SmallVal]
        FCOMPP
        FSTSW   ax                  ; Store the status word
        and     ax, 04500H          ; Isolate the comparison flags
        jz      TestReal            ; It is equal, so check real
        jmp     NextPeriod          ; No match so try another period

    TestReal:
        FLD     ST(2)               ; Load a copy of f8ZReal for compare
        FSUB    [f8LastReal]
        FABS
        FLD     [f8SmallVal]
        FCOMPP
        FSTSW   ax                  ; Store the status word
        WAIT                        ; Sync with CPU
        and     ax, 04500H          ; Isolate the comparison flags
        jnz     NextPeriod          ; No match so try another period

        cmp     edi, esi            ; If (c4LastPeriod == c4Periodicity)
        jne     SavePeriod
        inc     edx                 ; Bump up the times we got the same periodicity
        cmp     edx, 4              ; Have we matched 4 times?
        jae     PreExit             ; If so, then get out

        mov     ebx, 3              ; Reset the periodicity variables
        mov     esi, 1              ; Bump not the last period
        FLDZ
        FST     [f8LastReal]
        FSTP    [f8LastImag]
        jmp     IterLoop            ; And go back to the top

    SavePeriod:
        mov     edi, esi            ; else c4LastPeriod == c4Periodicity
        mov     esi, 1              ;      c4Periodicity = 1
        jmp     IterLoop            ; and go to the top again

    NextPeriod:
        inc     esi                 ; Bump up c4Periodicity
        cmp     esi, ebx            ; If (c4Periodicity >= c4CheckEvery)
        jae     CheckEvery          ;   then check the CheckEvery value
        jmp     IterLoop            ; else go to top of loop

    CheckEvery:
        cmp     ebx, 32             ; Is (c4CheckEvery > 32)
        ja      ResetEvery          ;   then reset it
    ;    inc     ebx
        shl     ebx, 2              ; else c4CheckEvery *= 2
        jmp     SaveLast            ; and save last

    ResetEvery:
        mov     ebx, 3              ; Reset the periodicity variables
        mov     edi, 1
        mov     esi, 1
        FLDZ
        FST     [f8LastReal]         ; Store zero once
        FSTP    [f8LastImag]         ; And again and pop off

    SaveLast:
        mov     esi, 1              ; Reset c4Periodicity
        FLD     ST(1)               ; Load a copy of f8ZImag
        FSTP    [f8LastImag]        ; Save it and pop
        FLD     ST(2)               ; Load a copy of f8ZReal
        FSTP    [f8LastReal]        ; Save it and pop

        jmp     IterLoop            ; And go to the top of the loop

    BadVal:
        mov     ecx, 0              ; Make it come out like a non-escape

    PreExit:
        mov     ecx, 0

    Exit:
        FADDP   ST(1), ST(0)
        FCOMPP
        FCOMPP                      ; Clean up the NP stack using junk ops

        mov     eax, [c4MaxIter]    ; Get the passed maximum iters
        sub     eax, ecx            ; Sub ending ecx for actual iterations
        mov     [c4Ret], eax        ; And move to return value
        WAIT

    NoOp:
    }
    #else
    tCIDLib::TFloat8    f8X = f8ZReal;
    tCIDLib::TFloat8    f8Y = f8ZImag;
    tCIDLib::TFloat8    f8XSqr = f8ZReal * f8ZReal;
    tCIDLib::TFloat8    f8YSqr = f8ZImag * f8ZImag;

    // Start at 1 since we've set up stuff as though one already was done
    c4Ret = 1;
    while (c4Ret <= c4MaxIter)
    {
        const tCIDLib::TFloat8 f8NewX = (f8XSqr - f8YSqr) + f8CReal;
        const tCIDLib::TFloat8 f8NewY = (2 * f8X * f8Y) + f8CImag;

        f8XSqr = f8NewX * f8NewX;
        f8YSqr = f8NewY * f8NewY;

        if (f8XSqr + f8YSqr >= 4.0)
            break;

        f8X = f8NewX;
        f8Y = f8NewY;

        c4Ret++;
    }

    #endif

    return c4Ret;
}


// -----------------------------------------------------------------------------
//  TMandelbrot: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TMandelbrot::CalcPixel( const   TQ1Point&   pntPixel
                        , const TFPoint&    pntFractal
                        ,       TFracCache& fcachToStoreIn) const
{
    // Get stuff into locals so we can get back their eventual new values
    tCIDLib::TFloat8 f8ZReal = pntFractal.f8X();
    tCIDLib::TFloat8 f8ZImag = pntFractal.f8Y();
    tCIDLib::TFloat8 f8CReal = f8ZReal;
    tCIDLib::TFloat8 f8CImag = f8ZImag;

    //
    //  Call the assembly function. When we come back, the values of the
    //  z and c components are as they were when we escaped (or failed to.)
    //  Since they are references, we now have the results in our locals.
    //
    tCIDLib::TCard4 c4Escape = c4MandelLoop
    (
        f8ZReal
        , f8ZImag
        , f8CReal
        , f8CImag
        , c4MaxIterations()
    );

    //
    //  According to our mapping mode we save one or the other of the
    //  values. We just call our parent and pass the candidate values.
    //  He will store the appropriate one.
    //
    _StoreValue
    (
        c4Escape
        , f8ZReal
        , f8ZImag
        , f8CReal
        , f8CImag
        , pntPixel
        , fcachToStoreIn
    );
}
