//
// NAME: CIDFractal_Lyapunov.Hpp
//
// DESCRIPTION: 
//
//  This is the header for the module CIDFractal_Lyapunov.Hpp. This module
//  implements the TLyapunov class, which is a sort of escape time fractal
//  but not in the sense that a Mandelbrot is. The Lyapunov equation is
//  often used to model population fluctuations in the precense of a limited
//  food source. This fractal throws in a twist where the inputs to the
//  equation is periodically modified by a user provided pattern. It really
//  requires at least 15 bit color to be even remotely useful.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 12/06/95
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  Forward references
// -----------------------------------------------------------------------------
class   TFracCache;


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//  CLASS: TLyapunov
// PREFIX: frac
// -----------------------------------------------------------------------------
class CIDFRACTALEXP TLyapunov : public TBaseFractal
{
    public  :
        // ---------------------------------------------------------------------
        //  Constructors and Destructors.
        // ---------------------------------------------------------------------
        TLyapunov();

        TLyapunov
        (
            const   TLyapunov&              fracToCopy
        );

        TLyapunov
        (
            const   tCIDLib::TCard4         c4SeqMap
            , const tCIDLib::TCard4         c4MapBits
            , const tCIDLib::TFloat8        f8InitX
            , const tCIDLib::TCard4         c4MaxIter
        );

        ~TLyapunov() {}


        // ---------------------------------------------------------------------
        //  Public operators
        // ---------------------------------------------------------------------
        TLyapunov& operator=
        (
            const   TLyapunov&              fracToAssign
        );


        // ---------------------------------------------------------------------
        //  Public, static methods
        // ---------------------------------------------------------------------
        static tCIDLib::TFloat8 ASMCALL f8LyapLoop
        (
            const   tCIDLib::TCard4         c4SeqMap
            , const tCIDLib::TCard4         c4BitCnt
            ,       tCIDLib::TCard4&        c4CurBit
            , const tCIDLib::TCard4         c4MaxIter
            , const tCIDLib::TFloat8        f8AVal
            , const tCIDLib::TFloat8        f8BVal
            , const tCIDLib::TFloat8        f8XComponent
        );


        // ---------------------------------------------------------------------
        //  Public, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid CalcPixel
        (
            const   TQ1Point&               pntPixel
            , const TFPoint&                pntFractal
            ,       TFracCache&             fcachToStoreIn
        )   const;

        tCIDLib::TCard4 c4BytesPerPixel() const;

        TFArea fareaDefSpace() const;


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TCard4 c4MaxIterations() const;

        tCIDLib::TCard4 c4MaxIterations
        (
            const   tCIDLib::TCard4         c4NewMax
        );


    private :
        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __c4MapBits
        //      The number of relevant bits in the sequence map. The map can
        //      have up to 32 bits and this indicates how many of them are
        //      to be used.
        //
        //  __c4MaxIter
        //      The maximum iteration count. Its defaulted at construction and
        //      changed by the user if desired during the configuration of the
        //      image.
        //
        //  __c4SeqMap
        //      This is the sequence map of bits. Each bit represents whether
        //      the A or B value should be used. __c4MapBits indicates how
        //      man bits are to be used, since there does not have to be 32
        //      bits.
        //
        //  __f8InitialX
        //      This is the X component of the lyapunov formula. It is
        //      defaulted and then possibly modified by the user during the
        //      configuration of the image.
        // ---------------------------------------------------------------------
        tCIDLib::TCard4         __c4MapBits;
        tCIDLib::TCard4         __c4MaxIter;
        tCIDLib::TCard4         __c4SeqMap;
        tCIDLib::TFloat8        __f8InitialX;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TLyapunov,TBaseFractal)
        DefPolyDup(TLyapunov)
};

#pragma pack(pop)



// -----------------------------------------------------------------------------
//  Public, inherited methods
// -----------------------------------------------------------------------------
inline tCIDLib::TCard4 TLyapunov::c4MaxIterations() const
{
    return __c4MaxIter;
}

inline tCIDLib::TCard4 TLyapunov::c4MaxIterations(const tCIDLib::TCard4 c4NewMax)
{
    __c4MaxIter = c4NewMax;
    return __c4MaxIter;
}

inline tCIDLib::TCard4 TLyapunov::c4BytesPerPixel() const
{
    return 8;
}

inline TFArea TLyapunov::fareaDefSpace() const
{
    return TFArea(-1, 1, -1, 1);
}
