//
// NAME: CIDFractal_Lyapunov.Cpp
//
// DESCRIPTION: 
//
//  This module implements the TLyapunov class and a few smaller,
//  more specialized derivatives.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 08/22/94
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  Facility includes
// -----------------------------------------------------------------------------
#include    "CIDFractal_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macro
// -----------------------------------------------------------------------------
RTTIData2(TLyapunov,TBaseFractal)



// -----------------------------------------------------------------------------
//   CLASS: TLyapunov
//  PREFIX: frac
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TLyapunov: Constructors and Destructors
// -----------------------------------------------------------------------------

TLyapunov::TLyapunov(   const   tCIDLib::TCard4   c4SeqMap
                        , const tCIDLib::TCard4   c4MapBits
                        , const tCIDLib::TFloat8  f8InitX
                        , const tCIDLib::TCard4   c4MaxIter) :
    TBaseFractal
    (
        tCIDFractal::EElemType_Float8
    )
    , __c4MapBits(c4MapBits)
    , __c4MaxIter(c4MaxIter)
    , __c4SeqMap(c4SeqMap)
    , __f8InitialX(f8InitX)
{
}

TLyapunov::TLyapunov(const TLyapunov& fracToCopy) :

    TBaseFractal(fracToCopy)
    , __c4MapBits(fracToCopy.__c4MapBits)
    , __c4MaxIter(fracToCopy.__c4MaxIter)
    , __c4SeqMap(fracToCopy.__c4SeqMap)
    , __f8InitialX(fracToCopy.__f8InitialX)
{
}

TLyapunov::TLyapunov() :

    TBaseFractal
    (
        tCIDFractal::EElemType_Float8
    )
    , __c4MapBits(2)
    , __c4MaxIter(1024)
    , __c4SeqMap(2)
    , __f8InitialX(0.15)
{
}


// -----------------------------------------------------------------------------
//  Lyapunov: Public operators
// -----------------------------------------------------------------------------

TLyapunov& TLyapunov::operator=(const TLyapunov& fracToAssign)
{
    if (this == &fracToAssign)
        return *this;

    TParent::operator=(fracToAssign);

    __c4MapBits     = fracToAssign.__c4MapBits;
    __c4MaxIter     = fracToAssign.__c4MaxIter;
    __c4SeqMap      = fracToAssign.__c4SeqMap;
    __f8InitialX    = fracToAssign.__f8InitialX;

    return *this;
}


// -----------------------------------------------------------------------------
//  Lyapunov: Public, static methods
// -----------------------------------------------------------------------------

tCIDLib::TFloat8 ASMCALL
TLyapunov::f8LyapLoop(  const   tCIDLib::TCard4     c4SeqMap
                        , const tCIDLib::TCard4     c4BitCnt
                        ,       tCIDLib::TCard4&    c4CurBit
                        , const tCIDLib::TCard4     c4MaxIter
                        , const tCIDLib::TFloat8    f8AVal
                        , const tCIDLib::TFloat8    f8BVal
                        , const tCIDLib::TFloat8    f8XComponent)
{
    tCIDLib::TFloat8    f8Total, f8Ret;

    #if defined(CIDLIB_CPU_X86)
    tCIDLib::TCard4     c4Tmp, c4Period;
    _asm
    {
        xor     eax, eax            ; Set up 0 return if 0 count
        mov     ecx, [c4MaxIter]    ; Set loop counter to maximum
        jecxz   NoOpExit            ; If c4MaxIter is 0, then exit
        jmp     SHORT OpCont
    NoOpExit:
        jmp     NoOp
    OpCont:

        FLDZ
        FSTP    [f8Total]           ; Zero out the total

        mov     edx, 1              ; Use edx as a test mask, we will shift it
        mov     ebx, [c4CurBit]
        mov     esi, [ebx]          ; Dereference to get the current bit to esi
        mov     ebx, [c4SeqMap]     ; Load up the sequence map to ebx
        mov     [c4Period], 0       ; Init period

        FLD     [f8AVal]
        FLD     [f8BVal]            ; Load up the A and B values
        FLD     [f8XComponent]      ; Load up the x component
        ;--------------------
        ; ST0=f8XComponent
        ; ST1=f8BVal
        ; ST2=f8AVal
        ;--------------------

    IterLoop:

        cmp     esi, 0              ; Have we made a loop through the sequence?
        jnz     NextSeq             ; If not, then move up in the seque

        mov     esi, [c4BitCnt]     ; Reset the bit count
        mov     edx, 1              ; Reset the test mask
        jmp     SHORT LoadR         ; And go load the correct R

    NextSeq:
        shl     edx, 1              ; Shift up the test mask

    LoadR:
        dec     esi                 ; And bump down the bit counter
        test    ebx, edx            ; See if the bit is on
        jz      LoadAVal            ; If not, then this is an A val

        FLD     ST(1)               ; Else, it is a B val
        jmp     SHORT CalcX         ; And lets go calculate the new X

    LoadAVal:
        FLD     ST(2)               ; Load up A as the new R

    CalcX:                          ; Now calc x = rx(1-x)
        FLD1                        ; Load up a 1
        ;--------------------
        ; ST0=1
        ; ST1=fRVal
        ; ST2=f8XComponent
        ; ST3=f8BVal
        ; ST4=f8AVal
        ;--------------------

        FLD     ST(2)               ; Load another copy of f8XComponent
        ;--------------------
        ; ST0=f8XComponent
        ; ST1=1
        ; ST2=fRVal
        ; ST3=f8XComponent
        ; ST4=f8BVal
        ; ST5=f8AVal
        ;--------------------

        FSUB
        ;--------------------
        ; ST0=(1-f8XComponent)
        ; ST1=fRVal
        ; ST2=f8XComponent
        ; ST3=f8BVal
        ; ST4=f8AVal
        ;--------------------

        FXCH    ST(1)
        ;--------------------
        ; ST0=fRVal
        ; ST1=(1-f8XComponent)
        ; ST2=f8XComponent
        ; ST3=f8BVal
        ; ST4=f8AVal
        ;--------------------

        FXCH    ST(2)
        ;--------------------
        ; ST0=f8XComponent
        ; ST1=(1-f8XComponent)
        ; ST2=fRVal
        ; ST3=f8BVal
        ; ST4=f8AVal
        ;--------------------

        FMUL
        ;--------------------
        ; ST0=f8XComponent*(1-f8XComponent)
        ; ST1=fRVal
        ; ST2=f8BVal
        ; ST3=f8AVal
        ;--------------------

        FMUL    ST, ST(1)
        ;--------------------
        ; ST0=fNewXVal
        ; ST1=fRVal
        ; ST2=f8BVal
        ; ST3=f8AVal
        ;--------------------

        FLD     ST(1)               ; Push another copy of fRVal
        mov     [c4Tmp], 2
        WAIT
        FILD    [c4Tmp]             ; Load up a constant 2
        ;--------------------
        ; ST0=2
        ; ST1=fRVal
        ; ST2=fNewXVal
        ; ST3=fRVal
        ; ST4=f8BVal
        ; ST5=f8AVal
        ;--------------------

        FMUL
        ;--------------------
        ; ST0=2*fRVal
        ; ST1=fNewXVal
        ; ST2=fRVal
        ; ST3=f8BVal
        ; ST4=f8AVal
        ;--------------------

        FMUL    ST, ST(1)
        ;--------------------
        ; ST0=fNewXVal*2*fRVal
        ; ST1=fNewXVal
        ; ST2=fRVal
        ; ST3=f8BVal
        ; ST4=f8AVal
        ;--------------------

        FSUBP   ST(2), ST
        ;--------------------
        ; ST0=fNewXVal
        ; ST1=fRVal-(fNewXVal*2*fRVal)
        ; ST2=f8BVal
        ; ST3=f8AVal
        ;--------------------

        FXCH
        ;--------------------
        ; ST0=fRVal-(fNewXVal*2*fRVal)
        ; ST1=fNewXVal
        ; ST2=f8BVal
        ; ST3=f8AVal
        ;--------------------

        FABS
        ;--------------------
        ; ST0=fabs(fRVal-(fNewXVal*2*fRVal))
        ; ST1=fNewXVal
        ; ST2=f8BVal
        ; ST3=f8AVal
        ;--------------------

        FLD1                        ; Load a 1.0 constant
        FXCH
        ;--------------------
        ; ST0=fabs(fRVal-(fNewXVal*2*fRVal))
        ; ST1=1.0
        ; ST2=fNewXVal
        ; ST3=f8BVal
        ; ST4=f8AVal
        ;--------------------

        FYL2X
        ;--------------------
        ; ST0=log(fabs(fRVal-(fNewXVal*2*fRVal)))
        ; ST1=fNewXVal
        ; ST2=f8BVal
        ; ST3=f8AVal
        ;--------------------

        FLDLN2                      ; Load Log(2)
        ;--------------------
        ; ST0=log(2)
        ; ST1=log(fabs(fRVal-(fNewXVal*2*fRVal)))
        ; ST2=fNewXVal
        ; ST3=f8BVal
        ; ST4=f8AVal
        ;--------------------

        FDIV                        ; Divide the log by log(2)
        ;--------------------
        ; ST0=log(fabs(fRVal-(fNewXVal*2*fRVal))) / log(2)
        ; ST1=fNewXVal
        ; ST2=f8BVal
        ; ST3=f8AVal
        ;--------------------

        FLD     [f8Total]           ; Load the total
        FADD                        ; Add current to the total and pop
        ;--------------------
        ; ST0=NewTotal
        ; ST1=fNewXVal
        ; ST2=f8BVal
        ; ST3=f8AVal
        ;--------------------

        FSTP    [f8Total]           ; Put total back and pop
        ;--------------------
        ; ST0=fNewXVal
        ; ST1=f8BVal
        ; ST2=f8AVal
        ;--------------------

        loop    IterLoop            ; And do another loop

        FCOMP                       ; Clean up the NP stack using junk ops
        FCOMPP

        mov     ebx, [c4CurBit]
        mov     [ebx], esi          ; Pass back the current bit position

        FLD     [f8Total]           ; Load the total in ST(0)
        FILD    [c4MaxIter]         ; Load the the iterations
        ;--------------------
        ; ST0=ulMaxItem
        ; ST1=f8Total
        ;--------------------

        FDIV
        ;--------------------
        ; ST0=(f8Total/ulMaxItem)
        ;--------------------
        FSTP    [f8Ret]
    NoOp:
    }
    #else
    tCIDLib::TFloat8    f8XVal;
    tCIDLib::TFloat8    f8RVal;
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < c4MaxIter-1; c4Ind++)
    {
        if (c4SeqMap & (0x1UL << c4CurBit))
            f8RVal = f8AVal;
        else
            f8RVal = f8BVal;

        f8XVal = f8RVal*f8XVal * (1-f8XVal);
        f8Total +=  TMathLib::f8Log(TMathLib::f8Abs(f8RVal-(2*f8RVal*f8XVal)))
                    /  kCIDLib::f8Loge_2;

        c4CurBit++;
        if (c4CurBit == c4BitCnt)
            c4CurBit = 0;
    }
    f8Total /= c4MaxIter;
    #endif
    return f8Ret;
}


// -----------------------------------------------------------------------------
//  Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TLyapunov::CalcPixel(   const   TQ1Point&   pntPixel
                        , const TFPoint&    pntFractal
                        ,       TFracCache& fcachToStoreIn) const
{
    //
    //  Use the x pixel value to know what bit in the sequence map we
    //  are in. Its just the modulus of the x position by the number of
    //  map bits.
    //
    tCIDLib::TCard4 c4CurMapBit = pntPixel.c4X() % __c4MapBits;

    tCIDLib::TFloat8 f8Val = f8LyapLoop
    (
        __c4SeqMap
        , __c4MapBits
        , c4CurMapBit
        , __c4MaxIter
        , pntFractal.f8X()
        , pntFractal.f8Y()
        , __f8InitialX
    );

    // Store the result in the passed data cache
    fcachToStoreIn.PutAt(pntPixel.c4X(), pntPixel.c4Y(), f8Val);
}
