//
// NAME: CIDFrac_Escape.Cpp
//
// DESCRIPTION: 
//
//  This module implements the TEscapeFractal class, a derivative of the base
//  fractal class that servers as the basis for the whole family of escape
//  time fractals.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 12/02/95
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//



// -----------------------------------------------------------------------------
//  Facility includes
// -----------------------------------------------------------------------------
#include    "CIDFractal_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macro
// -----------------------------------------------------------------------------
RTTIData(TEscapeFractal,TBaseFractal)




// -----------------------------------------------------------------------------
//   CLASS: TEscapeFractal
//  PREFIX: frac
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TEscapeFractal: Constructors and Destructors
// -----------------------------------------------------------------------------

TEscapeFractal::~TEscapeFractal()
{
}


// -----------------------------------------------------------------------------
//  TEscapeFractal: Hidden constructors and operators
// -----------------------------------------------------------------------------

TEscapeFractal::TEscapeFractal() :

    TBaseFractal(tCIDFractal::EElemType_Card1)
    , __c4MaxIterations(16)
    , __eInsideMap(tCIDFractal::EInsideMap_Prisoner)
    , __eOutsideMap(tCIDFractal::EOutsideMap_Iterations)
{
}

TEscapeFractal::TEscapeFractal( const   tCIDLib::ESymmetries    eSymmetry
                                , const tCIDLib::TCard4         c4MaxIterations) :

    TBaseFractal(tCIDFractal::EElemType_Card1)
    , __c4MaxIterations(c4MaxIterations)
    , __eInsideMap(tCIDFractal::EInsideMap_Prisoner)
    , __eOutsideMap(tCIDFractal::EOutsideMap_Iterations)
{
    //
    //  Go back now and update the element type, now that we know the mapping
    //  and max iterations.
    //
    //  <TBD> Take into account mapping modes. Right now we just assume
    //  that we map to to the escape time iteration.
    //
    eElementType(TFracCache::eTypeForIterations(c4MaxIterations));
}

TEscapeFractal::TEscapeFractal(const TEscapeFractal& fracToCopy) :

    TBaseFractal(fracToCopy)
    , __c4MaxIterations(fracToCopy.__c4MaxIterations)
    , __eInsideMap(fracToCopy.__eInsideMap)
    , __eOutsideMap(fracToCopy.__eOutsideMap)
{
}


TEscapeFractal& TEscapeFractal::operator=(const TEscapeFractal& fracToAssign)
{
    if (this == &fracToAssign)
        return *this;

    // Call our parent first
    TParent::operator=(fracToAssign);

    // And now copy our members over
    __c4MaxIterations   = fracToAssign.__c4MaxIterations;
    __eInsideMap        = fracToAssign.__eInsideMap;
    __eOutsideMap       = fracToAssign.__eOutsideMap;

    return *this;
}


tCIDLib::TBoolean
TEscapeFractal::operator==(const TEscapeFractal& fracToTest) const
{
    // Call our parent version
    if (!TParent::operator==(fracToTest))
        return kCIDLib::False;

    if (fracToTest.__c4MaxIterations != __c4MaxIterations)
        return kCIDLib::False;

    if (fracToTest.__eInsideMap != __eInsideMap)
        return kCIDLib::False;

    if (fracToTest.__eOutsideMap != __eOutsideMap)
        return kCIDLib::False;

    return kCIDLib::True;
}

// -----------------------------------------------------------------------------
//  TEscapeFractal: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TEscapeFractal::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Call our parent's version
    TParent::_StreamFrom(strmToReadFrom);

    strmToReadFrom >> __c4MaxIterations;
    strmToReadFrom >> __eInsideMap;
    strmToReadFrom >> __eOutsideMap;
}

tCIDLib::TVoid
TEscapeFractal::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Call our parent's version
    TParent::_StreamTo(strmToWriteTo);

    strmToWriteTo << __c4MaxIterations;
    strmToWriteTo << __eInsideMap;
    strmToWriteTo << __eOutsideMap;
}


// -----------------------------------------------------------------------------
//  TEscapeFractal: Protected, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TEscapeFractal::_StoreValue(const   tCIDLib::TCard4     c4Escape
                            , const tCIDLib::TFloat8    f8ZReal
                            , const tCIDLib::TFloat8    f8ZImag
                            , const tCIDLib::TFloat8    f8CReal
                            , const tCIDLib::TFloat8    f8CImage
                            , const TQ1Point&           pntPixel
                            ,       TFracCache&         fcachTarget) const
{
    tCIDLib::TCard4 c4StoredIter = 0;
    if (c4Escape < __c4MaxIterations)
        c4StoredIter = c4Escape;

    // <TBD> Take into account the other storage mechanism
    fcachTarget.PutAt(pntPixel.c4X(), pntPixel.c4Y(), c4StoredIter);
}
