//
// NAME: CIDLib_DataCache.Hpp
//
// DESCRIPTION: 
//
//  This is the data cache class for the fractal engine. The calc engine
//  uses one of these to store the calculated information. The information is
//  in the form of an escape value for each pixel in the image. So the cache
//  is a 2D array of values. It has to be smart enough to handle storage
//  of 1, 2, or 4 bytes according to the maximum iterations for the image. This
//  can make a big difference for a large image, 1 to 3 meg for a 1024x1024
//  image, so its worth it.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 06/04/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//   CLASS: TFracCache
//  PREFIX: fcach
// -----------------------------------------------------------------------------
class CIDFRACTALEXP TFracCache :

    public TObject, public MStreamable, public MDuplicable
{
    public  :
        // ---------------------------------------------------------------------
        //  Class specific types
        // ---------------------------------------------------------------------
        enum    eInitModes
        {
            eInitMode_Init
            , eInitMode_NoInit
        };

        // ---------------------------------------------------------------------
        //  Constructors and destructors
        // ---------------------------------------------------------------------
        TFracCache
        (
            const   tCIDFractal::EElemTypes eType
            , const TSize&                  szPelImage
        );

        TFracCache
        (
            const   tCIDLib::TCard4         c4MaxValueToStore
            , const TSize&                  szPelImage
        );

        TFracCache
        (
            const   TFracCache&             fcachToCopy
        );

        ~TFracCache();


        // ---------------------------------------------------------------------
        //  Public, static methods
        // ---------------------------------------------------------------------
        static tCIDLib::TCard4 c4BytesForElemType
        (
            const   tCIDFractal::EElemTypes eType
        );

        static tCIDFractal::EElemTypes eTypeForIterations
        (
            const   tCIDLib::TCard4         c4MaxIterations
        );


        // ---------------------------------------------------------------------
        //  Public operators
        // ---------------------------------------------------------------------
        TFracCache& operator=
        (
            const   TFracCache&             fcachToAssign
        );

        tCIDLib::TBoolean operator==
        (
            const   TFracCache&             fcachToCompare
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TFracCache&             fcachToCompare
        )   const;


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TCard4 c4At
        (
            const   tCIDLib::TCard4         c4X
            , const tCIDLib::TCard4         c4Y
        )   const;

        tCIDLib::TCard4 c4At
        (
            const   TSize&                  szElement
        )   const;

        tCIDLib::TCard4 c4BufferSize() const;

        tCIDLib::TCard4 c4BytesPerElem() const;

        tCIDFractal::EElemTypes eType() const;

        tCIDLib::TFloat8 f8At
        (
            const   tCIDLib::TCard4         c4X
            , const tCIDLib::TCard4         c4Y
        )   const;

        tCIDLib::TFloat8 f8At
        (
            const   TSize&                  szElement
        )   const;

        tCIDLib::TVoid PutAt
        (
            const   tCIDLib::TCard4         c4X
            , const tCIDLib::TCard4         c4Y
            , const tCIDLib::TCard4         c4NewValue
        );

        tCIDLib::TVoid PutAt
        (
            const   tCIDLib::TCard4         c4X
            , const tCIDLib::TCard4         c4Y
            , const tCIDLib::TFloat8        f8NewValue
        );

        tCIDLib::TVoid QueryCacheLine
        (
            const   tCIDLib::TCard4         c4RowNumber
            ,       TBaseArray<tCIDLib::TCard4>& baToFill
        )   const;

        tCIDLib::TVoid QueryCacheLine
        (
            const   tCIDLib::TCard4         c4RowNumber
            ,       TBaseArray<tCIDLib::TFloat8>& baToFill
        )   const;

        tCIDLib::TVoid Realloc
        (
            const   tCIDFractal::EElemTypes eType
            , const tCIDLib::TCard4         c4CX
            , const tCIDLib::TCard4         c4CY
        );

        tCIDLib::TVoid Realloc
        (
            const   tCIDFractal::EElemTypes eType
            , const TSize&                  szImage
        );


    protected   :
        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


    private :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        TFracCache();


        // ---------------------------------------------------------------------
        //  Private, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid __AllocBuf
        (
            const   eInitModes              eMode
        );

        tCIDLib::TCard4 __c4OffsetFor
        (
            const   tCIDLib::TCard4         c4X
            , const tCIDLib::TCard4         c4Y
        )   const;

        tCIDLib::TVoid __CheckIndex
        (
            const   tCIDLib::TCard4         c4X
            , const tCIDLib::TCard4         c4Y
            , const tCIDLib::TCard4         c4Line
        )   const;


        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __pc1Data
        //      This is the 2D array of elements. Its dyanmically allocated
        //      to be the size needed. Its also 'looked at' differently
        //      according to what element type we have.
        //
        //  ----- The pointer members come first -----
        //
        //  __c4CX
        //  __c4CY
        //      This is the size of the cache in both dimensions. So the full
        //      size is ((__c4CX*__c4CY)*elemsize)
        //
        //  __eType
        //      This is the type of element we use.
        //
        //  __mtxLock
        //      This is the semaphore used to control access to the cache.
        //      This class is intended to be accessed by multiple threads most
        //      of the time.
        // ---------------------------------------------------------------------
        tCIDLib::TCard1*        __pc1Data;

        tCIDLib::TCard4         __c4CX, __c4CY;
        tCIDFractal::EElemTypes __eType;
        TMutex                  __mtxLock;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TFracCache,TObject)
        DefPolyDup(TFracCache)
};

#pragma pack(pop)


// -----------------------------------------------------------------------------
//  Public operators
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean
TFracCache::operator!=(const TFracCache& fcachToCompare) const
{
    return !operator==(fcachToCompare);
}


// -----------------------------------------------------------------------------
//  Public, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::TCard4 TFracCache::c4BufferSize() const
{
    return (__c4CX * __c4CY) * c4BytesForElemType(__eType);
}

inline tCIDLib::TCard4 TFracCache::c4BytesPerElem() const
{
    return c4BytesForElemType(__eType);
}

inline tCIDFractal::EElemTypes TFracCache::eType() const
{
    return __eType;
}
