//
// NAME: CIDLib_DataCache.Cpp
//
// DESCRIPTION: 
//
//  This Cpp file exists just to hold a couple of statements that force some
//  instantiations of the data cache template class. We need to have a Cpp file
//  that can be used to force the creation of the static members required for
//  the standard RTTI/Typing support.
//
//  Others can create their own instantiations, but are responsible for the
//  correct declarations of statics.
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/14/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDFractal_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macro
// -----------------------------------------------------------------------------
RTTIData(TFracCache,TObject)



// -----------------------------------------------------------------------------
//   CLASS: TFracCache
//  PREFIX: cache
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TFracCache: Public, static methods
// -----------------------------------------------------------------------------

tCIDFractal::EElemTypes
TFracCache::eTypeForIterations(const tCIDLib::TCard4 c4MaxIterations)
{
    if (c4MaxIterations <= 0xFF)
        return tCIDFractal::EElemType_Card1;
    else if (c4MaxIterations <= 0xFFFF)
        return tCIDFractal::EElemType_Card2;

    return tCIDFractal::EElemType_Card4;
}


tCIDLib::TCard4
TFracCache::c4BytesForElemType(const tCIDFractal::EElemTypes eType)
{
    switch(eType)
    {
        case tCIDFractal::EElemType_Card1 :
            return 1;
            break;

        case tCIDFractal::EElemType_Card2 :
            return 2;
            break;

        case tCIDFractal::EElemType_Card4 :
            return 4;
            break;

        case tCIDFractal::EElemType_Float4 :
            return 4;
            break;

        case tCIDFractal::EElemType_Float8 :
            return 8;
            break;

        default :
            #if CID_DEBUG_ON
            facCIDFractal.LogErr
            (
                __FILE__
                , __LINE__
                , kFractErrs::errcGen_InvalidValue
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_Internal
                , TInteger(eType)
                , facCIDFractal.strMsg(kFractMsgs::midCache_ElemType)
            );
            #endif
            break;
    }

    // Make the compiler happy
    return 0;
}


// -----------------------------------------------------------------------------
//  TFracCache: Constructors and destructors
// -----------------------------------------------------------------------------

TFracCache::TFracCache( const   tCIDFractal::EElemTypes eType
                        , const TSize&                  szPelImage) :

    __pc1Data(0)
    , __c4CX(szPelImage.c4Width())
    , __c4CY(szPelImage.c4Height())
    , __eType(eType)
    , __mtxLock
      (
         tCIDLib::ELockState_Unlocked
      )
{
    // Allocate the buffer and init to zeros
    __AllocBuf(eInitMode_Init);
}

TFracCache::TFracCache( const   tCIDLib::TCard4 c4MaxValueToStore
                        , const TSize&          szPelImage) :
    __pc1Data(0)
    , __c4CX(szPelImage.c4Width())
    , __c4CY(szPelImage.c4Height())
    , __eType(eTypeForIterations(c4MaxValueToStore))
    , __mtxLock
      (
          tCIDLib::ELockState_Unlocked
      )
{
    // Allocate the buffer and init to zeros
    __AllocBuf(eInitMode_Init);
}

TFracCache::TFracCache(const TFracCache& fcachToCopy) :

    __pc1Data(0)
    , __c4CX(fcachToCopy.__c4CX)
    , __c4CY(fcachToCopy.__c4CY)
    , __eType(fcachToCopy.__eType)
    , __mtxLock
      (
          tCIDLib::ELockState_Unlocked
      )
{
    //
    //  Allocate the buffer, indicating no initialize since we are going
    //  to copy over it here anyway.
    //
    __AllocBuf(eInitMode_NoInit);

    // Copy over the caller's buffer
    TRawMem::CopyMemBuf(__pc1Data, fcachToCopy.__pc1Data, c4BufferSize());
}

TFracCache::~TFracCache()
{
    delete __pc1Data;
    __pc1Data = 0;
}


// -----------------------------------------------------------------------------
//  TFracCache: Public operators
// -----------------------------------------------------------------------------

TFracCache&
TFracCache::operator=(const TFracCache& fcachToAssign)
{
    if (this == &fcachToAssign)
        return *this;

    //
    //  See if we need to reallocate. Since we are faking the 2 dimensions,
    //  and element type, we only have to reallocate if the absolute size is
    //  different. The two dimensions and element type that give that size can
    //  change and we won't care.
    //
    tCIDLib::TBoolean bRealloc = kCIDLib::True;
    if (c4BufferSize() == fcachToAssign.c4BufferSize())
        bRealloc = kCIDLib::False;

    // Copy over the members
    __c4CX = fcachToAssign.__c4CX;
    __c4CY = fcachToAssign.__c4CY;
    __eType = fcachToAssign.__eType;

    // If we need to reallocate then do it
    if (bRealloc)
    {
        delete __pc1Data;
        __pc1Data = 0;
        __AllocBuf(eInitMode_NoInit);
    }

    // Copy over the contents of the source
    TRawMem::CopyMemBuf(__pc1Data, fcachToAssign.__pc1Data, c4BufferSize());

    return *this;
}

tCIDLib::TBoolean
TFracCache::operator==(const TFracCache& fcachToCompare) const
{
    // See if the simple stuff matches first
    if ((__c4CX != fcachToCompare.__c4CX)
    ||  (__c4CY != fcachToCompare.__c4CY)
    ||  (__eType != fcachToCompare.__eType))
    {
        return kCIDLib::False;
    }

    //
    //  The buffers should match exactly, which includes their existence
    //  or non-existence.
    //
    if (__pc1Data && fcachToCompare.__pc1Data)
    {
        if (TRawMem::eCompareMemBuf
        (
            __pc1Data
            , fcachToCompare.__pc1Data
            , c4BufferSize()))
        {
            return kCIDLib::False;
        }
    }
     else if (!__pc1Data || !fcachToCompare.__pc1Data)
    {
        return kCIDLib::False;
    }

    return kCIDLib::True;
}


// -----------------------------------------------------------------------------
//  TFracCache: Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TCard4 TFracCache::c4At( const   tCIDLib::TCard4 c4X
                                  , const tCIDLib::TCard4 c4Y) const
{
    __CheckIndex(c4X, c4Y, __LINE__);

    tCIDLib::TCard4     c4Offset = __c4OffsetFor(c4X, c4Y);
    tCIDLib::TCard1*    pc1Elem  = __pc1Data+c4Offset;
    switch(__eType)
    {
        case tCIDFractal::EElemType_Card1 :
            return tCIDLib::TCard4(*((tCIDLib::TCard1*)pc1Elem));

        case tCIDFractal::EElemType_Card2 :
            return tCIDLib::TCard4(*((tCIDLib::TCard2*)pc1Elem));

        case tCIDFractal::EElemType_Card4 :
            return *((tCIDLib::TCard4*)pc1Elem);

        case tCIDFractal::EElemType_Float4 :
            return tCIDLib::TCard4(*((tCIDLib::TFloat4*)pc1Elem));

        case tCIDFractal::EElemType_Float8 :
            return tCIDLib::TCard4(*((tCIDLib::TFloat8*)pc1Elem));

        default :
            #if CID_DEBUG_ON
            facCIDFractal.LogErr
            (
                __FILE__
                , __LINE__
                , kFractErrs::errcGen_InvalidValue
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_Internal
                , TInteger(__eType)
                , facCIDFractal.strMsg(kFractMsgs::midCache_ElemType)
            );
            #endif
            break;
    }
    // Make the compiler happy
    return 0;
}

tCIDLib::TCard4 TFracCache::c4At(const TQ1Point& pntElement) const
{
    return c4At(pntElement.c4X(), pntElement.c4Y());
}


tCIDLib::TFloat8
TFracCache::f8At(   const   tCIDLib::TCard4 c4X
                    , const tCIDLib::TCard4 c4Y) const
{
    __CheckIndex(c4X, c4Y, __LINE__);

    tCIDLib::TCard4     c4Offset = __c4OffsetFor(c4X, c4Y);
    tCIDLib::TCard1*    pc1Elem  = __pc1Data+c4Offset;

    switch(__eType)
    {
        case tCIDFractal::EElemType_Card1 :
            return tCIDLib::TFloat8(*((tCIDLib::TCard1*)pc1Elem));

        case tCIDFractal::EElemType_Card2 :
            return tCIDLib::TFloat8(*((tCIDLib::TCard2*)pc1Elem));

        case tCIDFractal::EElemType_Card4 :
            return tCIDLib::TFloat8(*((tCIDLib::TCard4*)pc1Elem));

        case tCIDFractal::EElemType_Float4 :
            return tCIDLib::TFloat8(*((tCIDLib::TFloat4*)pc1Elem));

        case tCIDFractal::EElemType_Float8 :
            return *((tCIDLib::TFloat8*)pc1Elem);

        default :
            #if CID_DEBUG_ON
            facCIDFractal.LogErr
            (
                __FILE__
                , __LINE__
                , kFractErrs::errcGen_InvalidValue
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_Internal
                , TInteger(__eType)
                , facCIDFractal.strMsg(kFractMsgs::midCache_ElemType)
            );
            #endif
            break;
    }
    // Make the compiler happy
    return 0.0;
}

tCIDLib::TFloat8 TFracCache::f8At(const TQ1Point& pntElement) const
{
    return f8At(pntElement.c4X(), pntElement.c4Y());
}


tCIDLib::TVoid
TFracCache::PutAt(  const   tCIDLib::TCard4 c4X
                    , const tCIDLib::TCard4 c4Y
                    , const tCIDLib::TCard4 c4NewValue)
{
    __CheckIndex(c4X, c4Y, __LINE__);

    tCIDLib::TCard4     c4Offset = __c4OffsetFor(c4X, c4Y);
    tCIDLib::TCard1*    pc1Elem  = __pc1Data+c4Offset;

    switch(__eType)
    {
        case tCIDFractal::EElemType_Card1 :
            *((tCIDLib::TCard1*)pc1Elem) = tCIDLib::TCard1(c4NewValue);
            break;

        case tCIDFractal::EElemType_Card2 :
            *((tCIDLib::TCard2*)pc1Elem) = tCIDLib::TCard2(c4NewValue);
            break;

        case tCIDFractal::EElemType_Card4 :
            *((tCIDLib::TCard4*)pc1Elem) = c4NewValue;
            break;

        case tCIDFractal::EElemType_Float4 :
            *((tCIDLib::TFloat4*)pc1Elem) = tCIDLib::TFloat4(c4NewValue);
            break;

        case tCIDFractal::EElemType_Float8 :
            *((tCIDLib::TFloat8*)pc1Elem) = tCIDLib::TFloat8(c4NewValue);
            break;

        default :
            #if CID_DEBUG_ON
            facCIDFractal.LogErr
            (
                __FILE__
                , __LINE__
                , kFractErrs::errcGen_InvalidValue
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_Internal
                , TInteger(__eType)
                , facCIDFractal.strMsg(kFractMsgs::midCache_CacheElem)
            );
            #endif
            break;
    }
}

tCIDLib::TVoid
TFracCache::PutAt(  const   tCIDLib::TCard4     c4X
                    , const tCIDLib::TCard4     c4Y
                    , const tCIDLib::TFloat8    f8NewValue)
{
    __CheckIndex(c4X, c4Y, __LINE__);

    tCIDLib::TCard4     c4Offset = __c4OffsetFor(c4X, c4Y);
    tCIDLib::TCard1*    pc1Elem  = __pc1Data+c4Offset;

    switch(__eType)
    {
        case tCIDFractal::EElemType_Card1 :
            *((tCIDLib::TCard1*)pc1Elem) = tCIDLib::TCard1(f8NewValue);
            break;

        case tCIDFractal::EElemType_Card2 :
            *((tCIDLib::TCard2*)pc1Elem) = tCIDLib::TCard2(f8NewValue);
            break;

        case tCIDFractal::EElemType_Card4 :
            *((tCIDLib::TCard4*)pc1Elem) = tCIDLib::TCard4(f8NewValue);
            break;

        case tCIDFractal::EElemType_Float4 :
            *((tCIDLib::TFloat4*)pc1Elem) = tCIDLib::TFloat4(f8NewValue);
            break;

        case tCIDFractal::EElemType_Float8 :
            *((tCIDLib::TFloat8*)pc1Elem) = f8NewValue;
            break;

        default :
            #if CID_DEBUG_ON
            facCIDFractal.LogErr
            (
                __FILE__
                , __LINE__
                , kFractErrs::errcGen_InvalidValue
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_Internal
                , TInteger(__eType)
                , facCIDFractal.strMsg(kFractMsgs::midCache_ElemType)
            );
            #endif
            break;
    }
}


tCIDLib::TVoid
TFracCache::QueryCacheLine( const   tCIDLib::TCard4 c4RowNumber
                            ,       TBaseArray<tCIDLib::TCard4>& baToFill) const
{
    //
    //  The row number be a valid row and the array must be big enough to
    //  hold the row. If either is not true, we log an exception.
    //
    if (c4RowNumber >= __c4CY)
    {
        facCIDFractal.LogErr
        (
            __FILE__
            , __LINE__
            , kFractErrs::errcGen_IndexError
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , facCIDFractal.strMsg(kFractMsgs::midCache_Row)
            , TCardinal(__c4CY)
            , TCardinal(c4RowNumber)
        );
    }

    if (baToFill.c4ElementCount() < __c4CX)
    {
        facCIDFractal.LogErr
        (
            __FILE__
            , __LINE__
            , kFractErrs::errcCache_ArraySize
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
            , TCardinal(baToFill.c4ElementCount())
            , TCardinal(__c4CX)
        );
    }

    //
    //  Get a pointer to the data, given the element type and the desired
    //  x,y pair.
    //
    const tCIDLib::TCard1* pc1Cur = __pc1Data + __c4OffsetFor(0, c4RowNumber);

    //
    //  The element type has to be one of the cardinal types. If not, then
    //  we throw an exception.
    //
    switch(__eType)
    {
        case tCIDFractal::EElemType_Card1 :
        {
            for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4CX; c4Index++)
            {
                baToFill[c4Index] = tCIDLib::TCard4(*pc1Cur);
                pc1Cur++;
            }
        }
        break;

        case tCIDFractal::EElemType_Card2 :
        {
            // Get a card2 pointer at the data
            const tCIDLib::TCard2* pc2Cur = (const tCIDLib::TCard2*)pc1Cur;
            for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4CX; c4Index++)
            {
                baToFill[c4Index] = tCIDLib::TCard4(*pc2Cur);
                pc2Cur++;
            }
        }
        break;

        case tCIDFractal::EElemType_Card4 :
        {
            // Get a card4 pointer at the data
            const tCIDLib::TCard4* pc4Cur = (const tCIDLib::TCard4*)pc1Cur;
            for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4CX; c4Index++)
            {
                baToFill[c4Index] = *pc4Cur;
                pc4Cur++;
            }
        }
        break;

        default :
            #if  CID_DEBUG_ON
            facCIDFractal.LogErr
            (
                __FILE__
                , __LINE__
                , kFractErrs::errcGen_InvalidValue
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_Internal
                , TInteger(__eType)
                , facCIDFractal.strMsg(kFractMsgs::midCache_CacheElem)
            );
            #endif
            break;
    }
}

tCIDLib::TVoid
TFracCache::QueryCacheLine( const   tCIDLib::TCard4 c4RowNumber
                            ,       TBaseArray<tCIDLib::TFloat8>& baToFill) const
{
    //
    //  The row number be a valid row and the array must be big enough to
    //  hold the row. If either is not true, we log an exception.
    //
    if (c4RowNumber >= __c4CY)
    {
        facCIDFractal.LogErr
        (
            __FILE__
            , __LINE__
            , kFractErrs::errcGen_IndexError
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , facCIDFractal.strMsg(kFractMsgs::midCache_Row)
            , TCardinal(__c4CY)
            , TCardinal(c4RowNumber)
        );
    }

    if (baToFill.c4ElementCount() < __c4CX)
    {
        facCIDFractal.LogErr
        (
            __FILE__
            , __LINE__
            , kFractErrs::errcCache_ArraySize
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
            , TCardinal(baToFill.c4ElementCount())
            , TCardinal(__c4CX)
        );
    }

    //
    //  Get a pointer to the data, given the element type and the desired
    //  x,y pair.
    //
    const tCIDLib::TCard1* pc1Cur = __pc1Data + __c4OffsetFor(0, c4RowNumber);

    switch(__eType)
    {
        case tCIDFractal::EElemType_Card1 :
        {
            for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4CX; c4Index++)
            {
                baToFill[c4Index] = tCIDLib::TFloat8(*pc1Cur);
                pc1Cur++;
            }
        }
        break;

        case tCIDFractal::EElemType_Card2 :
        {
            // Get a card2 pointer at the data
            const tCIDLib::TCard2* pc2Cur = (const tCIDLib::TCard2*)pc1Cur;
            for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4CX; c4Index++)
            {
                baToFill[c4Index] = tCIDLib::TFloat8(*pc2Cur);
                pc2Cur++;
            }
        }
        break;

        case tCIDFractal::EElemType_Card4 :
        {
            // Get a card4 pointer at the data
            const tCIDLib::TCard4* pc4Cur = (const tCIDLib::TCard4*)pc1Cur;
            for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4CX; c4Index++)
            {
                baToFill[c4Index] = tCIDLib::TFloat8(*pc4Cur);
                pc4Cur++;
            }
        }
        break;

        case tCIDFractal::EElemType_Float4 :
        {
            // Get a float4 pointer at the data
            const tCIDLib::TFloat4* pf4Cur = (const tCIDLib::TFloat4*)pc1Cur;
            for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4CX; c4Index++)
            {
                baToFill[c4Index] = tCIDLib::TFloat8(*pf4Cur);
                pf4Cur++;
            }
        }
        break;

        case tCIDFractal::EElemType_Float8 :
        {
            // Get a float4 pointer at the data
            const tCIDLib::TFloat8* pf8Cur = (const tCIDLib::TFloat8*)pc1Cur;
            for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4CX; c4Index++)
            {
                baToFill[c4Index] = *pf8Cur;
                pf8Cur++;
            }
        }
        break;

        default :
            #if  CID_DEBUG_ON
            facCIDFractal.LogErr
            (
                __FILE__
                , __LINE__
                , kFractErrs::errcGen_InvalidValue
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_Internal
                , TInteger(__eType)
                , facCIDFractal.strMsg(kFractMsgs::midCache_CacheElem)
            );
            #endif
            break;
    }
}


tCIDLib::TVoid
TFracCache::Realloc(const tCIDFractal::EElemTypes eNewType, const TSize& szImage)
{
    //
    //  See if we need to reallocate. Since we are faking the 2 dimensions,
    //  and element type, we only have to reallocate if the absolute size is
    //  different. The two dimensions and element type that give that size can
    //  change and we won't care.
    //
    tCIDLib::TBoolean bRealloc = kCIDLib::True;
    if (c4BufferSize() == (szImage.c4SquareUnits() * c4BytesForElemType(eNewType)))
        bRealloc = kCIDLib::False;

    // Copy over the members
    __c4CX = szImage.c4Width();
    __c4CY = szImage.c4Height();
    __eType = eNewType;

    // If we need to reallocate then do it
    if (bRealloc)
    {
        delete __pc1Data;
        __pc1Data = 0;
        __AllocBuf(eInitMode_Init);
    }
     else
    {
        // Just zero the buffer we got
        TRawMem::SetMemBuf(__pc1Data, tCIDLib::TCard1(0), c4BufferSize());
    }
}

// -----------------------------------------------------------------------------
//  TFracCache: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TFracCache::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Get the new size and element type to temps
    tCIDLib::TCard4  c4NewCX, c4NewCY;
    tCIDFractal::EElemTypes eNewType;

    strmToReadFrom >> eNewType;
    strmToReadFrom >> c4NewCX;
    strmToReadFrom >> c4NewCY;

    // Make sure that the type is valid
    if (!bIsValidEnum(eNewType))
    {
        facCIDFractal.LogErr
        (
            __FILE__
            , __LINE__
            , kFractErrs::errcGen_InvalidValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_StreamFormat
            , TInteger(eNewType)
            , facCIDFractal.strMsg(kFractMsgs::midCache_ElemType)
        );
    }

    // Calc the buffer size this will require.
    tCIDLib::TCard4 c4NewSize = (c4NewCX * c4NewCY) * c4BytesForElemType(eNewType);

    //
    //  See if we need to reallocate. Since we are faking the 2 dimensions,
    //  and element type, we only have to reallocate if the absolute size is
    //  different. The two dimensions and element type that give that size can
    //  change and we won't care.
    //
    tCIDLib::TBoolean bRealloc = kCIDLib::True;
    if (c4BufferSize() == c4NewSize)
        bRealloc = kCIDLib::False;

    // Copy over the members
    __c4CX  = c4NewCX;
    __c4CY  = c4NewCY;
    __eType = eNewType;

    // If we need to reallocate then do it
    if (bRealloc)
    {
        delete __pc1Data;
        __pc1Data = 0;
        __AllocBuf(eInitMode_NoInit);
    }

    // Read in the array
    switch(__eType)
    {
        case tCIDFractal::EElemType_Card1 :
            strmToReadFrom.ReadRawBuffer(__pc1Data, __c4CX * __c4CY);
            break;

        case tCIDFractal::EElemType_Card2 :
            strmToReadFrom.ReadCard2Array
            (
                (tCIDLib::TCard2*)__pc1Data
                , __c4CX * __c4CY
            );
            break;

        case tCIDFractal::EElemType_Card4 :
            strmToReadFrom.ReadCard4Array
            (
                (tCIDLib::TCard4*)__pc1Data
                , __c4CX * __c4CY
            );
            break;

        case tCIDFractal::EElemType_Float4 :
            strmToReadFrom.ReadFloat4Array
            (
                (tCIDLib::TFloat4*)__pc1Data
                , __c4CX * __c4CY
            );
            break;

        case tCIDFractal::EElemType_Float8 :
            strmToReadFrom.ReadFloat8Array
            (
                (tCIDLib::TFloat8*)__pc1Data
                , __c4CX * __c4CY
            );
    }
}

tCIDLib::TVoid
TFracCache::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    strmToWriteTo << __eType;
    strmToWriteTo << __c4CX;
    strmToWriteTo << __c4CY;

    switch(__eType)
    {
        case tCIDFractal::EElemType_Card1 :
            strmToWriteTo.ReadRawBuffer(__pc1Data, __c4CX * __c4CY);
            break;

        case tCIDFractal::EElemType_Card2 :
            strmToWriteTo.ReadCard2Array
            (
                (tCIDLib::TCard2*)__pc1Data
                , __c4CX * __c4CY
            );
            break;

        case tCIDFractal::EElemType_Card4 :
            strmToWriteTo.ReadCard4Array
            (
                (tCIDLib::TCard4*)__pc1Data
                , __c4CX * __c4CY
            );
            break;

        case tCIDFractal::EElemType_Float4 :
            strmToWriteTo.ReadFloat4Array
            (
                (tCIDLib::TFloat4*)__pc1Data
                , __c4CX * __c4CY
            );
            break;

        case tCIDFractal::EElemType_Float8 :
            strmToWriteTo.ReadFloat8Array
            (
                (tCIDLib::TFloat8*)__pc1Data
                , __c4CX * __c4CY
            );
            break;

        default :
            #if CID_DEBUG_ON
            facCIDFractal.LogErr
            (
                __FILE__
                , __LINE__
                , kFractErrs::errcGen_InvalidValue
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_Internal
                , TInteger(__eType)
                , facCIDFractal.strMsg(kFractMsgs::midCache_ElemType)
            );
            #endif
            break;
    }
}


// -----------------------------------------------------------------------------
//  TFracCache: Private, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __AllocBuf
//
// DESCRIPTION:
//
//  This method is called internally to allocate the buffer. It uses the
//  currently set size and element type and assumes that any previous
//  buffer is already cleaned up if needed.
// ---------------------------------------
//   INPUT: eMode is the init mode to use.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFracCache::__AllocBuf(const eInitModes eMode)
{
    __pc1Data = new tCIDLib::TCard1[c4BufferSize()];

    if (eMode == eInitMode_Init)
        TRawMem::SetMemBuf(__pc1Data, tCIDLib::TCard1(0), c4BufferSize());
}


//
// FUNCTION/METHOD NAME: __c4OffsetFor
//
// DESCRIPTION:
//
//  Returns the byte offset of the given element, given the current element
//  size.
// ---------------------------------------
//   INPUT: c4X, c4Y is the element to get the offset for
//
//  OUTPUT: None
//
//  RETURN: The raw offset of the element.
//
tCIDLib::TCard4
TFracCache::__c4OffsetFor(  const   tCIDLib::TCard4 c4X
                            , const tCIDLib::TCard4 c4Y) const
{
    return ((__c4CX * c4Y) + c4X) * c4BytesForElemType(__eType);
}


//
// FUNCTION/METHOD NAME: __CheckIndex
//
// DESCRIPTION:
//
//  This method will check the passed index against the cache size and log
//  an exception if its not legal.
// ---------------------------------------
//   INPUT: c4X, c4Y are the points to test.
//          pszMethod is the name of the calling method, to put into the
//              message logged if any.
//          c4Line is the line of the caller so we can put that in the mesasge
//              logged if any.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid
TFracCache::__CheckIndex(   const   tCIDLib::TCard4     c4X
                            , const tCIDLib::TCard4     c4Y
                            , const tCIDLib::TCard4     c4Line) const
{
    if ((c4X >= __c4CX) || (c4Y >= __c4CY))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , c4Line
            , kFractErrs::errcGen_IndexError
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_BadParms
            , TString(L"TFracCache")
            , TSize(__c4CX, __c4CY)
            , TSize(c4X, c4Y)
        );
    }
}
