//
// NAME: CIDFrac_CalcEngine.Hpp
//
// DESCRIPTION: 
//
//  This is the header for the CIDFrac_CalcEngine.Cpp module, which implements
//  the abstract calculation engine for the fractal system. A single calculation
//  engine handles the calculation of one image. A program can create as many
//  engines (simultaneously or over time) as desired, as the available hardware
//  can handle usefully. Each engine object will start up one or more threads
//  which do the calculation, leaving the calculated data in a cache object
//  owned by the calc engine. This data can then be queried out and displayed
//  in any means desirable.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 06/11/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//  CLASS: TFracCalcEngine
// PREFIX: fceng
// -----------------------------------------------------------------------------
class CIDFRACTALEXP  TFracCalcEngine :

    public TObject, public MStreamable, public MDuplicable
{
    public  :
        // ---------------------------------------------------------------------
        //  Constructors and Destructors.
        // ---------------------------------------------------------------------
        ~TFracCalcEngine();


        // ---------------------------------------------------------------------
        //  Public, virtual methods
        // ---------------------------------------------------------------------
        virtual tCIDLib::TVoid StartProcessing() = 0;

        virtual tCIDLib::TVoid StopProcessing() = 0;


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean bGetNextPoint
        (
                    TQ1Point&               pntNextPel
            ,       TFPoint&                pntNextFractal
        );

        tCIDLib::TBoolean bImageComplete();

        tCIDLib::TVoid CalcPixel
        (
            const   TQ1Point&               pntPixel
            , const TFPoint&                pntFractal
        );

        tCIDFractal::EElemTypes eElemType() const;

        tCIDLib::TFloat8 f8PerPixelX() const;

        tCIDLib::TFloat8 f8PerPixelY() const;

        tCIDLib::TVoid QueryCacheLine
        (
            const   tCIDLib::TCard4         c4RowNumber
            ,       TBaseArray<tCIDLib::TCard4>& baToFill
        )   const;

        tCIDLib::TVoid QueryCacheLine
        (
            const   tCIDLib::TCard4         c4RowNumber
            ,       TBaseArray<tCIDLib::TFloat8>& baToFill
        )   const;


    protected   :
        // ---------------------------------------------------------------------
        //  Hidden constructors and operators
        //
        //  These are hidden because we are an abstract base class. But they
        //  are all implemented.
        // ---------------------------------------------------------------------
        TFracCalcEngine();

        TFracCalcEngine
        (
            const   TSize&                  szPelImage
            , const TFArea&                 fareaImage
            ,       TBaseFractal* const     pfracToAdopt
            ,       TStatusController* const pstatcToAdopt
        );

        TFracCalcEngine
        (
            const   TFracCalcEngine&        fcengToCopy
        );

        TFracCalcEngine& operator=
        (
            const   TFracCalcEngine&        fcengToAssign
        );


        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


        // ---------------------------------------------------------------------
        //  Protected, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _SetImageInfo
        (
            const   TSize&                  szPelImage
            , const TFArea&                 fareaImage
            ,       TBaseFractal* const     pfracToAdopt
            ,       TStatusController* const pstatcToAdopt
         );

    private :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean operator==(const TFracCalcEngine&) const;


        // ---------------------------------------------------------------------
        //  Private, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid __CalcTransients();



        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __pfracImage
        //      This is the fractal that will be used to create the image.
        //
        //  __pstatcToUse
        //      This is the status controller that we should use to
        //      communicate out status to the code that stated us.
        //
        //  !!  The above members need to be first so they get set before
        //      anything else could happen.
        //
        //  __c4CurCol
        //  __c4CurRow
        //      These represent the next row and column to be calculated.
        //      This class 'hands out' new points to the threads that are
        //      doing the work. The are initialized to 0,0.
        //
        //  __eSymmetry
        //      This is the actual symmetry used. Each fractal object will
        //      tell us what their natural symmetry is; but, we only can
        //      make use of it when the image is centered over the fractal
        //      space origin. Its set when our image info is set, either
        //      during construction or in SetImageInfo().
        //
        //  __f8PerPixelX
        //  __f8PerPixelY
        //      These are precalculated when the fractal image area and/or
        //      pixel area chances. They hold the amount of fractal coordinate
        //      space per pixel coordinate. This is used in a lot of calcs,
        //      so its worth keeping it precalced.
        //
        //  __fareaSpace
        //      This is the fractal area of the image, i.e. the floating
        //      point coordinate space of the fractal world. Note that the
        //      fractal coordinates are, unlike the image's pixel size, not
        //      usually 0 based. So it cannot be just a size, it has to
        //      cover the whole area.
        //
        //  __fcachData
        //      This is where the data is stored. The fractal object knows
        //      what size elements it needs and tells us, so that we can
        //      create it. Then we just let him put data into it since he
        //      knows the needed format.
        //
        //  __mtxNextPelLock
        //      This is the semaphore used to serialize access to the current
        //      col and row members. This information in particular needs to
        //      specific protection because its accessed by multiple threads
        //      all the time.
        //
        //  __szPelImage
        //      This is the pixel area of the image, i.e. the size of the
        //      image in pixels. The fractal area, __fareaFracArea, is spread
        //      out evenly over this pixel area. So effectively this creates
        //      a 2D grid of discrete 'sample points' in the fractal space.
        // ---------------------------------------------------------------------
        TBaseFractal*           __pfracImage;
        TStatusController*      __pstatcToUse;

        tCIDLib::TCard4         __c4CurCol;
        tCIDLib::TCard4         __c4CurRow;
        tCIDLib::ESymmetries    __eSymmetry;
        tCIDLib::TFloat8        __f8PerPixelX;
        tCIDLib::TFloat8        __f8PerPixelY;
        TFArea                  __fareaSpace;
        TFracCache              __fcachData;
        TMutex                  __mtxNextPelLock;
        TSize                   __szPelImage;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TFracCalcEngine,TObject)
};

#pragma pack(pop)



// -----------------------------------------------------------------------------
//  Inline public non-virtuals
// -----------------------------------------------------------------------------
inline tCIDFractal::EElemTypes TFracCalcEngine::eElemType() const
{
    return __pfracImage->eElementType();
}

inline tCIDLib::TFloat8 TFracCalcEngine::f8PerPixelX() const
{
    return __f8PerPixelX;
}

inline tCIDLib::TFloat8 TFracCalcEngine::f8PerPixelY() const
{
    return __f8PerPixelY;
}

inline tCIDLib::TVoid
TFracCalcEngine::QueryCacheLine(const   tCIDLib::TCard4             c4RowNumber
                                ,       TBaseArray<tCIDLib::TCard4>& baToFill) const
{
    __fcachData.QueryCacheLine(c4RowNumber, baToFill);
}

inline tCIDLib::TVoid
TFracCalcEngine::QueryCacheLine(const   tCIDLib::TCard4 c4RowNumber
                                ,       TBaseArray<tCIDLib::TFloat8>& baToFill) const
{
    __fcachData.QueryCacheLine(c4RowNumber, baToFill);
}

