//
// NAME: CIDFrac_CalcEngine.Cpp
//
// DESCRIPTION: 
//
//  This module implements the TFracCalcEngine abstract base class, which
//  is the root of a family of classes that perform calculations on fractal
//  images.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 06/10/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  Facility includes
// -----------------------------------------------------------------------------
#include    "CIDFractal_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macros
// -----------------------------------------------------------------------------
RTTIData(TFracCalcEngine,TObject)



// -----------------------------------------------------------------------------
//   CLASS: TFracCalcEngine
//  PREFIX: fceng
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TFracCalcEngine: Constructors and Destructors
// -----------------------------------------------------------------------------

TFracCalcEngine::~TFracCalcEngine()
{
    delete __pfracImage;
    delete __pstatcToUse;
}


// -----------------------------------------------------------------------------
//  TFracCalcEngine: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TBoolean
TFracCalcEngine::bGetNextPoint(TQ1Point& pntPel, TFPoint& pntFractal)
{
    {
        // Use a locker to get control of the current row/col info
        TLocker lockNextPnt(&__mtxNextPelLock);

        // If we done, then return right now with a false
        if (__c4CurRow >= __szPelImage.c4Height())
        {
            // Tell the status controller we are done
            if (__pstatcToUse)
            {
                __pstatcToUse->WorkComplete
                (
                    facCIDFractal.strMsg(kFractMsgs::midCEng_RenderComplete)
                );
            }
            return kCIDLib::False;
        }

        // Put the current color and row into the point
        pntPel.Set(__c4CurCol, __c4CurRow);

        // Handle moving to the next point
        __c4CurCol++;
        if (__c4CurCol >= __szPelImage.c4Width())
        {
            //
            //  Tell the status controller our progress. Only do it every
            //  10 rows.
            //
            if (__pstatcToUse && !(__c4CurRow % 10))
            {
                TStringStream  strmMsg(64);
                strmMsg << facCIDFractal.strMsg(kFractMsgs::midCEng_FinishedRow)
                        << __c4CurRow;
                __pstatcToUse->Working
                (
                    strmMsg.strData(), __c4CurRow / __szPelImage.c4Height()
                );
            }
            __c4CurCol = 0;
            __c4CurRow++;
        }
    }

    //
    //  Now other threads can have at it. We've got our copy of the point
    //  and can continue our work. We need to calculate the fractal
    //  coordinates for this one.
    //
    pntFractal.Set
    (
        __fareaSpace.f8Left() + (__f8PerPixelX * __c4CurCol)
        , __fareaSpace.f8Bottom() + (__f8PerPixelY * __c4CurRow)
    );

    return kCIDLib::True;
}


tCIDLib::TBoolean TFracCalcEngine::bImageComplete()
{
    // Use a locker to get control of the current row/col info
    TLocker lockNextPnt(&__mtxNextPelLock);

    if (__c4CurRow >= __szPelImage.c4Height())
        return kCIDLib::True;

    return kCIDLib::False;
}


tCIDLib::TVoid
TFracCalcEngine::CalcPixel( const   TQ1Point&   pntPixel
                            , const TFPoint&    pntFractal)
{
    #if CID_DEBUG_ON
    if (!__pfracImage)
    {
        facCIDFractal.LogErr
        (
            __FILE__
            , __LINE__
            , kFractErrs::errcCEng_NotSetUp
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
        );
    }
    #endif

    __pfracImage->CalcPixel(pntPixel, pntFractal, __fcachData);

    //
    //  <TBD> Validate this point. I.e. we know this one is in the bag
    //  and does not have to be redone if we close down now. Up until now
    //  the calc engine's values showed the next available point, so if
    //  we closed before then, it would have not known to do this point
    //  again. By providing the validation, the engine can move ahead
    //  and give out more work, but never lose any pels if we close before
    //  a pel is actually completed.
    //

}


// -----------------------------------------------------------------------------
//  TFracCalcEngine: Hidden constructors and operators
// -----------------------------------------------------------------------------

TFracCalcEngine::TFracCalcEngine() :

    __pfracImage(0)
    , __pstatcToUse(0)

    , __c4CurCol(0)
    , __c4CurRow(0)
    , __fareaSpace()
    , __fcachData(tCIDFractal::EElemType_Card1, TSize(8, 8))
    , __mtxNextPelLock(tCIDLib::ELockState_Unlocked)
    , __szPelImage()
{
    __CalcTransients();
}

TFracCalcEngine::TFracCalcEngine(const   TSize&              szPelImage
                                 , const TFArea&             fareaSpace
                                 ,       TBaseFractal* const pfracToAdopt
                                 , TStatusController* const  pstatcToAdopt) :
    __pfracImage(pfracToAdopt)
    , __pstatcToUse(pstatcToAdopt)

    , __c4CurCol(0)
    , __c4CurRow(0)
    , __fareaSpace(fareaSpace)
    , __fcachData(pfracToAdopt->eElementType(), szPelImage)
    , __f8PerPixelX(0)
    , __f8PerPixelY(0)
    , __mtxNextPelLock(tCIDLib::ELockState_Unlocked)
    , __szPelImage(szPelImage)
{
    __CalcTransients();
}

TFracCalcEngine::TFracCalcEngine(const TFracCalcEngine& fcengToCopy) :

    __pfracImage(0)
    , __pstatcToUse(0)

    , __c4CurCol(fcengToCopy.__c4CurCol)
    , __c4CurRow(fcengToCopy.__c4CurRow)
    , __fareaSpace(fcengToCopy.__fareaSpace)
    , __fcachData(fcengToCopy.__fcachData)
    , __f8PerPixelX(fcengToCopy.__f8PerPixelX)
    , __f8PerPixelY(fcengToCopy.__f8PerPixelY)
    , __mtxNextPelLock(tCIDLib::ELockState_Unlocked)
    , __szPelImage(fcengToCopy.__szPelImage)
{
    //
    //  Polymorphically duplicate the fractal object and status control, if there
    //  is one
    //
    if (fcengToCopy.__pfracImage)
        __pfracImage = ::pDupObject<TBaseFractal>(*fcengToCopy.__pfracImage);

    if (fcengToCopy.__pstatcToUse)
        __pstatcToUse = ::pDupObject<TStatusController>(*fcengToCopy.__pstatcToUse);
}

TFracCalcEngine&
TFracCalcEngine::operator=(const TFracCalcEngine& fcengToAssign)
{
    // Copy over the by value stuff
    __c4CurCol          = fcengToAssign.__c4CurCol;
    __c4CurRow          = fcengToAssign.__c4CurRow;
    __eSymmetry         = fcengToAssign.__eSymmetry;
    __fareaSpace        = fcengToAssign.__fareaSpace;
    __fcachData         = fcengToAssign.__fcachData;
    __f8PerPixelX       = fcengToAssign.__f8PerPixelX;
    __f8PerPixelY       = fcengToAssign.__f8PerPixelY;
    __szPelImage        = fcengToAssign.__szPelImage;

    // Delete any existing image object and status controller
    delete __pfracImage;
    __pfracImage = 0;
    delete __pstatcToUse;
    __pstatcToUse = 0;

    // And polymorphically duplicate them from the source
    if (fcengToAssign.__pfracImage)
        __pfracImage = ::pDupObject<TBaseFractal>(*fcengToAssign.__pfracImage);

    if (fcengToAssign.__pstatcToUse)
        __pstatcToUse = ::pDupObject<TStatusController>(*fcengToAssign.__pstatcToUse);

    return *this;
}


// -----------------------------------------------------------------------------
//  TFracCalcEngine: Protected, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TFracCalcEngine::_SetImageInfo( const   TSize&              szPelImage
                                , const TFArea&             fareaSpace
                                ,       TBaseFractal* const pfracToAdopt
                                , TStatusController* const  pstatcToUse)
{
    //
    //  Make sure that the image was not already set. We don't allow just
    //  reseting the image info over an existing one.
    //
    if (__pfracImage || __pstatcToUse)
    {
        facCIDFractal.LogErr
        (
            __FILE__
            , __LINE__
            , kFractErrs::errcCEng_AlreadySet
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_Already
        );
    }

    __pfracImage        = pfracToAdopt;
    __pstatcToUse       = pstatcToUse;
    __c4CurCol          = 0;
    __c4CurRow          = 0;
    __fareaSpace        = fareaSpace;
    __szPelImage        = szPelImage;

    // Calc the transient stuff now that we know our info
    __CalcTransients();

    // Reallocate the cache
    __fcachData.Realloc(__pfracImage->eElementType(), szPelImage);
}


// -----------------------------------------------------------------------------
//  TFracCalcEngine: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TFracCalcEngine::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Stream in our fundamentals
    strmToReadFrom >> __c4CurCol;
    strmToReadFrom >> __c4CurRow;
    strmToReadFrom >> __eSymmetry;

    if (!bIsValidEnum(__eSymmetry))
    {
        facCIDFractal.LogErr
        (
            __FILE__
            , __LINE__
            , kFractErrs::errcGen_InvalidValue
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_Unknown
            , TInteger(__eSymmetry)
            , facCIDFractal.strMsg(kFractMsgs::midCEng_FractSymmetry)
        );
    }

    // And our by value objects
    strmToReadFrom >> __fareaSpace;
    strmToReadFrom >> __szPelImage;

    // Delete the current fractal object
    delete __pfracImage;
    __pfracImage = 0;

    // See if we stored a fractal object
    tCIDLib::TCard1 c1Stored;
    strmToReadFrom >> c1Stored;

    // If we stored one, then read it back in (polymorphically of course)
    if (c1Stored)
        ::PolymorphicRead(__pfracImage, strmToReadFrom);
}

tCIDLib::TVoid
TFracCalcEngine::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Stream out our fundamentals
    strmToWriteTo << __c4CurCol;
    strmToWriteTo << __c4CurRow;
    strmToWriteTo << __eSymmetry;

    // And our by value objects
    strmToWriteTo << __fareaSpace;
    strmToWriteTo << __szPelImage;

    //
    //  If we have a fractal object, stream it out too. We stream out a
    //  byte to denote whether we have one or not.
    //
    if (__pfracImage)
    {
        strmToWriteTo << tCIDLib::TCard1(1);
        ::PolymorphicWrite(__pfracImage, strmToWriteTo);
    }
     else
    {
        strmToWriteTo << tCIDLib::TCard1(0);
    }
}


// -----------------------------------------------------------------------------
//  TFracCalcEngine: Private, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TFracCalcEngine::__CalcTransients()
{
    __f8PerPixelX = __fareaSpace.f8CXDiv(__szPelImage.c4Width());
    __f8PerPixelY = __fareaSpace.f8CYDiv(__szPelImage.c4Height());
}
