//
// NAME: CIDCrypto_Key.Cpp
//
// DESCRIPTION:
//
//  This method implements the TCryptoKey class, which is a simple class to
//  abstract the concept of encryption keys.
//
// AUTHOR: Dean Roddey
//
// CREATE TDate: 10/24/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDCrypto_.Hpp"


// ----------------------------------------------------------------------------
//  Do our RTTI macros
// ----------------------------------------------------------------------------
RTTIData2(TCryptoKey, TObject)


// ----------------------------------------------------------------------------
//   CLASS: TCryptoKey
//  PREFIX: ckey
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TCryptoKey: Constructors and Destructor
// ----------------------------------------------------------------------------

TCryptoKey::TCryptoKey(const TString& strKey) :

    __c4Bytes(0)
    , __pc1Key(0)
{
    if (strKey.bEmpty())
    {
        facCIDCrypto.LogErr
        (
            __FILE__
            , __LINE__
            , kCryptoErrs::errcKey_EmptyKey
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
        );
    }

    __c4Bytes = strKey.c4Length() * kCIDLib::c4CharBytes;
    __pc1Key = new tCIDLib::TCard1[__c4Bytes];
    TRawMem::CopyMemBuf(__pc1Key, strKey.pszData(), __c4Bytes);
}

TCryptoKey::TCryptoKey(const TMemBuf& mbufKey, const tCIDLib::TCard4 c4KeyBytes) :

    __c4Bytes(c4KeyBytes)
    , __pc1Key(0)
{
    // If zero, then use the current bytes in buffer
    if (!c4KeyBytes)
    {
        __c4Bytes = mbufKey.c4Size();

        // If that's zero too, then an error
        if (!__c4Bytes)
        {
            facCIDCrypto.LogErr
            (
                __FILE__
                , __LINE__
                , kCryptoErrs::errcKey_EmptyKey
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_AppError
            );
        }
    }
    else if (__c4Bytes > mbufKey.c4Size())
    {
        // The bytes indicated is larger than the data supplied
        facCIDCrypto.LogErr
        (
            __FILE__
            , __LINE__
            , kCryptoErrs::errcKey_BadKeyBytes
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , TCardinal(__c4Bytes)
            , TCardinal(mbufKey.c4Size())
        );
    }

    // Ok, we can allocate our buffer and copy the info out
    __pc1Key = new tCIDLib::TCard1[__c4Bytes];
    TRawMem::CopyMemBuf(__pc1Key, mbufKey.pc1Data(), __c4Bytes);
}

TCryptoKey::TCryptoKey(const TCryptoKey& ckeyToCopy) :

    __c4Bytes(ckeyToCopy.__c4Bytes)
    , __pc1Key(0)
{
    __pc1Key = new tCIDLib::TCard1[__c4Bytes];
    TRawMem::CopyMemBuf(__pc1Key, ckeyToCopy.__pc1Key, __c4Bytes);
}

TCryptoKey::~TCryptoKey()
{
    delete __pc1Key;
    __pc1Key = 0;
}


// ----------------------------------------------------------------------------
//  TCryptoKey: Public operators
// ----------------------------------------------------------------------------

TCryptoKey& TCryptoKey::operator=(const TCryptoKey& ckeyToAssign)
{
    // Reallocate the buffer if its not the same
    if (__c4Bytes != ckeyToAssign.__c4Bytes)
    {
        delete __pc1Key;
        __pc1Key = 0;
        __c4Bytes = ckeyToAssign.__c4Bytes;
        __pc1Key = new tCIDLib::TCard1[__c4Bytes];
    }
    TRawMem::CopyMemBuf(__pc1Key, ckeyToAssign.__pc1Key, __c4Bytes);
    return *this;
}


// ----------------------------------------------------------------------------
//  TCryptoKey: Hidden Constructors
// ----------------------------------------------------------------------------

TCryptoKey::TCryptoKey() :

    __c4Bytes(0)
    , __pc1Key(0)
{
}


// ----------------------------------------------------------------------------
//  TCryptoKey: Protected, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TCryptoKey::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Check that we get a start of object marker
    TBinaryStream::EStreamMarkers eMarker;
    strmToReadFrom >> eMarker;

    if (eMarker != TBinaryStream::EStreamMarker_StartObject)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_NoStartMarker
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_StreamFormat
            , TInteger(eMarker)
        );
    }

    // Stream in the key bytes to a temp
    tCIDLib::TCard4 c4TmpBytes;
    strmToReadFrom >> c4TmpBytes;

    // If the bytes are different, then reallocate our buffer
    if (c4TmpBytes != __c4Bytes)
    {
        delete __pc1Key;
        __pc1Key = 0;
        __pc1Key = new tCIDLib::TCard1[c4TmpBytes];
        __c4Bytes = c4TmpBytes;
    }

    // And now read in the new data
    strmToReadFrom.ReadCard1Array(__pc1Key, __c4Bytes);
}

tCIDLib::TVoid TCryptoKey::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Stream out a start of object stream marker
    strmToWriteTo << TBinaryStream::EStreamMarker_StartObject;

    // Stream out the count of key bytes and then the bytes of data
    strmToWriteTo << __c4Bytes;
    strmToWriteTo.WriteCard1Array(__pc1Key, __c4Bytes);
}
