/*===========================================================================
SOLAR slrreply v0.94 :: Module extract.c

This source code has been released into the public domain.

History:  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
03-29-93 KJH  Adaptation from old mail.c and news.c
07-01-93 KJH  Added check for moderated newsgroups. Crossposting
              to moderated groups is not supported.
07-06-93 KJH  Added multiple return codes for scan_forums()
07-15-94 KJH  Changed all printf() to fprintf().
===========================================================================*/

/* Header Files */
#include <string.h>
#include <stdio.h>
#include <dir.h>
#include <stdlib.h>
#include <time.h>

/* Local Definitions */
#define MSGID_VERSION            "0931b"   /* Version for msgid field */
#define BUFSIZE                   1024    /* Size of buffer          */
#define REPLY_TMP       "SOLARREP.TMP"    /* Temporary reply message */
#define NEWS_HEAD_DENY  "NEWSHEAD.DNY"    /* News header deny list   */
#define MAIL_HEAD_DENY  "MAILHEAD.DNY"    /* Mail header deny list   */
#define NEWS_HEAD_ORDER "NEWSHEAD.ORD"    /* News header order list  */
#define MAIL_HEAD_ORDER "MAILHEAD.ORD"    /* Mail header order list  */
#define YES                          0
#define NO                           1

/* External Functions */
extern char *rfctime();                 /* From rfctime.c  */
extern int  scan_forums(char *header);  /* From forums.c   */

/* External Data */
extern char _slrerr[80];                /* From slrreply.c */
extern long _maxnewssize;               /* From slrreply.c */
extern long _maxmailsize;               /* From slrreply.c */
extern char solar_path[MAXPATH];        /* From slrreply.c */
extern char username[10];               /* From slrreply.c */
extern char uucp_name[10];              /* From slrreply.c */
extern char domain_name[40];            /* From slrreply.c */
extern char real_name[20];              /* From slrreply.c */
extern char organization[128];          /* From slrreply.c */
extern char msg_type;                   /* From process.c  */
extern char mod_address[80];            /* From forums.c   */

/* Local Data Types */
typedef struct _Header {
  char   info[BUFSIZE];
  int    used;
  struct _Header *next;
} HEADER;

/* Local Functions */
int  allow_header(char *string);
int  dump_header(HEADER *header, FILE *work_file);
int  write_msgid(FILE *work_file);
long binlen(char bigendian[4]);

/* Local Data */
char msgbuf[BUFSIZE];
char msgbuf2[BUFSIZE];   /* Use in case of linefeed in big endian value */

/*
Function: extract_msg(FILE *msg_file, char msg_type)
Purpose : Extract a single message from a MSG file into a
          temporary file for processing. Most of the header
          processing is done here.
Return  : -2 = article size error, sets _slrerr
          -1 = fatal error, sets _slrerr
           0 = end of file
           msg size in bytes on success
*/

long extract_msg(FILE *msg_file)
{
  long   endpos       = 0L;       /* End of article in bytes from zero    */
  long   curpos       = 0L;       /* Current position in article in bytes */
  long   msg_size     = 0L;       /* Size of message processed in bytes   */
  char   bigendian[4];            /* Stores msg length in 4-byte value    */
  FILE   *work_file   = NULL;     /* Final message to feed mail/news pgm. */
  HEADER *header      = NULL;     /* Base of msg header linked list       */
  HEADER *current     = NULL;     /* Current node in msg header list      */
  HEADER *scanlist    = NULL;     /* Used to scan the header list         */

  /* Initialize endpos relative to the start
     of the message file */

  endpos = ftell(msg_file);

  /* Get binary message length */

  if (feof(msg_file)) goto HeaderDone;
  bigendian[0] = getc(msg_file);
  if (feof(msg_file)) goto HeaderDone;
  bigendian[1] = getc(msg_file);
  if (feof(msg_file)) goto HeaderDone;
  bigendian[2] = getc(msg_file);
  if (feof(msg_file)) goto HeaderDone;
  bigendian[3] = getc(msg_file);
  if (feof(msg_file)) goto HeaderDone;

  /* Read through the message header and store in memory */

  while (fgets(msgbuf,BUFSIZE,msg_file) != NULL)
  {
    if (msg_size == 0L)  /* Beginning of message? */
    {
      /* Find the ending position and message size */

      if ((msg_size = binlen(bigendian)) < 0)
      {
        strcpy(_slrerr,"cannot determine binary message length");
        goto ExitFunct;
      }
      endpos += msg_size + 4; /* Allow for size field */

      /* Check the message's size in bytes to make
         sure it is within allowable limits. */

      switch (msg_type) {
				/* Binary format news message */
        case 'B'  : if (msg_size > _maxnewssize)
                    {
                      sprintf(_slrerr,"maximum size of %lu bytes exceeded for news",msg_size);
                      fseek(msg_file,endpos,SEEK_SET); /* Advance message */
                      msg_size = -2L;
                      goto ExitFunct;
                    }
                    break;
        /* Binary format mail message */
        case 'b'  : if (msg_size > _maxmailsize)
                    {
                      sprintf(_slrerr,"maximum size of %lu bytes exceeded for mail",msg_size);
											fseek(msg_file,endpos,SEEK_SET); /* Advance message */
                      msg_size = -2L;
                      goto ExitFunct;
                    }
                    break;
      }
    }

    /* Set current byte position in message file */

    curpos = ftell(msg_file);

    /* Check to see if we are at the end
       of the header */

    if (strlen(msgbuf) < 3) break;

    /* If this is a type 'news' message and we have
       found the 'Newsgroups:' header, and gotta scan the
       forum files to check for moderated groups. */

    if ((msg_type == 'B') && (strnicmp(msgbuf,"Newsgroups:",11) == 0))
    {
      switch(scan_forums(msgbuf)) {

        /* An error occurred */
        case -1   : msg_size = -1L;
                    goto ExitFunct;

        /* Article is OK to post */
        case  0   : break;

        /* Group is marked /nosolar */
        case  1   : sprintf(_slrerr,"group marked for no posting");
                    msg_size = -1L;
                    goto ExitFunct;

        /* Group is moderated */
        case  2   : msg_type = 'b';
										fprintf(stdout,"To moderator: %s\n",mod_address);
                    sprintf(msgbuf,"To: %s\n",mod_address);
                    break;
      }
    }

    if (allow_header(msgbuf) == 0)
    {
			if (!header)
      {
        /* First line of header. Create header object. */
        header = current = malloc(sizeof(HEADER));
        if (!current)
        {
          strcpy(_slrerr,"insufficient memory to store header");
          msg_size = -1L;
          goto ExitFunct;
        }
        header->next = NULL;
        strcpy(header->info,msgbuf);
        header->used = NO;
      }
      else
			{
        /* Trace thru existing headers. If this one isn't already listed */
        /* add the header to the end of the list.                        */
        scanlist = header;
        while (scanlist)
        {
          if (strcmp(msgbuf,scanlist->info) == 0) break;
          scanlist = scanlist->next;
        }
        if (!scanlist)
        {
          /* Create a new node in the list */
          current->next = malloc(sizeof(HEADER));
          if (!current->next)
          {
            strcpy(_slrerr,"insufficient memory to store header");
            msg_size = -1L;
            goto ExitFunct;
          }
          current = current->next;
          current->next = NULL;
          strcpy(current->info,msgbuf);
          current->used = NO;
        }
      }
    }

    /* Check for end of message, stop reading if end. */
    if (curpos == endpos) break;

    /* Check for article over-run. Prescribe some Pepto pronto! */
    if (curpos > endpos)
    {
      strcpy(_slrerr,"message file out of sync");
      msg_size = -1L;
      goto ExitFunct;
    }
	}
HeaderDone:
  /* Check for end of message file */
  if (msg_size == 0L) goto ExitFunct;

  /* Write the message header to temporary reply file */
  if ((work_file = fopen(REPLY_TMP,"wb")) == NULL)
  {
    strcpy(_slrerr,"cannot open work file");
    msg_size = -1L;
    goto ExitFunct;
  }
  if (dump_header(header,work_file) != 0)
  {
		msg_size = -1L;
    goto ExitFunct;
  }

  /* Write the rest of the article as is. */
  fprintf(work_file,"%s",msgbuf);
  while (fgets(msgbuf,BUFSIZE,msg_file) != NULL)
  {
    fprintf(work_file,"%s",msgbuf);
    curpos = ftell(msg_file);
    if (curpos == endpos) break; /* End of message, this is OK */
    /* Check for article over-run. Prescribe some Pepto pronto! */
    if (curpos > endpos)
		{
      strcpy(_slrerr,"message file out of sync");
      msg_size = -1L;
      goto ExitFunct;
    }
  }

ExitFunct:
  if (work_file) fclose(work_file);
  if (header) free(header);
  return msg_size;
}

/*
Function: allow_header()
Purpose : Check header deny file for a match.
Return  : 0 to allow header, non-zero to deny header
*/

int  allow_header(char *string)
{
  FILE *deny_file = NULL;
  char deny_path[MAXPATH];
  char deny_string[80];

  strcpy(deny_path,solar_path);
	strcat(deny_path,"\\");
  switch (msg_type) {
    /* Binary news message */
    case 'B'  : strcat(deny_path,NEWS_HEAD_DENY);
                break;
    /* Binary mail message */
    case 'b'  : strcat(deny_path,MAIL_HEAD_DENY);
                break;
  }
  if ((deny_file = fopen(deny_path,"rt")) == NULL)
  {
    goto AllowExit;
  }
	while (fgets(deny_string,80,deny_file) != NULL)
  {
    if (strnicmp(deny_string,string,strlen(deny_string) - 1) == 0)
    {
      fclose(deny_file);
      goto DenyExit;
    }
  }
  fclose(deny_file);
AllowExit:
  return 0;
DenyExit:
  return 1;
}

/*
Function: int dump_header(HEADER *header, FILE *work_file)
Purpose : Write header to work file.
Return  : zero on success, non-zero on error (sets _slrerr)
*/

int  dump_header(HEADER *header, FILE *work_file)
{
  /* Local Data */
  FILE   *order_file = NULL;
  HEADER *thisone = header;
  char   order_path[MAXPATH];
  char   orderbuf[80];
  char   date_time[35];

	strcpy(order_path,solar_path);
  strcat(order_path,"\\");
  switch (msg_type) {
    /* Binary news message */
    case 'B'  : strcat(order_path,NEWS_HEAD_ORDER);
                break;
    /* Binary mail message */
    case 'b'  : strcat(order_path,MAIL_HEAD_ORDER);
                break;
  }
  if (strcpy(date_time,rfctime()) == NULL)
  {
    strcpy(_slrerr,"internal error creating rfctime");
    goto ErrorExit;
  }
  if ((order_file = fopen(order_path,"rt")) == NULL)
  {
    goto ErrorExit;
  }
  while (fgets(orderbuf,80,order_file) != NULL)
  {
		while (thisone)
    {
      if (strnicmp(thisone->info,orderbuf,strlen(orderbuf) - 1) == 0) break;
      thisone = thisone->next;
    }
    if (thisone != NULL)
    {
      fprintf(work_file,"%s",thisone->info);
      thisone->used = YES;
    }
    else
    {
      if (strnicmp(orderbuf,"Path:",5) == 0)
        fprintf(work_file,"Path: %s!%s\n",uucp_name,username);

      if (strnicmp(orderbuf,"From:",5) == 0)
      {
        fprintf(work_file,"From: %s@%s",username,domain_name);
        if (strcmp(real_name,"NONE") != 0)
        {
          fprintf(work_file," (%s)",real_name);
        }
        fprintf(work_file,"\n");
      }
      if (strnicmp(orderbuf,"Date:",5) == 0)
        fprintf(work_file,"Date: %s\n",date_time);
      if (strnicmp(orderbuf,"Message-Id:",11) == 0)
        write_msgid(work_file);
      if ((strnicmp(orderbuf,"Organization:",13) == 0) && (strcmp(organization,"NONE") != 0))
        fprintf(work_file,"Organization: %s\n",organization);
    }
    thisone = header;
  }
  fclose(order_file);
  while (thisone)
  {
		if (thisone->used == NO)
    {
      fprintf(work_file,"%s",thisone->info);
    }
    thisone = thisone->next;
  }
GoodExit:
  return 0;
ErrorExit:
  return -1;
}

/*
Function: long binlen(char bigendian[4])
Purpose : Convert 4-byte field into 32-bit big endian value.
Return  : Length of message in bytes, or 0 on EOF or error.
*/

long binlen(char bigendian[4])
{
  return (((((long)(bigendian[0])) & 0xFF) << 24) |
      ((((long)(bigendian[1])) & 0xFF) << 16) |
      ((((long)(bigendian[2])) & 0xFF) <<  8) |
      ((((long)(bigendian[3])) & 0xFF)));
}

/*
Function: write_msgid()
Purpose : Write a unique Message-ID: header based on the time.
Return  : 0 on success, non-zero on error.
*/

int write_msgid(FILE *work_file)
{
  time_t t;
  struct tm *gmt;

  t = time(NULL);
  gmt = gmtime(&t);

  fprintf(work_file,"Message-Id: <%u%u%u.%u%u.s%ss@%s>\n",gmt->tm_hour,  \
          gmt->tm_min,gmt->tm_sec,gmt->tm_yday,gmt->tm_year,MSGID_VERSION, \
					domain_name);

  return 0;
}

