/* logtr.c
 * -------
 *
 * A program to truncate a UUPC logfile by a given number of days.
 *
 * Based on ideas by Jiri Kuchta <kuchta.dcse.fee.vutbr.cz>
 *
 * Author: Mike Lawrie <mlawrie@apies.frd.ac.za>
 *
 * Version 1.01 - 3 January 94
 */

#define VERSION 1.01

#include <stdio.h>
#include <dos.h>
#include <direct.h>


void usage( progname ) char *progname;
{
    fprintf( stdout, "Usage: %s filename days\n", progname );
}

/*
 * Program returns:-
 * 0 = all OK
 * 1 = incorrect number of parameters
 * 2 = invalid number of days in the command line
 * 3 = cannot open the old log file
 * 4 = no filename was detected on the command line
 * 5 = invalid format in the existing log file
 * 6 = cannot open the new output file
 * 7 = could not close the old log file (1) (impossible?)
 * 8 = could not delete the old log file (1)
 * 9 = could not write to the new log file (disk full?)
 * 10 = could not write final record to new log file
 * 11 = could not close the old log file (2) (impossible?)
 * 12 = could not close the new log file (impossible?)
 * 13 = could not delete the old log file (2)
 * 14 = could not rename the new output file
 */

/*
 * The year-wrapping is a sod. Assume that the .LOG file is in chronological
 * order, and consider the date in the first record of the file. If this
 * is greater than today's date, then this record is for last year. So,
 * adjust the "today" date to be 365 days ahead, and this affects the
 * "delete" date by 365 days as well. (If leap years worry, you, then
 * adjust the program yourself :-).
 * 
 * When calculating the day of the year for each record, keep track of the
 * day that was found in the record. If the current record is earlier than
 * the previous day, then the .LOG file has gone through an end of year, so
 * add 365 days to every date in the .LOG file from this point onwards.
 * 
 * 
 *     days
 *       |                             +=============================== today
 *       |                             !              D                 +365
 *       |                             !             D
 *       |                             !            D
 *       |                             !  +--------d------------------- delete
 *       |                             !  !       d                     +365
 *       |                             !  !      d
 *  365  |.............................!..!.....d...................... 365
 *       |                             !  !    d  
 *       |   a                         !  !   dd  
 *       |  a                          !  !  d d  
 *       | a                           !  ! d  d  
 *       |                             !  !    d  
 * today |=============================+==!====d====================
 *       |            B           C       !    d      D
 *       |           B           C        !    d     D
 *       |          B           C         !    d    D
 * delete|---------------------c----------+----d---d----------------
 *       |                    c                d  d
 *       |                   c                 d d
 *       |                                     dd
 *       +----------------------------------------------------------> records
 *
 *Consider 4 files, viz A, B, C and D.
 *
 *   "aaa" deletes, all records are later than "today" so they are
 *         last year's records (ie very old).
 *
 *   "BBB" passes into the new log file, as they are all this year's
 *         records, and are later than the delete date.
 *
 *   "ccc" deletes, being this year's records but before the delete date.
 *
 *   "CCC" passes into the new log file.
 *
 *   "dddddddd/ddddd" deletes, being last year's and this year's records
 *         until the delete date.
 *
 *   "DDD" passes into the new log file.
 *
 * Clearly, the line "delete+365" can be below the "365" line while the
 * "today+365" line is above the "365" line, and the principles are not
 * altered.
 *
 */

#define EXT ".$$$"
#define LOGREC 200


int md[12] = {31,28,31,30,31,30,31,31,30,31,30,31};
int amd[12];

/*
 * int dayinstr( char *buf )
 * {
 *     int mth, day;
 * 
 *     mth = (int)(buf[0]-'0')*10 + (int)(buf[1]-'0');
 *     day = (int)(buf[3]-'0')*10 + (int)(buf[4]-'0');
 *     return( amd[mth-1] + day );
 * }
 */

int dayinstr( char *buf) {

    static int last;  /* Set to zero initially */
    int mth, day, sum;

    mth = (int)(buf[0]-'0')*10 + (int)(buf[1]-'0');
    day = (int)(buf[3]-'0')*10 + (int)(buf[4]-'0');
    sum = amd[mth-1] + day;

    if( sum < last )
        last = sum + 365;
    else
        last = sum;

    return( last );
}


main (argc, argv ) int argc; char *argv[]; {

    int deldays, deldate;
    int i;

    int today;
    struct date datebuf;

    FILE *oldf, *newf;
    char newfname[MAXPATH],
         drive[MAXDRIVE], dir[MAXDIR], file[MAXFILE], ext[MAXEXT];
    int oldflags;
    char logrec[LOGREC];
    long int oldfsz = 0;
    long int newfsz = 0;
    int  firstdate;


    fprintf( stderr, "Version %5.2f: LOGTR %s %s\n", VERSION, argv[1], argv[2] );

    if( argc != 3 )
    {
        usage( argv[0] );
        exit( 1 );
    }
/*
 * Calculate accumulated month-days once.
 * amd[0]=0, amd[1]=31, amd[2]=58/59, amd[3] 89/90,....
 * Note that this program is unlikely to be operating in year 2000, so
 * let's not get into an argument as to whether or not that is a leap year.
 */

    getdate( &datebuf );
    if( datebuf.da_year == 1996 )
        md[1]++;

    amd[0] = 0;
    for( i=1; i<=11; i++ )
        amd[i] = md[i-1] + amd[i-1];

/*
 * Test for a non-zero numeric value of days as first parameter of
 * the command line.
 */

    deldays = atoi( argv[2] );

    if( (deldays <= 0) || (deldays >365)  )
    {
        fprintf( stderr, "Invalid number of days in command line (%s) \n",
                 argv[2] );
        usage( argv[0] );
        exit( 2 );
    }

    today = datebuf.da_day + amd[datebuf.da_mon-1];

/*
 * Open the old log file as the input file.
 */

    if( !(oldf=fopen(argv[1], "r")) )
    {
        fprintf( stderr, "Cannot open the old log file (%s)\n", argv[1] );
        exit( 3 );
    }
    oldfsz = 0;


/*
 * Form the name of the output file, don't open it yet.
 */

    oldflags = fnsplit( argv[1], drive, dir, file, ext );
    if( !(oldflags & FILENAME) )
    {
    fprintf( stderr, "A filename must be specified on the command line (%s)\n",
                 argv[1] );
        usage( argv[0] );
        exit( 4 );
    }
    fnmerge( newfname, drive, dir, file, EXT );


/*
 * Read the old log file, find the first day in that file. Watch out for
 * a wrap of a year.
 */

    if( fgets(logrec, LOGREC, oldf) != NULL )
    {
        oldfsz += strlen( logrec ) + 1;
        firstdate = dayinstr( logrec );
    } else {
        fprintf( stderr, "Invalid log file format (%s)\n", argv[1] );
        exit( 5 );
    }

    if( firstdate > today )
        today += 365;
    deldate = today - deldays;

/*
 * It may be that there are no old records in the log file....
 */

    if( deldate < firstdate )
    {
        while( !feof(oldf) )
        {
            if( fgets(logrec, LOGREC, oldf) )
            {
                oldfsz += strlen( logrec ) + 1;
            } else {
            newfsz = oldfsz;
            }
        }
/*
 * Read the old log file until deletion date
 */

    } else {
        while( deldate > dayinstr(logrec)
               && (fgets(logrec, LOGREC, oldf)) )
            oldfsz += strlen( logrec ) + 1;

/*
 * Maybe EOF co-incides with the cut-off date?
 */

        if( feof(oldf) )
        {
            if( fclose(oldf) != NULL )
            {
                fprintf( stderr, "Could not close the old log file (1)\n" );
                exit( 7 );
            }
            if( unlink(argv[1]) != NULL )
            {
                fprintf( stderr, "Could not delete the old log file (1)\n" );
                exit( 8 );
            }
            newfsz = 0;
/*
 * Transcribe the wanted records until EOF on the old log file
 */

        } else {
/*
 * Open the new log file, there is something to write.
 */

            if( !(newf=fopen(newfname,"w")) )
            {
                fprintf( stderr, "Cannot open the new log file (%s)\n", newfname );
                exit( 6 );
            }

            while( !feof(oldf) )
            {
                if( fputs(logrec, newf) != NULL )
                {
                    fprintf( stderr, "Could not write to new log file\n" );
                    exit( 9 );
                }
                newfsz += strlen( logrec ) + 1;
                fgets( logrec, LOGREC, oldf );
                oldfsz += strlen( logrec ) + 1;   /* Counts extra at EOF? */
            }
            oldfsz--;

            if( fputs(logrec, newf) != NULL )
            {
                fprintf( stderr, "Could not write final record to new log file\n" );
                exit( 10 );
            }

            if( fclose(oldf) != NULL )
            {
                fprintf( stderr, "Could not close the old log file (2)\n" );
                exit( 11 );
            }

            if( fclose(newf) != NULL )
            {
                fprintf( stderr, "Could not close the new log file\n" );
                exit( 12 );
            }

            if( unlink(argv[1]) != NULL )
            {
                fprintf( stderr, "Could not delete the old log file (2)\n" );
                exit( 13 );
            }

            if( rename(newfname, argv[1]) != NULL )
            {
                fprintf( stderr, "Could not rename the output file\n" );
                exit( 14 );
            }
         }
    }

    fprintf( stderr, "%s: Old size=%li, new size=%li\n",
             argv[1], oldfsz, newfsz );
    exit( 0 );
}
