//==========================================================================+
//|                                                                         |
//|    Card Game Toolkit(tm)                                                |
//|    Copyright (c) 1996, EverGreenSys Inc. All rights reserved.           |
//|                                                                         |
//|        file:   shuftree.c                                               |
//|        note:   ANSI C (except all int values MUST be 32-bit)            |
//|                                                                         |
//|      author:   D. Taylor (908) 935-1409                                 |
//|                Box 56, Rumson, N.J. 07760                               |
//|                dtaylor@monmouth.com                                     |
//|                                                                         |
//|--------------------------Comments---------------------------------------|
//|      This file contains general functions for building and using        |
//|    binary shuffle trees for card games of any type. The price for       |
//|    generality is somewhat slower performance than that offered by       |
//|    either shufflea.c, shuffleb.c or shufflex.asm (see those files       |
//|    for further details).                                                |
//|      The sample benchmark program shows the component modules and how   |
//|    they should be invoked.                                              |
//|                                                                         |
//|    1 makeDECISIONTREE(nodes);                                           |
//|      Dynamically creates a special balanced binary decision tree        |
//|      suitable as a shuffle tree. The only input is the count of         |
//|      of the number of nodes in the tree. This is found by the           |
//|      following calculation, which is driven simply by the number        |
//|      of cards in the card game of interest (poker=52 for example):      |
//|          actual=1;                                                      |
//|          nodes=1;                                                       |
//|          while( actual<number of cards in game of interest ) {          |
//|             actual=(actual*2); nodes+=actual; }                         |
//|      Call this function first.                                          |
//|                                                                         |
//|    2 dealHANDS();                                                       |
//|      This function uses the existing shuffle tree to deal the number    |
//|      of cards required. As they are dealt, the card index values are    |
//|      put into,                                                          |
//|          aDEAL[]                                                        |
//|                                                                         |
//|    For further details on the definitions of nodes and recursive        |
//|    tree building and traversal, see the following functions that        |
//|    are managed by makeDECISIONTREE(),                                   |
//|                                                                         |
//|    3 growTREE();                                                        |
//|      Recursively builds a complete, balanced binary decision tree.      |
//|      This function could be useful in any application.                  |
//|                                                                         |
//|    4 trimTREE();                                                        |
//|      This function recursively trims off the nodes from a balanced      |
//|      binary decision tree that do not lead to a card outcome, thus      |
//|      converting the binary decision tree into a card shuffle tree.      |
//|                                                                         |
//|    5 showTREE();                                                        |
//|      This function simply prints out the node identifications of the    |
//|      shuffle tree in their correct positions (the tree is tipped on     |
//|      it's side). You can see what your tree looks like.                 |
//|                                                                         |
//|      We tested these functions using the benchmark main program         |
//|    that is also included in this file, by compiling it under each of    |
//|    Borland C/C++ 4.5, Visual C/C++ and Watcom C/C++ 10.5 on a           |
//|    Pentium-S/133. The best performance was in 32-bit mode using         |
//|    the Watcom compiler where it shuffled and dealt an average of        |
//|    40.5 million rounds (that's 405 million hands!) per hour. For        |
//|    this, we set the following switches:                                 |
//|          wcl386 /oneatx /zp4 /5 /fp5 ....                               |
//|      Each card is dealt by traversing the shuffle tree guided by a      |
//|    primitive modulo 2 algorithm that randomly selects each branch.      |
//|    We believe this to be not only a fast, but also a robustly 'random'  |
//|    bit generator.                                                       |
//|-------------------------------------------------------------------------|
//|      This is NOT public domain software. It is fully copyrighted        |
//|    and made available to you to evaluate for 90 days. You may NOT       |
//|    re-distribute the source code or any compiled version of the         |
//|    source code for any purpose without prior permission of the          |
//|    author. If you register this shareware, however, you will receive    |
//|    an extended license for its use and other related materials.         |
//|      See the register.txt file for details.                             |
//+-------------------------------------------------------------------------+
//
#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <time.h>       // used for demonstration program only

typedef struct DNode {  // Define decision node structure with 5 members
  int nodeID;           // (1) i.d. number for a decision node
  int nodeSTATE;        // (2) node state (2, 1 or <1)
  int nodeCARD;         // (3) card i.d. assigned to this node (or NOCARD) 
  struct DNode *down;   // (4) pointer to down-ward descendant
  struct DNode *up;     // (5) pointer to up-ward descendant
  } aDNode;             // creates aDNode as instance of decision node
typedef aDNode *aDNodePTR; // aDNodePTR is a pointer to a decision node
aDNodePTR enterTREE;       // enterTREE will point to 1st node
int nCARDS;             // number of cards needed for a particular game
int nNODES;             // nodes required for a complete binary tree
int IDnumber=1;         // values 1, 2, ... nNODES
int CARDnumber=1;       // values 1, 2, ... nCARDS
int nDECKS;             // number of decks in the game (i.e., poker=1)
int nDEAL;              // number of cards to deal in a complete round
int aDEAL[53];          // the cards dealt in the current round
//
//   -----------------------------------------------------------------
//   This is a simple benchmark program.
//   50,000 rounds of play of Texas Hold'em are simulated.
//   The shuffle tree is initialized with a single deck of 52 cards.
//   There are 10 players, each of whom gets 2 cards dealt face down. 
//   Then 5 community cards are dealt face up, for a total of 25 cards
//   on each round. 
//   -----------------------------------------------------------------
void main() {  
  int j,k;    
  double dbeg,dend,dtpersec,elapsed,perhour;
    makeDECISIONTREE(52); 
    nDEAL=25;
    nDECKS=1;
    dbeg=(double)clock();
    for(k=1;k<=50000;k++) { dealHANDS(); }
    dend=(double)clock();
    dtpersec=(double)CLOCKS_PER_SEC;
    elapsed=(dend-dbeg+1.0)/dtpersec;
    perhour=50000.0*(3600.0/elapsed);
    printf("%.1f per hour\n",perhour);
}  
//+------------------------------------------------------------------------+
//|   dealHANDS()                                                          |
//|------------------------------------------------------------------------|
//|     Deals <nDEAL> cards from a shuffled deck (or shoe).                |
//|     This C function uses an existing shuffle tree to sample            |
//|     without replacement from the deck (or decks) appropriate to        |
//|     the card game of interest (each card can be picked no more         |
//|     than once for each deck in the shoe). Traversal through the        |
//|     binary decision tree is under control of a primitive polynomial    |
//|     modulo 2 algorithm that generates random bits. The card i.d.       |
//|     numbers assigned when the tree was generated are recorded          |
//|     sequentially in array <aDEAL[]> as they are dealt.                 |       
//+------------------------------------------------------------------------+
void dealHANDS() {
  aDNodePTR p,pGO;
  int nth,card,nCASES[53];
  static unsigned int cseed=004125;
    for(nth=0;nth<nDEAL;nth++) nCASES[nth]=0;  // sample w/o replacement
    for(nth=0;nth<nDEAL;nth++) { 
     astart:
       p=enterTREE;                            // enter shuffle tree
     again:  
       if( cseed&010000000000 ) { cseed=((cseed^000000000122)<<1)|01; 
         pGO=p->up; }                          // take UP branch in tree
       else { cseed=cseed<<1; pGO=p->down; }   // take DOWN branch in tree
       if( pGO->nodeSTATE<1 ) { goto astart; } // at dead end node
       card=pGO->nodeCARD;
       if( card<1 ) { p=pGO; goto again; }     // at intermediate node
       if( nCASES[card]>=nDECKS ) goto astart; // card already picked
       nCASES[card]+=1;
       aDEAL[nth]=card;                        // deal this card out
     }
}
//+------------------------------------------------------------------------+
//|   growTREE()                                                           |
//|------------------------------------------------------------------------|
//|     Recursively builds a complete, balanced binary decision tree.      |
//|     Input is number of nodes remaining in tree.                        |
//|     Output is a pointer to the tree's entry point.                     |
//+------------------------------------------------------------------------+
aDNodePTR growTREE(int nodes) {
  int nD,nU,nUD; aDNodePTR p;
    if( nodes==0 ) return NULL; 
    nUD=nodes-1; // subtract this node from count, and assign
    nD=nUD/2;    // half of remaining nodes to the <down> sub-tree,
    nU=nUD-nD;   // the remainder then to the <up> sub-tree
    p=(aDNodePTR)malloc(sizeof(aDNode));  // create this node
    p->down=growTREE(nD);   // build down-ward subtrees 
    p->nodeID=IDnumber;     // assign current node i.d.
    p->nodeSTATE=2;         // default is 2 successors per node
    IDnumber+=1;            // bump node i.d. stamp
    p->up=growTREE(nU);     // build up-ward subtrees
    p->nodeCARD=0;          // default is no card at this node
    if( p->down==NULL && p->up==NULL ) {  // is this a terminal node?
      if( CARDnumber<=nCARDS ) {  // finished assigning card i.d.'s?
        p->nodeCARD=CARDnumber;   // if not, assign the next one
        CARDnumber+=1;            // and bump card i.d. stamp
        }
      }
    return p;               // return pointer to this decision node
}
//+------------------------------------------------------------------------+
//|   trimTREE()                                                           |
//|------------------------------------------------------------------------|
//|     Recursively trims off decision nodes from a balanced binary        |
//|     decision tree that do not lead to a card outcome                   |
//+------------------------------------------------------------------------+
void trimTREE( aDNodePTR p) {
  if( p!=NULL ) {      // 
    trimTREE( p->up );
    if( (p->up)->nodeSTATE<1 && (p->down)->nodeSTATE<1 ) {  
      p->up=NULL;      // when the <up> successor to this node does 
      p->down=NULL;    //   not itself have successors, trim it
      }
    if( p->up==NULL   && p->nodeCARD<1 ) p->nodeSTATE-=1;
    if( p->down==NULL && p->nodeCARD<1 ) p->nodeSTATE-=1;
    trimTREE(p->down);
    if( (p->up)->nodeSTATE<1 && (p->down)->nodeSTATE<1 ) {  
      p->up=NULL;      // when the <down> successor to this node does
      p->down=NULL;    //   not itself have successors, trim it
      }
    if( p->up==NULL   && p->nodeCARD<1 ) p->nodeSTATE-=1;
    if( p->down==NULL && p->nodeCARD<1 ) p->nodeSTATE-=1;
    }
}
//+------------------------------------------------------------------------+
//|   showTREE()                                                           |
//|------------------------------------------------------------------------|
//|     Recursively prints out the node i.d.'s in their correct            |
//|     position for the current tree.                                     |
//+------------------------------------------------------------------------+
void showTREE( aDNodePTR p,int tab) {
  if( p!=NULL ) {
    showTREE( p->up,tab+4 );
    if( p->nodeSTATE>0 ) printf("%*d \n",tab,p->nodeID);
    showTREE(p->down,tab+4);
    }
}
//+------------------------------------------------------------------------+
//|   makeDECISIONTREE()                                                   |
//|------------------------------------------------------------------------|
//|     This function controls the build of a decision tree that           |
//|     can be the basis for a shuffle tree. The only information          |
//|     needed is the number of cards in the game of interest,             |
//|     which is provided as an input parameter.                           |         
//+------------------------------------------------------------------------+
void makeDECISIONTREE(int cards) {
  int actual;    
    nCARDS=cards; nNODES=1; actual=1;
    while( actual<nCARDS ) { actual=(actual*2); nNODES+=actual; }
    enterTREE=growTREE(nNODES);
    trimTREE(enterTREE);
    printf("Shuffle Tree for %d deck(s):\n",nDECKS);
    showTREE(enterTREE,4);     // optional
}
