//
// Written by: Robert C. Pendleton
// 
// This code is derived from code I wrote
// myself and from several public domain
// sources. 
//
// Placed in the public domain by the author.
//

#include <dos.h>
#include <conio.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "tg.h"

#include "ptypes.h"

#define max(a,b) (((a) > (b)) ? (a) : (b))
#define min(a,b) (((a) < (b)) ? (a) : (b))
#define abs(a) (((a)<0) ? -(a) : (a))
#define sign(a) (((a)<0) ? -1 : (a)>0 ? 1 : 0)

#define vgapage ((uint8 *) 0xa0000)

#define MOR_ADDR  (0x3c2)           // misc. output register
#define MOR_READ  (0x3cc)           // misc. output register

#define IST1_ADDR (0x3da)           // input status #1 register

#define SEQ_ADDR  (0x3c4)           // base port of the Sequencer
#define SEQ_DATA  (0x3c5)           // data port of the Sequencer

#define CRTC_ADDR (0x3d4)           // base port of the CRT Controller (color)
#define CRTC_DATA (0x3d5)           // data port of the CRT Controller (color)

#define GCR_ADDR  (0x3ce)           // graphics controller address register
#define GCR_DATA  (0x3cf)           // graphics controller data register

#define ACR_ADDR  (0x3c0)           // attribute registers
#define ACR_DATA  (0x3c1)           // attribute registers

#define PAL_WRITE_ADDR (0x3c8)      // palette write address
#define PAL_READ_ADDR  (0x3c7)      // palette write address
#define PAL_DATA       (0x3c9)      // palette data register

//---------------------------------------------------
//
// low level long word fill routine
//

void
fillLong(void *addr, uint32 value, int32 count);
#pragma aux fillLong = \
    "cld" \
    "repe stosd" \
    parm [edi] [eax] [ecx];

//---------------------------------------------------
//
// Use the bios to make sure we have a VGA display
//
// Not really sure this works, but I have two books
// that say it does... 
//

boolean
isVga()
{
    union REGS rg;

    rg.w.ax = 0x1a00;       // if this function is supported
                            // then we have a vga bios
    int386(0x10, &rg, &rg);

    return rg.h.al == 0x1a;
}

//---------------------------------------------------
//
// Use the bios to get the address of the 8x8 font
//
// You need a font if you are going to draw text.
//

uint8 far *
getFont()
{
    union REGPACK rg;
    uint32 seg;
    uint32 off;

    memset(&rg, 0, sizeof(rg));
    rg.w.ax = 0x1130;
    rg.h.bh = 0x03;
    intr(0x10, &rg);
    seg = rg.w.es;
    off = rg.w.bp;
    

    return (uint8 far *)MK_FP(seg, off);
}

//---------------------------------------------------
//
// Use the bios to get the current video mode
//

long
getMode()
{
    union REGS rg;

    rg.h.ah = 0x0f;
    int386(0x10, &rg, &rg);

    return rg.h.al;
}

//---------------------------------------------------
//
// Use the bios to set the video mode
//

void
setMode(long mode)
{
    union REGS rg;

    rg.h.ah = 0x00;
    rg.h.al = mode;
    int386(0x10, &rg, &rg);
}

//---------------------------------------------------
//
// Set a 160x120 unchained 256 color video mode
//

void
tg::mode160x120()
{
    int crtc11;

    outp(0x3d4, 0x11); // unlock crtc
    crtc11 = inp(0x3d5) & 0x7f;
    outp(0x3d4, 0x11);
    outp(0x3d5, crtc11);

    width   = 160;
    height  = 120;
    maxx    = 159;
    maxy    = 119;
    pages   = 13;
    lineSize = 40;
    pageSize = 19200;
    modeName = "160x120";

    outp(0x3c2, 0xe3);   // mor

    outp(0x3d4, 0x00); // crtc
    outp(0x3d5, 0x32);

    outp(0x3d4, 0x01); // crtc
    outp(0x3d5, 0x27);

    outp(0x3d4, 0x02); // crtc
    outp(0x3d5, 0x28);

    outp(0x3d4, 0x03); // crtc
    outp(0x3d5, 0x20);

    outp(0x3d4, 0x04); // crtc
    outp(0x3d5, 0x2b);

    outp(0x3d4, 0x05); // crtc
    outp(0x3d5, 0x70);

    outp(0x3d4, 0x06); // crtc
    outp(0x3d5, 0x0d);

    outp(0x3d4, 0x07); // crtc
    outp(0x3d5, 0x3e);

    outp(0x3d4, 0x08); // crtc
    outp(0x3d5, 0x00);

    outp(0x3d4, 0x09); // crtc
    outp(0x3d5, 0x43);

    outp(0x3d4, 0x10); // crtc
    outp(0x3d5, 0xea);

    outp(0x3d4, 0x11); // crtc
    outp(0x3d5, 0xac);

    outp(0x3d4, 0x12); // crtc
    outp(0x3d5, 0xdf);

    outp(0x3d4, 0x13); // crtc
    outp(0x3d5, 0x14);

    outp(0x3d4, 0x14); // crtc
    outp(0x3d5, 0x00);

    outp(0x3d4, 0x15); // crtc
    outp(0x3d5, 0xe7);

    outp(0x3d4, 0x16); // crtc
    outp(0x3d5, 0x06);

    outp(0x3d4, 0x17); // crtc
    outp(0x3d5, 0xe3);

    outp(0x3c4, 0x01); // seq
    outp(0x3c5, 0x01);

    outp(0x3c4, 0x03); // seq
    outp(0x3c5, 0x00);

    outp(0x3c4, 0x04); // seq
    outp(0x3c5, 0x06);

    outp(0x3ce, 0x05); // gcr
    outp(0x3cf, 0x40);

    outp(0x3ce, 0x06); // gcr
    outp(0x3cf, 0x05);

    inp(0x3da);          // acr
    outp(0x3c0, 0x10 | 0x20);
    outp(0x3c0, 0x41);

    inp(0x3da);          // acr
    outp(0x3c0, 0x11 | 0x20);
    outp(0x3c0, 0x00);

    inp(0x3da);          // acr
    outp(0x3c0, 0x12 | 0x20);
    outp(0x3c0, 0x0f);

    inp(0x3da);          // acr
    outp(0x3c0, 0x13 | 0x20);
    outp(0x3c0, 0x00);

    inp(0x3da);          // acr
    outp(0x3c0, 0x14 | 0x20);
    outp(0x3c0, 0x00);

    outp(0x3d4, 0x11); // lock crtc
    crtc11 = inp(0x3d5) | 0x80;
    outp(0x3d4, 0x11);
    outp(0x3d5, crtc11);
}

//---------------------------------------------------
//
// Set a 296x220 unchained 256 color mode
//

void
tg::mode296x220()
{
    int crtc11;

    outp(0x3d4, 0x11); // unlock crtc
    crtc11 = inp(0x3d5) & 0x7f;
    outp(0x3d4, 0x11);
    outp(0x3d5, crtc11);

    width   = 296;
    height  = 220;
    maxx    = 295;
    maxy    = 219;
    pages   = 4;
    lineSize = 74;
    pageSize = 65120;
    modeName = "296x220";

    outp(0x3c2, 0xe3);   // mor

    outp(0x3d4, 0x00); // crtc
    outp(0x3d5, 0x5f);

    outp(0x3d4, 0x01); // crtc
    outp(0x3d5, 0x49);

    outp(0x3d4, 0x02); // crtc
    outp(0x3d5, 0x50);

    outp(0x3d4, 0x03); // crtc
    outp(0x3d5, 0x82);

    outp(0x3d4, 0x04); // crtc
    outp(0x3d5, 0x53);

    outp(0x3d4, 0x05); // crtc
    outp(0x3d5, 0x80);

    outp(0x3d4, 0x06); // crtc
    outp(0x3d5, 0x0d);

    outp(0x3d4, 0x07); // crtc
    outp(0x3d5, 0x3e);

    outp(0x3d4, 0x08); // crtc
    outp(0x3d5, 0x00);

    outp(0x3d4, 0x09); // crtc
    outp(0x3d5, 0x41);

    outp(0x3d4, 0x10); // crtc
    outp(0x3d5, 0xd7);

    outp(0x3d4, 0x11); // crtc
    outp(0x3d5, 0xac);

    outp(0x3d4, 0x12); // crtc
    outp(0x3d5, 0xb7);

    outp(0x3d4, 0x13); // crtc
    outp(0x3d5, 0x25);

    outp(0x3d4, 0x14); // crtc
    outp(0x3d5, 0x00);

    outp(0x3d4, 0x15); // crtc
    outp(0x3d5, 0xe7);

    outp(0x3d4, 0x16); // crtc
    outp(0x3d5, 0x06);

    outp(0x3d4, 0x17); // crtc
    outp(0x3d5, 0xe3);

    outp(0x3c4, 0x01); // seq
    outp(0x3c5, 0x01);

    outp(0x3c4, 0x04); // seq
    outp(0x3c5, 0x06);

    outp(0x3ce, 0x05); // gcr
    outp(0x3cf, 0x40);

    outp(0x3ce, 0x06); // gcr
    outp(0x3cf, 0x05);

    inp(0x3da);          // acr
    outp(0x3c0, 0x10 | 0x20);
    outp(0x3c0, 0x41);

    inp(0x3da);          // acr
    outp(0x3c0, 0x13 | 0x20);
    outp(0x3c0, 0x00);

    outp(0x3d4, 0x11); // lock crtc
    crtc11 = inp(0x3d5) | 0x80;
    outp(0x3d4, 0x11);
    outp(0x3d5, crtc11);
}

//---------------------------------------------------
//
// Set an unchained version of mode 0x13
//

void
tg::mode320x200()
{
    setMode(0x13);

    width   = 320;
    height  = 200;
    maxx    = 319;
    maxy    = 199;
    pages   = 4;
    lineSize = 80;
    pageSize = 64000;
    modeName = "320x200";

    /*

    Turn on the "mode X" effect by turning off chain 4,
    word, and double word mode. 

    */
    //
    // Turn off the Chain-4 bit in SEQ #4
    //
    outp(SEQ_ADDR, 0x04);
    outp(SEQ_DATA, 0x06);

    //
    // Turn off word mode, by setting the Mode Control register
    // of the CRTC #17
    //
    outp(CRTC_ADDR, 0x17);
    outp(CRTC_DATA, 0xE3);

    //
    // Turn off doubleword mode, by setting CRTC #14
    //
    outp(CRTC_ADDR, 0x14);
    outp(CRTC_DATA, 0x00);
}

//---------------------------------------------------
//
// This is the famous mode-x used in so many video
// games. It is a 320x240 unchained 256 color mode.
//

void
tg::mode320x240()
{
    int crtc11;

    outp(0x3d4, 0x11); // unlock crtc
    crtc11 = inp(0x3d5) & 0x7f;
    outp(0x3d4, 0x11);
    outp(0x3d5, crtc11);

    width   = 320;
    height  = 240;
    maxx    = 319;
    maxy    = 239;
    pages   = 3;
    lineSize = 80;
    pageSize = 76800;
    modeName = "320x240";

    outp(0x3c2, 0xe3);   // mor

    outp(0x3d4, 0x00); // crtc
    outp(0x3d5, 0x5f);

    outp(0x3d4, 0x01); // crtc
    outp(0x3d5, 0x4f);

    outp(0x3d4, 0x02); // crtc
    outp(0x3d5, 0x50);

    outp(0x3d4, 0x03); // crtc
    outp(0x3d5, 0x82);

    outp(0x3d4, 0x04); // crtc
    outp(0x3d5, 0x54);

    outp(0x3d4, 0x05); // crtc
    outp(0x3d5, 0x80);

    outp(0x3d4, 0x06); // crtc
    outp(0x3d5, 0x0d);

    outp(0x3d4, 0x07); // crtc
    outp(0x3d5, 0x3e);

    outp(0x3d4, 0x08); // crtc
    outp(0x3d5, 0x00);

    outp(0x3d4, 0x09); // crtc
    outp(0x3d5, 0x41);

    outp(0x3d4, 0x10); // crtc
    outp(0x3d5, 0xea);

    outp(0x3d4, 0x11); // crtc
    outp(0x3d5, 0xac);

    outp(0x3d4, 0x12); // crtc
    outp(0x3d5, 0xdf);

    outp(0x3d4, 0x13); // crtc
    outp(0x3d5, 0x28);

    outp(0x3d4, 0x14); // crtc
    outp(0x3d5, 0x00);

    outp(0x3d4, 0x15); // crtc
    outp(0x3d5, 0xe7);

    outp(0x3d4, 0x16); // crtc
    outp(0x3d5, 0x06);

    outp(0x3d4, 0x17); // crtc
    outp(0x3d5, 0xe3);

    outp(0x3c4, 0x01); // seq
    outp(0x3c5, 0x01);

    outp(0x3c4, 0x04); // seq
    outp(0x3c5, 0x06);

    outp(0x3ce, 0x05); // gcr
    outp(0x3cf, 0x40);

    outp(0x3ce, 0x06); // gcr
    outp(0x3cf, 0x05);

    inp(0x3da);          // acr
    outp(0x3c0, 0x10 | 0x20);
    outp(0x3c0, 0x41);

    inp(0x3da);          // acr
    outp(0x3c0, 0x13 | 0x20);
    outp(0x3c0, 0x00);

    outp(0x3d4, 0x11); // lock crtc
    crtc11 = inp(0x3d5) | 0x80;
    outp(0x3d4, 0x11);
    outp(0x3d5, crtc11);
}

//---------------------------------------------------
//
// This is a 320x400 unchaned 256 color video mode.
//

void
tg::mode320x400()
{
    width   = 320;
    height  = 400;
    maxx    = 319;
    maxy    = 399;
    pages   = 2;
    lineSize = 80;
    pageSize = 128000;
    modeName = "320x400";

    /*

    Turn on the "mode X" effect by turning off chain 4,
    word, and double word mode. 

    */
    //
    // Turn off the Chain-4 bit in SEQ #4
    //
    outp(SEQ_ADDR, 0x04);
    outp(SEQ_DATA, 0x06);

    //
    // Turn off word mode, by setting the Mode Control register
    // of the CRTC #17
    //
    outp(CRTC_ADDR, 0x17);
    outp(CRTC_DATA, 0xE3);

    //
    // Turn off doubleword mode, by setting CRTC #14
    //
    outp(CRTC_ADDR, 0x14);
    outp(CRTC_DATA, 0x00);

    //
    // Set MSL to 0 in the maximum scan line register.
    // This action turns 320x200 mode 13 into 320x400 mode 13+
    //
    outp(CRTC_ADDR, 0x09);
    outp(CRTC_DATA, 0x40);

}

//---------------------------------------------------
//
// This is a 360x360 unchained 256 color video mode.
//

void
tg::mode360x360()
{
    int crtc11;

    outp(0x3d4, 0x11); // unlock crtc
    crtc11 = inp(0x3d5) & 0x7f;
    outp(0x3d4, 0x11);
    outp(0x3d5, crtc11);

    width   = 360;
    height  = 360;
    maxx    = 359;
    maxy    = 359;
    pages   = 2;
    lineSize = 90;
    pageSize = 129600;
    modeName = "360x360";

    outp(0x3c2, 0x67);   // mor

    outp(0x3d4, 0x00); // crtc
    outp(0x3d5, 0x6b);

    outp(0x3d4, 0x01); // crtc
    outp(0x3d5, 0x59);

    outp(0x3d4, 0x02); // crtc
    outp(0x3d5, 0x5a);

    outp(0x3d4, 0x03); // crtc
    outp(0x3d5, 0x8e);

    outp(0x3d4, 0x04); // crtc
    outp(0x3d5, 0x5e);

    outp(0x3d4, 0x05); // crtc
    outp(0x3d5, 0x8a);

    outp(0x3d4, 0x06); // crtc
    outp(0x3d5, 0xbf);

    outp(0x3d4, 0x07); // crtc
    outp(0x3d5, 0x1f);

    outp(0x3d4, 0x08); // crtc
    outp(0x3d5, 0x00);

    outp(0x3d4, 0x09); // crtc
    outp(0x3d5, 0x40);

    outp(0x3d4, 0x10); // crtc
    outp(0x3d5, 0x88);

    outp(0x3d4, 0x11); // crtc
    outp(0x3d5, 0x85);

    outp(0x3d4, 0x12); // crtc
    outp(0x3d5, 0x67);

    outp(0x3d4, 0x13); // crtc
    outp(0x3d5, 0x2d);

    outp(0x3d4, 0x14); // crtc
    outp(0x3d5, 0x00);

    outp(0x3d4, 0x15); // crtc
    outp(0x3d5, 0x6d);

    outp(0x3d4, 0x16); // crtc
    outp(0x3d5, 0xba);

    outp(0x3d4, 0x17); // crtc
    outp(0x3d5, 0xe3);

    outp(0x3c4, 0x01); // seq
    outp(0x3c5, 0x01);

    outp(0x3c4, 0x04); // seq
    outp(0x3c5, 0x06);

    outp(0x3ce, 0x05); // gcr
    outp(0x3cf, 0x40);

    outp(0x3ce, 0x06); // gcr
    outp(0x3cf, 0x05);

    inp(0x3da);          // acr
    outp(0x3c0, 0x10 | 0x20);
    outp(0x3c0, 0x41);

    inp(0x3da);          // acr
    outp(0x3c0, 0x13 | 0x20);
    outp(0x3c0, 0x00);

    outp(0x3d4, 0x11); // lock crtc
    crtc11 = inp(0x3d5) | 0x80;
    outp(0x3d4, 0x11);
    outp(0x3d5, crtc11);
}

//---------------------------------------------------
//
// This sets up a 400x300 unchained 256 color video mode
//

void
tg::mode400x300()
{
    int crtc11;

    outp(0x3d4, 0x11); // unlock crtc
    crtc11 = inp(0x3d5) & 0x7f;
    outp(0x3d4, 0x11);
    outp(0x3d5, crtc11);

    width   = 400;
    height  = 300;
    maxx    = 399;
    maxy    = 299;
    pages   = 2;
    lineSize = 100;
    pageSize = 120000;
    modeName = "400x300";

//    outp(0x3c2, 0xa7);   // mor
    outp(0x3c2, 0xe7);   // mor

    outp(0x3d4, 0x00); // crtc
    outp(0x3d5, 0x71);

    outp(0x3d4, 0x01); // crtc
    outp(0x3d5, 0x63);

    outp(0x3d4, 0x02); // crtc
    outp(0x3d5, 0x64);

    outp(0x3d4, 0x03); // crtc
    outp(0x3d5, 0x92);

    outp(0x3d4, 0x04); // crtc
//    outp(0x3d5, 0x65);
    outp(0x3d5, 0x67);

    outp(0x3d4, 0x05); // crtc
    outp(0x3d5, 0x82);

    outp(0x3d4, 0x06); // crtc
    outp(0x3d5, 0x46);

    outp(0x3d4, 0x07); // crtc
    outp(0x3d5, 0x1f);

    outp(0x3d4, 0x08); // crtc
    outp(0x3d5, 0x00);

    outp(0x3d4, 0x09); // crtc
    outp(0x3d5, 0x40);

    outp(0x3d4, 0x10); // crtc
    outp(0x3d5, 0x31);

    outp(0x3d4, 0x11); // crtc
    outp(0x3d5, 0x80);

    outp(0x3d4, 0x12); // crtc
    outp(0x3d5, 0x2b);

    outp(0x3d4, 0x13); // crtc
    outp(0x3d5, 0x32);

    outp(0x3d4, 0x14); // crtc
    outp(0x3d5, 0x00);

    outp(0x3d4, 0x15); // crtc
    outp(0x3d5, 0x2f);

    outp(0x3d4, 0x16); // crtc
    outp(0x3d5, 0x44);

    outp(0x3d4, 0x17); // crtc
    outp(0x3d5, 0xe3);

    outp(0x3c4, 0x01); // seq
    outp(0x3c5, 0x01);

    outp(0x3c4, 0x02); // seq
    outp(0x3c5, 0x0f);

    outp(0x3c4, 0x04); // seq
    outp(0x3c5, 0x06);

    outp(0x3ce, 0x05); // gcr
    outp(0x3cf, 0x40);

    outp(0x3ce, 0x06); // gcr
    outp(0x3cf, 0x05);

    inp(0x3da);          // acr
    outp(0x3c0, 0x10 | 0x20);
    outp(0x3c0, 0x41);

    inp(0x3da);          // acr
    outp(0x3c0, 0x13 | 0x20);
    outp(0x3c0, 0x00);

    outp(0x3d4, 0x11); // lock crtc
    crtc11 = inp(0x3d5) | 0x80;
    outp(0x3d4, 0x11);
    outp(0x3d5, crtc11);
}

//---------------------------------------------------
//
// Set up one of the 7 tweaked modes.
//
// Start out by setting mode 0x13 to make sure that
// the palette and fonts are set up correctly for
// 256 color modes.
//

void
tg::setTweakMode(int16 mode)
{
    setMode(0x13);

    switch (mode)
    {
    case t160x120:
        mode160x120();
        break;

    case t296x220:
        mode296x220();
        break;

    case t320x200:
        mode320x200();
        break;

    case t320x240:
        mode320x240();
        break;

    case t320x400:
        mode320x400();
        break;

    case t360x360:
        mode360x360();
        break;

    case t400x300:
        mode400x300();
        break;

    default:
        error = modeNotSupported;
    }
}

//---------------------------------------------------
//
// Remember the mode we were in before we started.
//

tg::tg()
{
    error = none;

    origMode = getMode();                   // save the current mode
    if (!isVga())
    {
        error = notVgaDisplay;
        return;
    }
}

//---------------------------------------------------
//
// Put things back the way they were before we tried
// to set up a new video mode.
//

tg::~tg()
{
    setMode(origMode);
}

//---------------------------------------------------
//
// Put the graphcs system into the requested mode and
// clear 256k of video memory to pixel value 0.
//

void
tg::mode(int16 mode)
{
    modeName = NULL;
    fontAddr = getFont();
    setTweakMode(mode);                 // set the new mode
    if (error != none)
    {
        return;
    }
    outp(SEQ_ADDR, 0x02);               // enable all planes
    outp(SEQ_DATA, 0x0F);               // enable all planes
    fillLong(vgapage, 0, 16 * 1024);    // clear the whole page

    activeOffset = vgapage;
}

//---------------------------------------------------
//
//

void
tg::resetMode()
{
    setMode(origMode);

    width = 0;
    height = 0;
    maxx = 0;
    maxy = 0;
    pages = 0;
    lineSize = 0;
    pageSize = 0;
    modeName = NULL;
}

//---------------------------------------------------
//
// Clear the current page to the given pixel value.
//

void
tg::clear(uint8 color)
{
    int32 c = 0;

    c = (color & 0xff);
    c = (c << 8) | (color & 0xff);
    c = (c << 8) | (color & 0xff);
    c = (c << 8) | (color & 0xff);

    outp(SEQ_ADDR, 0x02);     // enable all planes
    outp(SEQ_DATA, 0x0F);     // enable all planes
    fillLong(activeOffset, c, pageSize / 16);
}

//---------------------------------------------------
//
// Make all drawing go the the selected page.
//

void
tg::setActivePage(int16 page)
{
    if (page < 0 || page >= pages)
    {
        return;
    }
    activeOffset = vgapage + (page * pageSize / 4);
}

//---------------------------------------------------
//
// Make the selected page visible.
//

void
tg::setVisiblePage(int16 page)
{
    int32 offset;

    if (page < 0 || page >= pages)
    {
        return;
    }

    offset = page * pageSize / 4;
    
    while(inp(IST1_ADDR) & 0x01);       // wait for display disable

    outp(CRTC_ADDR, 0x0c);
    outp(CRTC_DATA, (offset & 0xff00) >> 8);

    outp(CRTC_ADDR, 0x0d);
    outp(CRTC_DATA, offset &0x00ff);

    //
    // you can spend a lot of CPU time in this little loop
    //
    while(!(inp(IST1_ADDR) & 0x08));    // wait for vertical retrace
}

//---------------------------------------------------
//
// Draw a point in the current page with the given
// pixel value.
//

void
tg::pixel(int16 x, int16 y, uint8 color)
{
    //
    // first clip to the mode width and height
    //
    if (x < 0 || x > maxx || y < 0 || y > maxy)
    {
        return;
    }

    //
    // set the mask so that only one pixel gets written
    //
    outp(SEQ_ADDR, 0x02);
    outp(SEQ_DATA, 1 << (x & 0x3));

    *(activeOffset + (lineSize * y) + (x >> 2)) = color;
}

//---------------------------------------------------
//
// Read the value of a pixel.
//

int16
tg::pixel(int16 x, int16 y)
{
    //
    // first clip to the mode width and height
    //
    if (x < 0 || x > maxx || y < 0 || y > maxy)
    {
        return -1;
    }

    //
    // set the mask so that only one pixel gets read
    //
    outp(GCR_ADDR, 0x04);
    outp(GCR_DATA, x & 0x3);

    return *(activeOffset + (lineSize * y) + (x >> 2));
}

//---------------------------------------------------
//
// Please note that lines are NOT clipped.
//

/*
 * Derived from:
 *
 * Digital Line Drawing
 * by Paul Heckbert
 * from "Graphics Gems", Academic Press, 1990
 */

void
tg::line(int16 x1, int16 y1, int16 x2, int16 y2, uint8 color)
{
    int16 d;
    int16 x;
    int16 y;
    int16 ax;
    int16 ay;
    int16 sx;
    int16 sy;
    int16 dx;
    int16 dy;

    uint8 *lineAddr;
    int16 yOffset;

    dx = x2 - x1;  
    ax = abs(dx) << 1;  
    sx = sign(dx);

    dy = y2 - y1;  
    ay = abs(dy) << 1;  
    sy = sign(dy);
    yOffset = sy * lineSize;

    x = x1;
    y = y1;

    lineAddr = (activeOffset + (lineSize * y));
    if (ax>ay)
    {                       /* x dominant */
        d = ay - (ax >> 1);
        for (;;)
        {
            outp(SEQ_ADDR, 0x02);
            outp(SEQ_DATA, 1 << (x & 0x3));
            *((lineAddr) + (x >> 2)) = color;

            if (x == x2)
            {
                return;
            }
            if (d>=0)
            {
                y += sy;
                lineAddr += yOffset;
                d -= ax;
            }
            x += sx;
            d += ay;
        }
    }
    else
    {                       /* y dominant */
        d = ax - (ay >> 1);
        for (;;)
        {
            outp(SEQ_ADDR, 0x02);
            outp(SEQ_DATA, 1 << (x & 0x3));
            *((lineAddr) + (x >> 2)) = color;

            if (y == y2)
            {
                return;
            }
            if (d>=0) 
            {
                x += sx;
                d -= ay;
            }
            y += sy;
            lineAddr += yOffset;
            d += ax;
        }
    }
}

//---------------------------------------------------
//
// Draw a character in the active page using the
// 8x8 character font.
//

void
tg::drawChar(int16 x, int16 y, uint8 color, char c)
{
    uint32 i, j;
    uint8 mask;
    uint8 far *font = fontAddr + (c * 8);

    for (i = 0; i < 8; i++)
    {
        mask = *font;
        for (j = 0; j < 8; j++)
        {
            if (mask & 0x80)
            {
                pixel(x + j, y + i, color);
            }
            mask <<= 1;
        }
        font++;
    }
}

//---------------------------------------------------
//
// Draw a text string on the screen.
//

void
tg::drawText(int16 x, int16 y, uint8 color, char *string)
{
    while (*string)
    {
        drawChar(x, y, color, *string);
        x += 8;
        string++;
    }
}

//---------------------------------------------------
//
// Fill a rectangle on the screen. 
//

int32 middleMask[4][4] = 
{
    {0x1, 0x3, 0x7, 0xf},
    {0x0, 0x2, 0x6, 0xe},
    {0x0, 0x0, 0x4, 0xc},
    {0x0, 0x0, 0x0, 0x8},
};

int32 leftMask[4] = {0xf, 0xe, 0xc, 0x8};

int32 rightMask[4] = {0x1, 0x3, 0x7, 0xf};

void
tg::fillRect(int16 x, int16 y, int16 width, int16 height, uint8 color)
{
    int32 i;

    int32 x1;
    int32 x2;
    int32 y1;
    int32 y2;

    int32 leftBand;
    int32 rightBand;
    int32 leftBit;
    int32 rightBit;
    int32 mask;
    int32 bands;

    char *top;
    char *where;

    x1 = x;
    x2 = x + width - 1;
    y1 = y;
    y2 = y + height - 1;

    //
    // first off, clip to the screen.
    //
    if (x1 < 0)
    {
        x1 = 0;
    }

    if (x2 > maxx)
    {
        x2 = maxx;
    }

    if (y1 < 0)
    {
        y1 = 0;
    }

    if (y2 > maxy)
    {
        y2 = maxy;
    }

    //
    // is there a rectangle left?
    //
    if (y2 < y1 || x2 < x1)
    {
        return;
    }

    //
    // in mode X we want to paint from the top down
    // and make use of the 4 pixel wide vertical bands
    //

    leftBand = x1 >> 2; 
    rightBand = x2 >> 2;

    leftBit = x1 & 3; 
    rightBit = x2 & 3;

    if (leftBand == rightBand)  // the whole rectangle is in one band
    {
        mask = middleMask[leftBit][rightBit];
        outp(SEQ_ADDR, 0x02);
        outp(SEQ_DATA, mask);

        top = activeOffset + (lineSize * y1) + leftBand;
        for (i = y1; i <= y2; i++)
        {
            *top = color;
            top += lineSize;
        }
    }
    else                        // spans 2 or more bands
    {
        mask = leftMask[leftBit];
        outp(SEQ_ADDR, 0x02);
        outp(SEQ_DATA, mask);

        top = activeOffset + (lineSize * y1) + leftBand;
        where = top;
        for (i = y1; i <= y2; i++)          // fill the left edge
        {
            *where = color;
            where += lineSize;
        }
        top++;

        outp(SEQ_ADDR, 0x02);
        outp(SEQ_DATA, 0x0f);

        bands = rightBand - (leftBand + 1);
        if (bands > 0)
        {
            where = top;
            for (i = y1; i <= y2; i++)      // fill the middle
            {
                memset(where, color, bands);
                where += lineSize;
            }
            top += bands;
        }

        mask = rightMask[rightBit];

        outp(SEQ_ADDR, 0x02);
        outp(SEQ_DATA, mask);

        where = top;
        for (i = y1; i <= y2; i++)          // fill the right edge
        {
            *where = color;
            where += lineSize;
        }

    }
}

//---------------------------------------------------
//
// Set one entry in the color palatte.
//

void
tg::setColor(int16 index, uint16 r, uint16 g, uint16 b)
{
    if (index < 0 || index > 255)
    {
        return;
    }

    while(!(inp(0x3da) & 0x01));    // wait for blanking

    outp(PAL_WRITE_ADDR, index);
    outp(PAL_DATA, r);
    outp(PAL_DATA, g);
    outp(PAL_DATA, b);
}

//---------------------------------------------------
//
// Set a range of entries in the color palette.
//

void
tg::setPalette(int16 start, int16 count, color *p)
{
    int16 i;

    if (start < 0 || (start + count - 1) > 255)
    {
        return;
    }

    while(!(inp(0x3da) & 0x08));    // wait vertical retrace

    outp(PAL_WRITE_ADDR, start);
    for (i = 0; i < count; i++)
    {
        outp(PAL_DATA, p->red);
        outp(PAL_DATA, p->green);
        outp(PAL_DATA, p->blue);
        p++;
    }
}

//---------------------------------------------------
