//
// Written by: Robert C. Pendleton
// 
// Copyright 1993 by Robert C. Pendleton
//
// Non-commercial use by individuals
// is permitted.
//

#include <dos.h>
#include <conio.h>
#include <memory.h>
#include <stdio.h>
#include <stdlib.h>
#include "vbe.h"
#include "vg.h"

#define FORCE_PTYPES
#include "ptypes.h"
#undef FORCE_PTYPES

#define max(a,b) (((a) > (b)) ? (a) : (b))
#define min(a,b) (((a) < (b)) ? (a) : (b))

//---------------------------------------------------
//
// Try to put the system into the given video mode.
//
// Refuse if we don't know how to do graphics in
// that mode or if the mode isn't supported
//

vg::vg(int16 mode)
{
    int32 temp;

    error = none;

    //
    // get info about the vbe
    //

    if (!vbeGetInfo(&info))
    {
        error = notVbe12;
        return;
    }

    //
    // is it a valid VESA
    // info block?
    //

    if (info.vesa[0] != 'V' ||
        info.vesa[1] != 'E' ||
        info.vesa[2] != 'S' ||
        info.vesa[3] != 'A')
    {
        error = notVbe12;
        return;
    }

    //
    // is it version 1.2 or
    // greater?
    //

    if (info.majorMode == 1 &&
        info.minorMode < 2)
    {
        error = notVbe12;
        return;
    }

    //
    // is the requested mode supported
    // on this machine?
    //

    currentMode = mode;

    if (!vbeGetModeInfo(currentMode, &modeInfo) ||
        !(modeInfo.modeAttr & 1))
    {
        error = modeNotSupported;
        return;
    }

    //
    // is it an 8 bit, 256 color mode?
    //

    if (modeInfo.bitsPerPixel != 8)
    {
        error = not8bits;
        return;
    }

    maxx = modeInfo.width - 1;
    maxy = modeInfo.height - 1;

    width = modeInfo.width;
    height = modeInfo.height;

    currentBank = -1;

    activePage = 0;
    activePageOffset = 0;
    visiblePage = 0;
 
    temp = (int32)modeInfo.width * (int32)modeInfo.height;
    temp = (temp + 0xffffL) >> 16;
    banksPerPage = temp;

    //
    // everything was ok
    // so save the current mode
    // and set the new mode
    //
    vbeGetMode(&origMode);      // save the current mode
    vbeSetMode(currentMode);    // set the new mode
}

//---------------------------------------------------
//
// Put things back the way they were before we tried
// to set up a new video mode
//

vg::~vg()
{
    vbeSetMode(origMode);
}

//---------------------------------------------------
//
// clear the current page to the given pixel value
//

void
vg::clear(int16 color)
{
    int16 i;

    for (i = 0; i < banksPerPage; i++)
    {
        currentBank = i + activePageOffset;
        vbeSetBankAAddr(i + activePageOffset);
        memset(MK_FP(0xa000, 0), color, 0x8000);
        memset(MK_FP(0xa000, 0x8000), color, 0x8000);
    }
}

//---------------------------------------------------
//
// make all drawing go the the selected page
//

void
vg::setActivePage(int16 page)
{
    if (page < 0 || page >= getNumPages())
    {
        error = badPage;
        return;
    }

    activePage = page;
    activePageOffset = page * banksPerPage;
}

//---------------------------------------------------
//
// display the selected page.
//

void
vg::setVisiblePage(int16 page)
{
    int16 pixel, line;
    int32 address;

    if (page < 0 || page >= getNumPages())
    {
        error = badPage;
        return;
    }
    visiblePage = page;

    address = ((int32)banksPerPage * (int32)page) << 16;
    line = address / (int32)modeInfo.width;
    pixel = address % (int32)modeInfo.width;

    while(inport(0x3da) & 0x09);    // wait for blanking to end

    vbeSetDisplayStart(pixel, line);

    while(!(inp(0x3da) & 0x08));    // wait for vertical retrace
}

//---------------------------------------------------
//
// draw a point in the current page with the given
// pixel value
//

void
vg::pixel(int16 x, int16 y, int16 color)
{
    int32 offset;
    int16 bank;
    uint8 *address;

    //
    // first clip to the mode width and height
    //
    if (x < 0 || x > maxx || y < 0 || y > maxy)
    {
        return;
    }

    //
    // compute the pixel address and then convert
    // it into a bank number and an offset.
    // finally generate a pointer into the
    // video page
    //
    offset = ((int32)y * (int32)modeInfo.width) + (int32)x;
    bank = (offset >> 16) + activePageOffset;
    address = (uint8 *) MK_FP(0xa000, (int)(offset & 0xffffL));

    //
    // select the video page if we need to
    //
    if (bank != currentBank)
    {
        currentBank = bank;
        vbeSetBankAAddr(currentBank);
    }

    //
    // write the pixel
    //
    *address = (uint8)color;
}

//---------------------------------------------------
//
//
//
//

void
vg::fillRect(int16 x, int16 y, int16 width, int16 height, int16 color)
{
    int16 i;
    int16 lines;

    int16 x1 = x;
    int16 x2 = x + width - 1;
    int16 y1 = y;
    int16 y2 = y + height - 1;

    int32 offset;
    int32 startBank;

    int32 endOffset;
    int32 endBank;

    uint8 *address;
    uint8 *here;

    //
    // first off, clip to the screen.
    //
    if (x1 < 0)
    {
        x1 = 0;
    }

    if (x2 > maxx)
    {
        x2 = maxx;
    }

    if (y1 < 0)
    {
        y1 = 0;
    }

    if (y2 > maxy)
    {
        y2 = maxy;
    }

    //
    // is there a rectangle left?
    //
    if (y2 < y1 || x2 < x1)
    {
        return;
    }

    width = x2 - x1 + 1;

    //
    // compute the start and end pages
    //
    offset = ((int32)y1 * (int32)modeInfo.width) + (int32)x1;
    startBank = (offset >> 16) + activePageOffset;

    endOffset = ((int32)y2 * (int32)modeInfo.width) + (int32)x2;
    endBank = (endOffset >> 16) + activePageOffset;

    //
    // select the video page if we need to
    //
    if (startBank != currentBank)
    {
        currentBank = startBank;
        vbeSetBankAAddr(currentBank);
    }

    //
    // if the rectangle is all in one page
    // draw it the easy way
    //
    offset &= 0xffffL;
    if (startBank == endBank)
    {
        address = (uint8 *) MK_FP(0xa000, (int)offset);

        for (i = y1; i <= y2; i++)
        {
            memset(address, color, width);
            address += modeInfo.width;
        }
    }
    else
    {
        //
        // the rectangle spans more than one page
        // do as much as you can in each page
        //
        height = y2 - y1 + 1;
        while (height > 0)
        {
            uint32 start;

            //
            // how many scan lines of the rectangle
            // fit on this page?
            // 
            lines = (0x10000L - offset) / modeInfo.width;
            lines = min(lines, height);

            height -= lines;

            //
            // fill as many lines as possible
            // in this page
            //
            for (i = 0; i < lines; i++)
            {
                memset((uint8 *) MK_FP(0xa000, (uint16) offset),
                       color,
                       width);
                offset += modeInfo.width;
            }

            //
            // handle the case where a scan line crosses
            // a page boundry
            //
            if (height > 0)
            {
                height--;
                start = 0x10000L - offset;
                if (start >= width)
                {
                    memset((uint8 *) MK_FP(0xa000, (uint16) offset),
                           color,
                           width);
    
                    currentBank++;
                    vbeSetBankAAddr(currentBank);
                }
                else
                {
                    memset((uint8 *) MK_FP(0xa000, (uint16) offset),
                           color,
                           start);
    
                    currentBank++;
                    vbeSetBankAAddr(currentBank);

                    memset((uint8 *) MK_FP(0xa000, (uint16) 0),
                           color,
                           width - start);
                }
            }
            offset += modeInfo.width;
            offset &= 0xffffL;
        }
    }
}

//---------------------------------------------------
