/* pngdemo.c - an example of writing a file using libpng */

#include "png.h"

void write_png_file(FILE *,int,int,int,int,int,png_color *);
void set_palette(png_color *);

#define WIDTH 640
#define HEIGHT 480

void main(int argc, char *argv[])
{
   /* these variables are static just to save stack space */
   static char name[] = {"demo.png"};
   static png_color palette[256];
   FILE *fp;
   int depth = 8;		/* 256 colors */
   int interlace = 1;		/* use interlacing */
   int gray = 0;		/* not grayscale */

   set_palette(palette);	/* set up color palette */
   if (fp = fopen(name, "wb"))
   {
      write_png_file(fp, WIDTH, HEIGHT,
		     depth, interlace, gray, palette);
      fclose(fp);
   }
}

/* this function does all the work of writing the image */
void write_png_file(FILE * fp, int width, int height,
	       int depth, int interlace, int gray,
	       png_color * palette)
{
   png_info info;
   png_struct png;
   static png_color ppalette[256];
   static png_byte line_buf[WIDTH];
   int x, y, pass, number_passes;

   if (setjmp(png.jmpbuf))
      return;

   /* initialize the PNG structures */
   png_write_init(&png);
   png_info_init(&info);
   png_init_io(&png, fp);

   /* fill in the png_info values */
   info.width = width;
   info.height = height;
   if (depth > 8)
   {
      info.bit_depth = 8;
      info.color_type = PNG_COLOR_TYPE_RGB;
   }
   else if (gray)
   {
      info.bit_depth = depth;
      info.color_type = PNG_COLOR_TYPE_GRAY;
   }
   else
   {
      info.bit_depth = depth;
      info.color_type = PNG_COLOR_TYPE_PALETTE;
      info.palette = ppalette;
      info.num_palette = (1 << depth);
      info.valid |= PNG_INFO_PLTE;
      for (x = 0; x < (1 << depth); x++)
      {
	 ppalette[x].red = palette[x].red;
	 ppalette[x].green = palette[x].green;
	 ppalette[x].blue = palette[x].blue;
      }
   }

   if (interlace)
      info.interlace_type = 1;
   else
      info.interlace_type = 0;

   png_write_info(&png, &info);

   /* turn on interlace handling */
   if (interlace)
      number_passes = png_set_interlace_handling(&png);
   else
      number_passes = 1;

   /* write the image number_passes times; interlacing details taken care of
    * internally */
   for (pass = 0; pass < number_passes; pass++)
      for (y = 0; y < height; y++)
      {
	 for (x = 0; x < width; x++)
	 line_buf[x] = (y+x)%256; /* diagonal shaded bars */
	 png_write_row(&png, line_buf);
	 printf("writing row %3d\r", y);
      }

   /* deallocate structures */
   png_write_end(&png, NULL);
   png_write_destroy(&png);
   printf("Done             \n");
   fclose(fp);
}

/* make an interesting palette for our image demo */
void
set_palette(png_color * palbuf)
{
   int x, y;
   for (x = 0; x < 256; x++)
   {
      y = x & 255;
      if (x & 256)
	 y = 255 - y;
      palbuf[x].red = y;
      palbuf[x].green = y * y / 255;
      y = (x >> 1) & 255;
      if (x & 512)
	 y = 255 - y;
      palbuf[x].blue = y;
   }
}


