Unit SockCtv;
Interface
 Uses
  SysUtils,
  WinSock,
  Messages;

  const
   CRLF : PChar   = #13#10#0;
   MaxBufferSize  = 255;

  type
    CharArray = array[0..MaxBufferSize] of char;
    Conditions = (Success, Failure, None);

    SockPtr = ^SockInfo;
    SockInfo = object
                Status      : Conditions;
                Host        : pHostent;
                Serv        : pServent;
                ServName,
                Proto       : PChar;
                SocketNo    : tSocket;
                PortNo      : Integer;
                HostName    : CharArray;
                SockAddress : TSockAddr_In;
                Response    : Integer;
                h_addr      : PChar;
                Data        : PChar;
                ErrMsg      : CharArray;
                function    GetWSADataOk : Boolean;
                procedure   CloseWinSock;
                procedure   GetHost;
                procedure   GetService(Service, Protocol : PChar);
                procedure   CreateSock;
                procedure   SetUpAddr;
                procedure   ConnectToHost;
                function    ShowIPAddress : String;
                procedure   SendData; virtual;
                function    GetData(Buffer : PChar) : Boolean; virtual;
                procedure   Abort;
                procedure   WSAErrorMsg;
                constructor Init;
                destructor  Done;
               end;

  var
   wsaData     : PWSAData;
   Buffer      : array[0..MaxBufferSize] of char;

Implementation

  function SockInfo.GetWSADataOK : Boolean;
  var
   VersionReqD : WordRec;
   ErrNo : Integer;

  begin
   with VersionReqD do
   begin
    Hi := 1;
    Lo := 1;
   end;
   New(wsaData);
   GetWSADataOK := WSAStartUp(Word(VersionReqD),wsaData) = 0;
  end;

  procedure SockInfo.CloseWinSock;
  begin
   if wsaData <> NIL then
   begin
    Dispose(wsaData);
    wsaData := NIL;
    WSACleanUp;
   end;
  end;

  procedure SockInfo.GetHost;
  begin
  { Attempt to get host name }
   Host := getHostByName(HostName);
   if Host = NIL then
   begin{Unknown host, so aborting...}
    WSAErrorMsg;
    Status := Failure;
    Exit;
   end;
   Move(Host^.h_addr_list^, h_addr, SizeOf(Host^.h_addr_list^));
   Status := Success;
  end;

  procedure SockInfo.GetService(Service, Protocol : PChar);
  begin
   PortNo := 0;
   ServName := Service;
   Proto := Protocol;
   { attempt to get service }
   Serv := GetServByName(ServName,Proto);
   if Serv = NIL then
   begin { No service available }
    WSAErrorMsg;
    Status := Failure;
    Exit;
   end;
   PortNo := htons(Serv^.s_port);
   Status := Success;
  end;

  procedure SockInfo.CreateSock;
  begin
   SocketNo := socket(PF_INET, SOCK_STREAM, IPPROTO_IP);
   if SocketNo = INVALID_SOCKET then
   begin { Failure to create a socket}
    WSAErrorMsg;
    Status := Failure;
    Exit;
   end;
   Status := Success;
  end;

  procedure SockInfo.SetUpAddr;
  begin
   SockAddress.sin_family            := PF_INET;
   SockAddress.sin_port              := htons(PortNo);
   SockAddress.sin_addr.S_un_b.s_b1  := h_addr[0];
   SockAddress.sin_addr.S_un_b.s_b2  := h_addr[1];
   SockAddress.sin_addr.S_un_b.s_b3  := h_addr[2];
   SockAddress.sin_addr.S_un_b.s_b4  := h_addr[3];
  end;

  procedure SockInfo.ConnectToHost;
  begin
   if connect(SocketNo, @SockAddress, SizeOf(TSockAddr_In)) <> 0 then
   begin
    WSAErrorMsg;
    Status := Failure;
    CloseSocket(SocketNo);
    Exit;
   end;
   Status := Success;
  end;

  function SockInfo.ShowIPAddress : String;
  begin
    ShowIPAddress := ConCat('Connected to ',StrPas(inet_ntoa(SockAddress.sin_addr)));
  end;

  procedure SockInfo.SendData;
  begin
   Response := Send(SocketNo, Data, StrLen(Data), 0);
   if Response = SOCKET_ERROR then
   begin { Error sending data to remote host }
    WSAErrorMsg;
    CloseSocket(SocketNo);
    Status := Failure;
    Exit;
   end;
   Status := Success;
  end;

  function SockInfo.GetData(Buffer : PChar) : Boolean;
  begin
   GetData := True;
   Response := recv(SocketNo, Buffer, MaxBufferSize, 0);
   if Response = SOCKET_ERROR then{ problem, must abandon }
   begin
    WSAErrorMsg;
    CloseSocket(SocketNo);
    GetData := False;
    Status := Failure;
    Exit;
   end
   else
   if Response = 0 then{no more data from the host}
   begin
    Buffer := '';
    CloseSocket(SocketNo);
    Status := Success;
    GetData := False;
    Exit;
   end;
   Buffer[Response] := #0;
  end;

  constructor SockInfo.Init;
  begin
   New(Data);
   Data :='';
  end;

  destructor SockInfo.Done;
  begin
  end;

  procedure SockInfo.WSAErrorMsg;
  begin
   StrPCopy(ErrMsg,'');
   Case WSAGetLastError of
     WSAEINTR           : StrPCopy(ErrMsg,'Interrupted system call');{WSAEINTR}
     WSAEBADF           : StrPCopy(ErrMsg,'Bad file number'); {WSAEBADF}
     WSAEACCES          : StrPCopy(ErrMsg,'Permission denied'); {WSAEINTR}
     WSAEFAULT          : StrPCopy(ErrMsg,'Bad address');{WSAEFAULT}
     WSAEINVAL          : StrPCopy(ErrMsg,'Invalid argument');{WSAEINVAL}
     WSAEMFILE          : StrPCopy(ErrMsg,'Too many open files');{WSAEMFILE}
     WSAEWOULDBLOCK     : StrPCopy(ErrMsg,'Operation would block');{WSAEWOULDBLOCK}
     WSAEINPROGRESS     : StrPCopy(ErrMsg,'Operation now in progress');{WSAEINPROGRESS}
     WSAEALREADY        : StrPCopy(ErrMsg,'OPeration already in progress');{WSAEALREADY}
     WSAENOTSOCK        : StrPCopy(ErrMsg,'Socket operation on nonsocket');{WSAENOTSOCK}
     WSAEDESTADDRREQ    : StrPCopy(ErrMsg,'Destination address required');{WSAEDESTADDRREQ}
     WSAEMSGSIZE        : StrPCopy(ErrMsg,'Message too long');{WSAEMSGSIZE}
     WSAEPROTOTYPE      : StrPCopy(ErrMsg,'Protocol wrong type for socket');{WSAEPROTOTYPE}
     WSAENOPROTOOPT     : StrPCopy(ErrMsg,'Protocol not available');{WSAENOPROTOOPT}
     WSAEPROTONOSUPPORT : StrPCopy(ErrMsg,'Protocol not supported');{WSAEPROTONOSUPPORT}
     WSAESOCKTNOSUPPORT : StrPCopy(ErrMsg,'Socket not supported');{WSAESOCKTNOSUPPORT}
     WSAEOPNOTSUPP      : StrPCopy(ErrMsg,'Operation not supported on socket');{WSAEOPNOTSUPP}
     WSAEPFNOSUPPORT    : StrPCopy(ErrMsg,'Protocol family not supported');{WSAEPFNOSUPPORT}
     WSAEAFNOSUPPORT    : StrPCopy(ErrMsg,'Address family not supported');{WSAEAFNOSUPPORT}
     WSAEADDRINUSE      : StrPCopy(ErrMsg,'Address already in use');{WSAEADDRINUSE}
     WSAEADDRNOTAVAIL   : StrPCopy(ErrMsg,'Can''t assign requested address');{WSAEADDRNOTAVAIL}
     WSAENETDOWN        : StrPCopy(ErrMsg,'Network is down');{WSAENETDOWN}
     WSAENETUNREACH     : StrPCopy(ErrMsg,'Network is unreachable');{WSAENETUNREACH}
     WSAENETRESET       : StrPCopy(ErrMsg,'Network dropped connection on reset');{WSAENETRESET}
     WSAECONNABORTED    : StrPCopy(ErrMsg,'Software caused connection abort');{WSAECONNABORTED}
     WSAECONNRESET      : StrPCopy(ErrMsg,'Connection reset by peer');{WSAECONNRESET}
     WSAENOBUFS         : StrPCopy(ErrMsg,'No buffer space available');{WSAENOBUFS}
     WSAEISCONN         : StrPCopy(ErrMsg,'Socket is already connected');{WSAEISCONN}
     WSAENOTCONN        : StrPCopy(ErrMsg,'Socket is not connected');{WSAENOTCONN}
     WSAESHUTDOWN       : StrPCopy(ErrMsg,'Can''t send after socket shutdown');{WSAESHUTDOWN}
     WSAETOOMANYREFS    : StrPCopy(ErrMsg,'Too many references:can''t splice');{WSAETOOMANYREFS}
     WSAETIMEDOUT       : StrPCopy(ErrMsg,'Connection timed out');{WSAETIMEDOUT}
     WSAECONNREFUSED    : StrPCopy(ErrMsg,'Connection refused');{WSAECONNREFUSED}
     WSAELOOP           : StrPCopy(ErrMsg,'Too many levels of symbolic links');{WSAELOOP}
     WSAENAMETOOLONG    : StrPCopy(ErrMsg,'File name is too long');{WSAENAMETOOLONG}
     WSAEHOSTDOWN       : StrPCopy(ErrMsg,'Host is down');{WSAEHOSTDOWN}
     WSAEHOSTUNREACH    : StrPCopy(ErrMsg,'No route to host');{WSAEHOSTUNREACH}
     WSAENOTEMPTY       : StrPCopy(ErrMsg,'Directory is not empty');{WSAENOTEMPT}
     WSAEPROCLIM        : StrPCopy(ErrMsg,'Too many processes');{WSAEPROCLIM}
     WSAEUSERS          : StrPCopy(ErrMsg,'Too many users');{WSAEUSERS}
     WSAEDQUOT          : StrPCopy(ErrMsg,'Disk quota exceeded');{WSAEDQUOT}
     WSAESTALE          : StrPCopy(ErrMsg,'Stale NFS file handle');{WSAESTALE}
     WSAEREMOTE         : StrPCopy(ErrMsg,'Too many levels of remote in path');{WSAEREMOTE}
     WSASYSNOTREADY     : StrPCopy(ErrMsg,'Network subsystem is unusable');{WSASYSNOTREADY}
     WSAVERNOTSUPPORTED : StrPCopy(ErrMsg,'Winsock DLL cannot support this appliaction');{WSAVERNOTSUPPORTED}
     WSANOTINITIALISED  : StrPCopy(ErrMsg,'Winsock not initialized');{WSANOTINITIALISED}
     WSAHOST_NOT_FOUND  : StrPCopy(ErrMsg,'Host not found');{WSAHOST NOT FOUND}
     WSATRY_AGAIN       : StrPCopy(ErrMsg,'Non authoritative - host not found');{WSATRY_AGAIN}
     WSANO_RECOVERY     : StrPCopy(ErrMsg,'Non recoverable error');
     WSANO_DATA         : StrPCopy(ErrMsg,'Valid name, no data record of requested type');
  end;
end;

 procedure SockInfo.Abort;
 begin
  if wsaData <> NIL then
  begin
   Dispose(wsaData);
   wsaData := NIL;
   WSACleanUp;
  end;
  Done;
 end;

 End.
