/*********************************************************
*  Check a PNG file for legal structure.  This program   *
*  does everything except actually decompress and        *
*  display the graphics data.                            *
*  Convention: Most functions return ERR on error.       *
*********************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#define ERR 1 /* Returned on error */
#define OK 0 /* Returned if no error */

typedef unsigned long U32;  /* 32-bit unsigned value */
typedef unsigned short U16; /* 16-bit unsigned value */
typedef unsigned char BYTE; /* unsigned byte */

/*********************************************************
  Global Symbols
*/
typedef void *PNG;
PNG OpenPNGFile(const char *name, const char *mode);
void ClosePNGFile(PNG f);
int ReadPNG(PNG f);

/*********************************************************
*                                                        *
*  Main Function                                         *
*                                                        *
*********************************************************/

int main(int argc, char **argv)
{ while (argc > 1) {
    char *p=argv[--argc];
    PNG f = OpenPNGFile(p,"r");
    if (f != NULL) {
      printf("Reading PNG file %s.\n",p);
      if(ReadPNG(f)) fprintf(stderr,"!PNG file is bad.\n");
      else printf("PNG file is OK.\n");
      ClosePNGFile(f);
    } else
      fprintf(stderr,"!Couldn't open PNG file %s.\n",p);
  }
  return 0;
}

/*********************************************************
*                                                        *
*  UTILITY FUNCTIONS                                     *
*                                                        *
*********************************************************/

/*********************************************************
  CharToWord: Convert two bytes into a 16-bit integer
  CharToLong: Convert four bytes into a 32-bit integer
  ChunkCode: Generate 4-byte integer from chunk name
*/
#define CharToWord(p) \
   ((((U16)(p)[0]<<8)&0xff00)|(((U32)(p)[1])&0xff))
#define CharToLong(p) \
   (((U32)(p)[0] << 24) | (((U32)(p)[1] & 0xff) << 16) \
    | (((U32)(p)[2] & 0xff) << 8) | ((U32)(p)[3] & 0xff) )
#define ChunkCode(a,b,c,d) \
  (((((((U32)a<<8)+(U32)b)<<8)+(U32)c)<<8)+(U32)d)

/*********************************************************
  InitCRC: Pre-compute table of constants for 32-bit CRC
*/
static U32 Crc32Table[256];
static void InitCRC(void)
{ U32 crc; int i, j;
  for(i=0; i<256; i++) { crc = i;
    for (j=0; j<8; j++)
      crc = (crc >> 1) ^ ((crc & 1)? 0xEDB88320UL : 0);
    Crc32Table[i] = crc;
}}

/*********************************************************
  Crc32: Compute 32-bit CRC
*/
static U32 Crc32(const BYTE *buff, size_t length, U32 crc)
{ const unsigned char *p = buff;
  crc = ~crc;
  while (length-- > 0)
    crc = Crc32Table[(crc^*p++)&0xFF]^((crc>>8)&0xFFFFFF);
  return ~crc;
}

/**********************************************************
**                                                       **
**  PNG DECODER                                          **
**                                                       **
**********************************************************/

/**********************************************************
  PNG_PRIVATE structure
  This structure contains all the information about the
  PNG file.
*/
typedef struct {
  FILE *f;   /* File I/O information */
  int eof;
  struct {  /* IHDR information */
    U32 width, height;
    BYTE bitDepth, colorType, compress, filter, interlace;
    BYTE seen;  /* non-zero after IHDR chunk read */
  } IHDR;
  struct {    /* PLTE: palette information */
    U16 size; /* Can be up to 256 */
    struct { BYTE red, green, blue; } palette[256];
    BYTE seen; /* non-zero after PLTE chunk read */
  } PLTE;
  struct { BYTE seen; } IDAT;  /* IDAT information */
  struct { BYTE seen; } IEND;  /* IEND information */
  struct {   /* gAMA information */
    U32 gamma; /* gamma times 100,000 */
    BYTE seen;
  } gAMA;
  struct {   /* sBIT: number of significant bits */
    BYTE redBits, greenBits, blueBits, alphaBits;
    BYTE seen;
  } sBIT;
  struct {   /* cHRM: Chromaticity information */
    /* All colors are CIE XY values * 100,000 */
    U32 whiteX, whiteY; /* White point */
    U32 redX,redY,greenX,greenY,blueX,blueY;/* primaries */
    BYTE seen;
  } cHRM;
  struct {  /* tRNS: Simplified transparency information */
    BYTE plteAlpha[256]; /* Alpha for each palette */
    U16 trnsRed, trnsBlue, trnsGreen; /*transparent RGB */
    U16 trnsGray; /* transparent Gray */
    BYTE seen;
  } tRNS;
  struct {   /* bKGD: background color */
    U16 bkgdGray;
    U16 bkgdRed,bkgdGreen,bkgdBlue;
    BYTE bkgdPalette;
    BYTE seen;
  } bKGD;
  struct {   /* hIST: Color usage information */
    U16 usage[256];
    BYTE seen;
  } hIST;
  struct {   /* tEXt/zTXt: Text string */
    /* This gets overwritten by each new text string */
    BYTE keyword[32]; /* First 32 bytes of keyword */
    BYTE value[128]; /* First 128 bytes of value */
    BYTE seen;
  } tEXt;
  struct {   /* pHYs: Physical size/aspect ratio */
    U32 xResolution, yResolution;
    BYTE units; /* 0=unknown, 1=meter */
    BYTE seen;
  } pHYs;
  struct {   /* oFFs: Offset of image on page */
    U32 xPosition, yPosition;
    BYTE unit; /* 0=pixel, 1=millionth of meter */
    BYTE seen;
  } oFFs;
  struct {   /* tIME: Time image last modified */
    U16 year;
    BYTE month, day, hour, minute, second;
    BYTE seen;
  } tIME;
} PNG_PRIVATE ;

/*********************************************************
  OpenPNGFile: Returns NULL on error
*/
PNG OpenPNGFile(const char *name, const char *mode)
{ PNG_PRIVATE *pF;
  if ((pF = malloc(sizeof (*pF))) == NULL) return NULL;
  memset(pF,0,sizeof(*pF)); /* Clear out the structure */
  pF->f = fopen(name,mode);
  if (pF->f == NULL) { free(pF); return NULL; }
  return (PNG)pF;
}

/*********************************************************
  ClosePNGFile
*/
void ClosePNGFile(PNG fPublic)
{ PNG_PRIVATE *pF = fPublic;
  fclose(pF->f);
  free(pF);
}

/*********************************************************
  GetBytes: Read sequence of bytes from PNG file
  Returns ERR on error (including EOF)
*/
static int GetBytes(PNG_PRIVATE *pF, size_t length,
                    BYTE *buff)
{ size_t read;
  if ((length == 0) || (pF->eof)) return ERR;
  read = fread(buff,sizeof(BYTE),length, pF->f);
  if (read < length) return ERR;
  if (feof(pF->f)) pF->eof = 1;
  return OK;
}

/*********************************************************
  ReadLong: Read 32-bit value from PNG file
*/
static U32 ReadLong(PNG_PRIVATE *pF, U32 *pLong)
{ BYTE buff[4];
  if(GetBytes(pF,4,buff)) return ERR;
  *pLong = CharToLong(buff);
  return OK;
}

/*********************************************************
  ReadSignature: Verify the 8-byte PNG signature
*/
static int ReadSignature(PNG_PRIVATE *pF)
{ BYTE fileSignature[8];
  BYTE realSignature[] = {137,'P','N','G',13,10,26,10};
  int i;

  if (GetBytes(pF,8,fileSignature)) {
    fprintf(stderr,"!Couldn't read signature.\n");
    return ERR;
  }
  for (i=0; i<8; i++)
    if (fileSignature[i] != realSignature[i]){
      fprintf(stderr,"!PNG signature incorrect.\n");
      return ERR;
    }
  return OK;
}

/*********************************************************
*                                                        *
*  Functions to read and verify chunk data               *
*                                                        *
*********************************************************/

/*********************************************************
  ChunkData: Reads chunk data, verifies CRC
  The `crc' argument gives the CRC of the chunk name.
*/
static int ChunkData(PNG_PRIVATE *pF, U32 length,
                     U32 crc, BYTE *buff)
{ U32 fileCrc;
  if ((GetBytes(pF,length,buff))||(ReadLong(pF,&fileCrc))){
    fprintf(stderr,"!Error reading chunk.\n");
    return ERR;
  }
  if (fileCrc != Crc32(buff,length,crc)) {
    fprintf(stderr,"!Chunk CRC is incorrect.\n");
    return ERR;
  }
  return OK;
}

/*********************************************************
  ChunkLarge: Reads data for chunks of arbitrary size.
  Data is passed to a callback function in small pieces.
  Note that chunks can (in theory) be up to 2gig!!
*/
typedef int (*CHUNKCALLBACK)
     (PNG_PRIVATE *pF,  /* PNG file handle */
      void *clientData, /* Private data for callback */
      const BYTE *data,  /* Part of actual chunk data */
      U32 length); /* Size of partial data */

static int ChunkLarge(PNG_PRIVATE *pF, U32 length, U32 crc,
                      void *clientData,
                      CHUNKCALLBACK CallBack)
{
  BYTE buff[1000]; /* Read 1000 bytes at a time */
  U32 readSize, fileCrc;

  for ( ;length > 0; length-= readSize) {
    if (length > sizeof(buff)) readSize = sizeof(buff);
    else readSize = length;
    if (GetBytes(pF,readSize,buff)) {
      fprintf(stderr,"!Error reading chunk data\n");
      return ERR;
    }
    crc = Crc32(buff,readSize,crc);
    if ((*CallBack)(pF,clientData,buff,readSize))
      return ERR;
  }
  if ((ReadLong(pF,&fileCrc))||(fileCrc != crc)) {
    fprintf(stderr,"!Error in chunk CRC\n");
    return ERR;
  }
  return OK;
}

/*********************************************************
  ChunkUnknown: Verify CRC for unknown chunk type.
*/
static int CBk_Unknown(PNG_PRIVATE *pF, void *unused,
                       const BYTE *data, U32 length)
{ return OK; }

static int ChunkUnknown(PNG_PRIVATE *pF,
                        U32 length, U32 crc)
{ return ChunkLarge(pF,length,crc,NULL,&CBk_Unknown); }

/*********************************************************
*                                                        *
*   Required Chunk Types                                 *
*                                                        *
*********************************************************/

/*********************************************************
  ChunkIDAT: Image Data Chunk
  The callback should decompress and display the image
*/
static int CBk_IDAT(PNG_PRIVATE *pF, void *unused,
                    const BYTE *data, U32 length)
{ return OK; }

static int ChunkIDAT(PNG_PRIVATE *pF, U32 length, U32 crc)
{ /* One quick sanity check */
  if ((pF->IHDR.colorType & 1) && (!pF->PLTE.seen)) {
    fprintf(stderr,"!Missing palette.\n");
    return ERR;
  }
  /* Most of the work is done in the callback */
  if (ChunkLarge(pF,length,crc,NULL,&CBk_IDAT))return ERR;
  pF->IDAT.seen = 1;
  return OK;
}

/*********************************************************
  ChunkIHDR
  The IHDR chunk is always 13 bytes long.
*/
static int ChunkIHDR(PNG_PRIVATE *pF, U32 length, U32 crc)
{ BYTE buff[13];

  if (pF->IHDR.seen) {
    fprintf(stderr,"!Redundant IHDR chunk.\n");
    return ERR;
  }
  if(ChunkData(pF, length, crc, buff)) return ERR;

  pF->IHDR.width = CharToLong(buff);
  pF->IHDR.height = CharToLong(buff+4);
  pF->IHDR.bitDepth = buff[8];
  pF->IHDR.colorType = buff[9];
  pF->IHDR.compress = buff[10];
  pF->IHDR.filter = buff[11];
  pF->IHDR.interlace = buff[12];

  /* Check for illegal values */
  if (  ((pF->IHDR.colorType & 3) == 1)/* mono & palette */
      ||(pF->IHDR.colorType > 6) /* invalid color type */
      ||(pF->IHDR.compress != 0) ) { /* bad compress */
    fprintf(stderr,"!Error in IHDR chunk\n");
    return ERR;
  }

  /* Dump it to stdout. */
  printf("    width: %ld, height: %ld, bits/pixel: %d\n",
         pF->IHDR.width,pF->IHDR.height,pF->IHDR.bitDepth);
  printf("    colorType: %d (%s%s%s), compress: %d (%s)\n",
         pF->IHDR.colorType,
         (pF->IHDR.colorType & 1)?"palette":"direct color",
         (pF->IHDR.colorType & 2)?", color":", monochrome",
         (pF->IHDR.colorType & 4)?", alpha":", no alpha",
         pF->IHDR.compress,
         (pF->IHDR.compress == 0)?"deflated":"unknown");
  printf("       filter: %d (%s), interlace: %d (%s)\n",
         pF->IHDR.filter,
         (pF->IHDR.filter == 0)?"adaptive":"unknown",
         pF->IHDR.interlace,
         (pF->IHDR.interlace == 0) ? "none":
         (pF->IHDR.interlace == 1) ? "Adam7":
         "unknown");
  pF->IHDR.seen = 1;
  return OK;
}

/*********************************************************
  ChunkPLTE: The PLTE chunk contains palette information.
*/
static int ChunkPLTE(PNG_PRIVATE *pF, U32 length, U32 crc)
{ BYTE palette[768], *pltePtr;
  int i;

  if (pF->PLTE.seen) { /* Can only be one PLTE chunk */
    fprintf(stderr,"!Redundant PLTE chunk.\n");
    return ERR;
  } else if ((length % 3) != 0) { /* 3 bytes per palette */
    fprintf(stderr,"!Illegal PLTE chunk size.\n");
    return ERR;
  } else if ((length / 3) > 256) { /* <= 256 entries */
    fprintf(stderr,"!PLTE chunk too large.\n");
    return ERR;
  }
  if(ChunkData(pF, length, crc, palette)) return ERR;

  pF->PLTE.size = length/3;
  for (pltePtr=palette,i=0; length > 0; i++, length -= 3) {
    printf("    Palette %3d: (%3d,%3d,%3d)\n",
           i,pltePtr[0],pltePtr[1],pltePtr[2]);
    pF->PLTE.palette[i].red = *pltePtr++;
    pF->PLTE.palette[i].green = *pltePtr++;
    pF->PLTE.palette[i].blue = *pltePtr++;
  }
  pF->PLTE.seen = 1;
  return OK;
}

/********************************************************
  ChunkIEND: read IEND chunk
*/
static int ChunkIEND(PNG_PRIVATE *pF, U32 length, U32 crc)
{ if(ChunkUnknown(pF, length, crc)) return ERR;
  pF->IEND.seen = 1;
  return OK;
}

/*********************************************************
*                                                        *
*   Ancillary (optional) Chunk Types                     *
*                                                        *
*********************************************************/

/*********************************************************
  ChunkgAMA: Gamma information
*/
static int ChunkgAMA(PNG_PRIVATE *pF, U32 length, U32 crc)
{ BYTE buff[4];
  if(ChunkData(pF, length, crc, buff)) return ERR;
  pF->gAMA.seen = 1;
  pF->gAMA.gamma = CharToLong(buff);
  return OK;
}

/*********************************************************
  ChunksBIT: Significant bits of data
*/
static int ChunksBIT(PNG_PRIVATE *pF, U32 length, U32 crc)
{ BYTE buff[4];
  int expectedLength;

  if (pF->sBIT.seen) {  /* Sanity checks */
    fprintf(stderr,"!Redundant sBIT chunk.\n");
    return ERR;
  }
  /* Length is 3 for color, 1 for mono, +1 for alpha */
  expectedLength = (pF->IHDR.colorType & 2)? 3 : 1;
  if (pF->IHDR.colorType & 4) expectedLength++;
  if (length != expectedLength) {
    fprintf(stderr,"!sBIT length %ld should be %d\n",
            length,expectedLength);
    return ERR;
  }
  if(ChunkData(pF, length, crc, buff)) return ERR;
  if (pF->IHDR.colorType & 2) { /* Color */
    pF->sBIT.redBits = buff[0];
    pF->sBIT.greenBits = buff[1];
    pF->sBIT.blueBits = buff[2];
    if (pF->IHDR.colorType & 4) /* alpha? */
      pF->sBIT.alphaBits = buff[3];
  } else { /* Monochrome */
    pF->sBIT.redBits = pF->sBIT.greenBits 
      = pF->sBIT.blueBits = buff[0];
    if (pF->IHDR.colorType & 4) /* alpha? */
      pF->sBIT.alphaBits = buff[1];
  }

  pF->sBIT.seen = 1;
  return OK;
}


/*********************************************************
  ChunkcHRM: Chromaticity information
*/
static int ChunkcHRM(PNG_PRIVATE *pF, U32 length, U32 crc)
{ BYTE buff[32];

  if (pF->cHRM.seen) {   /* Sanity checks */
    fprintf(stderr,"!Redundant cHRM chunk.\n");
    return ERR;
  }
  if(ChunkData(pF, length, crc, buff)) return ERR;
  pF->cHRM.whiteX = CharToLong(buff);
  pF->cHRM.whiteY = CharToLong(buff+4);
  pF->cHRM.redX = CharToLong(buff+8);
  pF->cHRM.redY = CharToLong(buff+12);
  pF->cHRM.greenX = CharToLong(buff+16);
  pF->cHRM.greenY = CharToLong(buff+20);
  pF->cHRM.blueX = CharToLong(buff+24);
  pF->cHRM.blueY = CharToLong(buff+28);
  pF->cHRM.seen = 1;
  return OK;
}


/*********************************************************
  ChunktRNS: Simplified transparency information
*/
static int ChunktRNS(PNG_PRIVATE *pF, U32 length, U32 crc)
{ BYTE buff[256]; /* Length of longest tRNS chunk */
  U32 expectedLength = length;

  if (pF->tRNS.seen) {   /* Sanity checks */
    fprintf(stderr,"!Redundant tRNS chunk.\n");
    return ERR;
  }
  switch(pF->IHDR.colorType) {
  case 0: expectedLength = 2; break;
  case 2: expectedLength = 6; break;
  case 3: expectedLength = length; break;
  case 4:  case 6:
    fprintf(stderr,"!Cannot use tRNS with full alpha.\n");
    return ERR;
  }
  if (length != expectedLength) {
    fprintf(stderr,"!tRNS length %ld should be %ld\n",
            length, expectedLength);
    return ERR;
  }
  if(ChunkData(pF, length, crc, buff)) return ERR;

  if (pF->IHDR.colorType == 0) { /* grayscale */
    pF->tRNS.trnsGray = CharToWord(buff);
    printf("    Gray %d is transparent.\n",
           pF->tRNS.trnsGray);
  } else if (pF->IHDR.colorType == 2) { /* RGB */
    pF->tRNS.trnsRed = CharToWord(buff);
    pF->tRNS.trnsGreen = CharToWord(buff+2);
    pF->tRNS.trnsBlue = CharToWord(buff+4);
    printf("    Color (%d, %d, %d) is transparent.\n",
           pF->tRNS.trnsRed, pF->tRNS.trnsGreen,
           pF->tRNS.trnsBlue );
  } else { /* pF->IHDR.colorType == 3, palette */
    int i = 0;
    for (;i<length;i++) {
      pF->tRNS.plteAlpha[i] = buff[i];
      printf("    Palette %d has transparency %d.\n", i,
             pF->tRNS.plteAlpha[i]);
    }
    for (;i<256;i++) /* Rest are fully opaque */
      pF->tRNS.plteAlpha[i]=255;
  }
  pF->tRNS.seen = 1;
  return OK;
}


/*********************************************************
  ChunkbKGD: Background color
*/
static int ChunkbKGD(PNG_PRIVATE *pF, U32 length, U32 crc)
{ BYTE buff[6]; /* Length of longest bKGD chunk */
  U32 expectedLength;

  if (pF->bKGD.seen) {   /* Sanity checks */
    fprintf(stderr,"!Cannot have multiple bKGD chunks.\n");
    return ERR;
  }
  switch(pF->IHDR.colorType) {
  case 0: case 4: expectedLength = 2; break;
  case 2: case 6: expectedLength = 6; break;
  case 3: expectedLength = 1; break;
  default: fprintf(stderr,"!Invalid color type.\n");
    return ERR;
  }
  if (length != expectedLength) {
    fprintf(stderr,"!bKGD length %ld should be %ld\n",
            length, expectedLength);
    return ERR;
  }
  if(ChunkData(pF, length, crc, buff)) return ERR;
  if ((pF->IHDR.colorType & 1) == 0) { /* grayscale */
    pF->bKGD.bkgdGray = CharToWord(buff);
  } else if ((pF->IHDR.colorType & 1) == 0) {/*full color*/
    pF->bKGD.bkgdRed = CharToWord(buff);
    pF->bKGD.bkgdGreen = CharToWord(buff+2);
    pF->bKGD.bkgdBlue = CharToWord(buff+4);
  } else { /* pF->IHDR.colorType == 3, palette */
    pF->bKGD.bkgdPalette = buff[0];
  }
  pF->tRNS.seen = 1;
  return OK;
}


/*********************************************************
  ChunkhIST: Histogram
*/
static int ChunkhIST(PNG_PRIVATE *pF, U32 length, U32 crc)
{ BYTE buff[512]; /* Length of longest hIST chunk */
  int i;

  if (pF->hIST.seen) {   /* Sanity checks */
    fprintf(stderr,"!Redundant hIST chunk.\n");
    return ERR;
  }
  if (length > 512) {
    fprintf(stderr,"!hIST length %ld  >= 512.\n", length);
    return ERR;
  }
  if(ChunkData(pF, length, crc, buff)) return ERR;
  for(i=0;i<length;i++)
    pF->hIST.usage[i]=CharToWord(buff+(i*2));
  for(;i<256;i++) /* Rest must be unused */
    pF->hIST.usage[i]=0;
  pF->hIST.seen = 1;
  return OK;
}

/*********************************************************
  ChunktEXt: Text comment

  Because there's no a priori limit on the length of the
  keyword or text value, this uses the callback method to
  process the argument.  The textInfo structure keeps
  track of whether the keyword has been completely scanned
  and the current position in the keyword or value string.
*/
struct textInfo {
  BYTE finishedKeyword;
  U32 position;
};

static int CBk_tEXt(PNG_PRIVATE *pF, void *clientData,
                    const BYTE *data, U32 length)
{
  struct textInfo *info = clientData;

  if (!info->finishedKeyword) { /* Reading keyword */
    while ((*data)&&(length > 0)) {
      if (info->position < sizeof(pF->tEXt.keyword))
        pF->tEXt.keyword[info->position] = *data;
      info->position++; data++; length--;
    }
    if (info->position < sizeof(pF->tEXt.keyword))
      pF->tEXt.keyword[info->position] = 0;
    else
      pF->tEXt.keyword[sizeof(pF->tEXt.keyword)-1] = 0;
    if (length > 0) { /* reset for the value part */
      info->finishedKeyword = 1;
      info->position = 0; /* Start at beginning of value */
      data++; length--; /* Skip the null */
    }
  }
  if (info->finishedKeyword) { /* Reading value */
    while (length > 0) {
      if (info->position < sizeof(pF->tEXt.value))
        pF->tEXt.value[info->position] = *data;
      info->position++; data++; length--;
    }
    if (info->position < sizeof(pF->tEXt.value))
      pF->tEXt.value[info->position] = 0;
    else
      pF->tEXt.value[sizeof(pF->tEXt.value)-1] = 0;
  }
  return OK;  /* No error */
}

static int ChunktEXt(PNG_PRIVATE *pF, U32 length, U32 crc)
{
  struct textInfo info = {0, /* Keyword not finished */
                          0};/* Start at first character */
  if (ChunkLarge(pF,length,crc,&info,&CBk_tEXt))
    return ERR;
  printf("    %s: %s\n",pF->tEXt.keyword,pF->tEXt.value);
  return OK;
}

/*********************************************************
  ChunkzTXt: Compressed Text comment

  The compressed text chunk is identical to the text chunk
  except that the text value is compressed with the
  `deflate' algorithm.  This code retrieves the
  (uncompressed) keyword and just ignores the rest.
*/
static int ChunkzTXt(PNG_PRIVATE *pF, U32 length, U32 crc)
{
  struct textInfo info = {0, 0};
  if (ChunkLarge(pF,length,crc,&info,&CBk_tEXt))
    return ERR;
  /* Should decompress text here */
  printf("    %s: <compressed text>\n",pF->tEXt.keyword);
  return OK;
}


/*********************************************************
  ChunkpHYs: Physical image size/Aspect ratio
*/
static int ChunkpHYs(PNG_PRIVATE *pF, U32 length, U32 crc)
{ BYTE buff[9]; /* Length of pHYs chunk */

  if (pF->pHYs.seen) {   /* Sanity checks */
    fprintf(stderr,"!Redundant pHYs chunk.\n");
    return ERR;
  }
  if(ChunkData(pF, length, crc, buff)) return ERR;
  pF->pHYs.xResolution = CharToLong(buff);
  pF->pHYs.yResolution = CharToLong(buff+4);
  pF->pHYs.units = buff[8];
  pF->pHYs.seen = 1;
  return OK;
}

/*********************************************************
  ChunkoFFs: Offset of image on page
*/
static int ChunkoFFs(PNG_PRIVATE *pF, U32 length, U32 crc)
{ BYTE buff[9]; /* Length of oFFs chunk */

  if (pF->oFFs.seen) {   /* Sanity checks */
    fprintf(stderr,"!Redundant oFFs chunk.\n");
    return ERR;
  }
  if(ChunkData(pF, length, crc, buff)) return ERR;

  pF->oFFs.xPosition = CharToLong(buff);
  pF->oFFs.yPosition = CharToLong(buff+4);
  pF->oFFs.unit = buff[8];
  pF->oFFs.seen = 1;
  return OK;
}

/*********************************************************
  ChunktIME: Time picture last modified
*/
static int ChunktIME(PNG_PRIVATE *pF, U32 length, U32 crc)
{ BYTE buff[7]; /* Length of tIME chunk */
  if (pF->tIME.seen) {   /* Sanity checks */
    fprintf(stderr,"!Redundant tIME chunk.\n");
    return ERR;
  }
  if(ChunkData(pF, length, crc, buff)) return ERR;

  pF->tIME.year = CharToWord(buff);
  pF->tIME.month = buff[2];    pF->tIME.minute = buff[5];
  pF->tIME.day = buff[3];      pF->tIME.second = buff[6];
  pF->tIME.hour = buff[4];     pF->tIME.seen = 1;
  return OK;
}

/*********************************************************
  ReadChunk: returns ERR if error
*/
#define PRE_IDAT 1

static struct {
  U32 code;
  int (*function)(PNG_PRIVATE *, U32 length, U32 crc);
  int size;
  int where;
} chunkInfo[] = {
  {ChunkCode('I','H','D','R'), &ChunkIHDR, 13, 0},
  {ChunkCode('P','L','T','E'), &ChunkPLTE, -1, PRE_IDAT},
  {ChunkCode('g','A','M','A'), &ChunkgAMA,  4, PRE_IDAT},
  {ChunkCode('s','B','I','T'), &ChunksBIT, -1, PRE_IDAT},
  {ChunkCode('c','H','R','M'), &ChunkcHRM, 32, PRE_IDAT},
  {ChunkCode('t','R','N','S'), &ChunktRNS, -1, PRE_IDAT},
  {ChunkCode('b','K','G','D'), &ChunkbKGD, -1, PRE_IDAT},
  {ChunkCode('h','I','S','T'), &ChunkhIST, -1, PRE_IDAT},
  {ChunkCode('p','H','Y','s'), &ChunkpHYs,  9, PRE_IDAT},
  {ChunkCode('o','F','F','s'), &ChunkoFFs,  9, PRE_IDAT},
  {ChunkCode('I','D','A','T'), &ChunkIDAT, -1, 0},
  {ChunkCode('t','E','X','t'), &ChunktEXt, -1, 0},
  {ChunkCode('z','T','X','t'), &ChunkzTXt, -1, 0},
  {ChunkCode('t','I','M','E'), &ChunktIME,  7, 0},
  {ChunkCode('I','E','N','D'), &ChunkIEND,  0, 0},
  {0UL} /* Mark end of list */
};

static int ReadChunk(PNG_PRIVATE *pF)
{
  U32 length, chunkCode, crc;
  BYTE chunkName[5];
  int i = 0;

  if ((ReadLong(pF,&length))||(GetBytes(pF,4,chunkName))){
    fprintf(stderr,"!Couldn't read length and name.\n");
    return ERR;
  }
  chunkName[4]=0;

  /* Dump and interpret name of chunk */
  printf("Chunk: %s (%s%s%s%s) length: %ld\n",
         chunkName, /* Name of chunk */
         (chunkName[0]&0x20)?"ancillary":"critical",
         (chunkName[1]&0x20)?", private":", public",
         (chunkName[2]&0x20)?", <illegal?>":"",
         (chunkName[3]&0x20)?", copy":", don't copy",
         length);

  crc = Crc32(chunkName,4,0); /* Begin accumulating CRC */

  chunkCode = CharToLong(chunkName);
  for (i=0; chunkInfo[i].code; i++) {
    if (chunkInfo[i].code == chunkCode) { /* Found it! */
      if ( (chunkInfo[i].size >= 0)  /* Have fixed size?*/
          && (length != chunkInfo[i].size) ) {
        fprintf(stderr,"!Chunk size %ld should be %d.\n",
                length, chunkInfo[i].size);
        return ERR;
      }
      if (  (chunkInfo[i].where == PRE_IDAT)
          &&(pF->IDAT.seen)) { /* Must precede IDAT? */
        fprintf(stderr,"!%s must precede IDAT.\n",
		chunkName);
        return ERR;
      }
      return ((*chunkInfo[i].function)(pF,length,crc));
    }
  }
  return (ChunkUnknown(pF,length,crc));
}

/*********************************************************
  ReadPNG: Read PNG file, return ERR if error, else 0
*/
int ReadPNG(PNG fPublic)
{ PNG_PRIVATE *pF = fPublic;

  InitCRC();
  if (ReadSignature(pF)) return ERR;
  while (!pF->IEND.seen) { /* Read until IEND */
    if(ReadChunk(pF)) return ERR;
    if (!pF->IHDR.seen) {
      fprintf(stderr,"!First chunk must be IHDR.\n");
      return ERR;
    }
  }
  if (!pF->IDAT.seen) { /* No image data seen?? */
    fprintf(stderr,"!End of file, but no image seen?\n");
    return ERR;
  }
  return OK;
}


