; Sample triple-buffered animation for the VGA under DOS.
; Uses mode 0dh (320x200, 16 colors) so there's enough
; memory for three pages; other modes that support three
; pages on standard VGA include mode 0eh (640x200,
; 16 colors) and both the 320x200 and 320x240 variants
; of Mode X.
;
; The jerkiness of the animation has nothing to do with
; triple buffering. The animation looks jerky because of the
; 8-pixel movement increments, which are used to simplify
; 16-color drawing by only requiring byte-aligned drawing.
;
; Tested with MASM 5.10a

        .286
        .model  small
        .stack 200h
        .data
BlockX          dw      96
BlockY          dw      96
BlockXInc       dw      8
BlockYInc       dw      8

BLOCK_HEIGHT    equ     32
BLOCK_WIDTH     equ     32      ;must be a multiple of 16

CurrentPage     dw      0

SCR_WIDTH               equ     320
SCR_HEIGHT              equ     200
SCR_WIDTH_IN_BYTES      equ     (SCR_WIDTH/8)

; Segments at whch page 0, page1, and page 2 start
PageSegs        label   word
        dw      0a000h
        dw      0a200h
        dw      0a400h

CRTC_INDEX              equ     03d4h
START_ADDRESS_HIGH      equ     0ch

; Start Address High register settings for the pages
PageStarts      db      0, 20h, 40h

TIMER_COUNT     equ     46ch    ;address in segment at 0
                                ; of BIOS timer tick count
        .code
Start:
        cld

        mov     ax,@data
        mov     ds,ax

        mov     ax,0dh
        int     10h     ;set mode 0dh

; Draw each page in turn, moving the block from frame to
; frame
FrameLoop:
        mov     bx,[CurrentPage]
        add     bx,bx           ;*2 for word lookup
        mov     es,[PageSegs+bx] ;point ES:0 to current page
        sub     di,di
        sub     ax,ax
        mov     cx,(SCR_HEIGHT*SCR_WIDTH_IN_BYTES)/2
        rep     stosw           ;clear the page

        mov     ax,[BlockY]
        mov     dx,SCR_WIDTH_IN_BYTES
        mul     dx      ;top row of block to draw
        mov     di,[BlockX]
        shr     di,3    ;byte address of left edge of block
        add     di,ax   ;ES:DI points to block start

; Draw the block
        mov     ax,0ffffh
        mov     dx,BLOCK_HEIGHT
BlockLoop:
        mov     cx,BLOCK_WIDTH/16
        rep     stosw
        add     di,SCR_WIDTH_IN_BYTES-(BLOCK_WIDTH/8)
        dec     dx
        jnz     BlockLoop

; Move the block
        mov     ax,[BlockX]
        add     ax,[BlockXInc]
        cmp     ax,0
        jnl     CheckXRight
        neg     [BlockXInc]
        add     ax,[BlockXInc]
        add     ax,[BlockXInc]
CheckXRight:
        cmp     ax,SCR_WIDTH-BLOCK_WIDTH
        jng     SaveX
        neg     [BlockXInc]
        add     ax,[BlockXInc]
        add     ax,[BlockXInc]
SaveX:
        mov     [BlockX],ax

        mov     ax,[BlockY]
        add     ax,[BlockYInc]
        cmp     ax,0
        jnl     CheckYBottom
        neg     [BlockYInc]
        add     ax,[BlockYInc]
        add     ax,[BlockYInc]
CheckYBottom:
        cmp     ax,SCR_HEIGHT-BLOCK_HEIGHT
        jng     SaveY
        neg     [BlockYInc]
        add     ax,[BlockYInc]
        add     ax,[BlockYInc]
SaveY:
        mov     [BlockY],ax

; Wait for the timer tick to change before flipping the
; page, guaranteeing at least 54 ms between flips, so
; the previous page flip is sure to have finished first
        sub     ax,ax
        mov     es,ax   ;point to the segment at 0, where
                        ; the timer tick count resides
        mov     ax,word ptr es:[TIMER_COUNT]
                        ;get the low word of the timer count
WaitTickLoop:
        cmp     word ptr es:[TIMER_COUNT],ax
        jz      WaitTickLoop

; Flip the page in hardware (this will not actually take
; effect until the end of the screen frame, which happens
; once every 15 ms or so, but we won't care because we'll
; be drawing to neither the old or new displayed page, but
; to a third page)
        mov     bx,[CurrentPage]
        mov     ah,[PageStarts+bx]
        mov     al,START_ADDRESS_HIGH
        mov     dx,CRTC_INDEX
        out     dx,ax

; Change the current page to a third page that is neither
; the old or new displayed page
        mov     ax,[CurrentPage]
        inc     ax
        cmp     ax,3
        jb      NoPageWrap
        sub     ax,ax   ;wrap back from 2 to 0
NoPageWrap:
        mov     [CurrentPage],ax


;See if a key has been pressed.
        mov     ah,0bh          ;DOS check standard input status fn
        int     21h
        and     al,al           ;is a key pending?
        jnz     Done            ;yes, done
        jmp     FrameLoop       ;no
Done:
        mov     ah,1
        int     21h             ;clear the key & done

        mov     ax,3
        int     10h     ;restore text mode

        mov     ah,4ch
        int     21h     ;return to DOS

        end     Start
