/* File: VRSYNC.C
** Description:
**   Syncs the system timer to the display vertical retrace.
** Copyright:
**   Copyright 1994, David G. Roberts
** Acknowledgements:
**   The basic technique demonstrated here comes from public domain
**   code written by Petteri Kangaslampi.
*/

#include <assert.h>
#include <dos.h>
#include <stdlib.h>
#include "gamedefs.h"
#include "retrace.h"
#include "timer.h"
#include "vrsync.h"

/* constants */
#define MARGIN 500 /* 500 1.1925 MHz clocks of margin = ~420 microsec */

static BOOL     RetraceTimerInstalled = FALSE;
static UINT16   ReinitCount;
static void far interrupt (*OldSysTimerVect)();

/*
    Function: RetraceTimerInit
    Description:
        Installs a retrace interrupt procedure to be called
        whenever a retrace occurs.

        THE FUNCTION SPECIFIED BY TIMERPROC IS AN INTERRUPT
        FUNCTION.  IT SHOULD NOT CALL ANY DOS FUNCTIONS AND
        SHOULD REENABLE INTERRUPTS AS SOON AS POSSIBLE AFTER
        TAKING CONTROL OF THE SYSTEM.
*/
void RetraceTimerInit(void far interrupt (*TimerProc)())
{
    UINT16 FirstRead;
    UINT16 SecondRead;
    int Delta;

    assert(RetraceTimerInstalled == FALSE);

    /* figure out refresh rate */
    do {
        /* wait for refresh */
        WaitVerticalRetraceStart();
        /* init timer with max count */
        ProgramTimer(SELECT_0 | RW_LSB_MSB | MODE_EVENT | TYPE_BINARY, 0);
        /* wait for refresh again */
        WaitVerticalRetraceStart();
        /* check to see what the count is */
        FirstRead = ReadTimerCount(0);

        /* now do it again */
        WaitVerticalRetraceStart();
        /* init timer with max count */
        ProgramTimer(SELECT_0 | RW_LSB_MSB | MODE_EVENT | TYPE_BINARY, 0);
        /* wait for refresh again */
        WaitVerticalRetraceStart();
        /* check to see what the count is */
        SecondRead = ReadTimerCount(0);

        /* compute the delta and do this again if it's too big */
        /* if it's too big, we probably got an interrupt at some point */
        /*   that skewed our timing */
        Delta = SecondRead - FirstRead;
    } while (Delta > 2 || Delta < -2);

    /* set ReinitCount to the quickest time we got minus some margin */
    /* remember 0 = 0x10000, so 0 - FirstRead = 0x10000 - FirstRead */
    ReinitCount = MIN(0 - FirstRead, 0 - SecondRead) - MARGIN;

    /* sync timer with refresh and install user hook function */
    WaitVerticalRetraceStart();
    OldSysTimerVect = HookAndProgramSysTimer(TimerProc,
        SELECT_0 | RW_LSB_MSB | MODE_EVENT | TYPE_BINARY, ReinitCount);
    outportb(PIC, NONSPECIFIC_EOI);

    RetraceTimerInstalled = TRUE;
}

/*
    Function: RetraceTimerStop
    Description:
        Deinstalls the retrace interrupt procedure.  The original
        timer vector is put back in place and the timer is
        programmed with its default values.  Finally, the BIOS
        tick count is reinitialized from the RTC because the
        old vector was never called.
*/
void RetraceTimerStop(void)
{
    assert(RetraceTimerInstalled == TRUE);

    UnhookAndRestoreSysTimer(OldSysTimerVect);
    RestoreDOSTime();
}

/*
    Function: RetraceTimerReinit
    Description:
        This function should be called at the end of the user
        retrace timer procedure to reinitialize the timer with
        the correct count values and restart it.  Do not call
        this function before calling RetraceTimerInit to
        calculate the correct count values first.
*/
void RetraceTimerReinit(void)
{
    assert(RetraceTimerInstalled == TRUE);

    ProgramTimer(SELECT_0 | RW_LSB_MSB | MODE_EVENT | TYPE_BINARY,
        ReinitCount);
}

