/* MACROS */
#define GET_COUNTER(Mode)   ((Mode & SELECT_MASK) >> 6)
#define GET_RWL(Mode)   ((Mode & RWL_MASK) >> 4)

/*
    Function: ProgramTimer
    Description:
        Programs a timer with the mode and count specified.  The
        routine understands writing only the MSB or LSB and does
        the right thing.  Don't set the mode byte to latch
        counters or do the read-back command.
*/
void ProgramTimer(UINT8 Mode, UINT16 Count)
{
    int Timer;
    int Bytes;

    Timer = GET_COUNTER(Mode);
    Bytes = GET_RWL(Mode);

    assert(Timer != 0x3); /* catch read-back command */

    asm pushf;
    asm cli; /* turn off interrupts just so nobody messes with us */

    /* output mode byte */
    outportb(COUNTER_MODE, Mode);

    /* write count */
    switch (Bytes) {
        case 1: /* write MSB */
            outportb(COUNTER_0 + Timer, HIGHBYTE(Count));
            break;
        case 2: /* write LSB */
            outportb(COUNTER_0 + Timer, LOWBYTE(Count));
            break;
        case 3: /* write LSB, MSB */
            /* output in little endian order (i.e., LSB first) */
            outportb(COUNTER_0 + Timer, LOWBYTE(Count));
            outportb(COUNTER_0 + Timer, HIGHBYTE(Count));
            break;
        default:
            assert(Bytes != 0x0); /* catch latch command */
            break;
    }

    asm popf; /* interrupts back on if they were previously on */
}

