/* Portable Joystick Calls, By John Gilbert */
#include <stdio.h>
#include <conio.h>
#include <dos.h>
#include "joystick.h"

js_bios(joy) /* uses the bios calls */
  struct joystick *joy;
{
  union REGS js_regs;

  js_regs.h.ah = JS_FNCT;
  js_regs.w.dx = 0x00; /*get the buttons*/
  int386(JS_INT, &js_regs, &js_regs);
  if (js_regs.w.cflag & INTR_CF) {
    /*if the carry flag set, no joystick*/
    joy->js_ax = joy->js_ay = joy->js_bx =
    joy->js_by = joy->js_but = 0;
    return(-1);
    }
  joy->js_but = ~js_regs.w.ax & 0x00f0;
  js_regs.h.ah = JS_FNCT;
  js_regs.w.dx = 0x01; /* get the positions */
  int386(JS_INT, &js_regs, &js_regs);
  joy->js_ax = js_regs.w.ax; joy->js_ay = js_regs.w.bx;
  joy->js_bx = js_regs.w.cx; joy->js_by = js_regs.w.dx;
  return(0);
}

js_get(joy) /* a much faster Joystick code */
  struct joystick *joy;
{
  static unsigned char joyray[JS_TIMEOUT + 1];
  register unsigned char *rayp = joyray, *mid, *end;
  register int value = JS_TIMEOUT;

 /* disable inturupts so timeing is not affected */
  _disable();
  outp(JS_PORT, value); /* send anything to port */
  /* read the ports into an array until no more data */
  while (value-- && ((*rayp++ = inp(JS_PORT)) & 0x0F))
    ; /* if no more data, it's done */
  /* enable inturupts, all timeing dep. code is done */
  _enable();
  /* renormalize value */
  value = JS_TIMEOUT - value - 1;

/* binary search for event in array */
#define BSEARCH(val,mask) \
  if (*(end = joyray + value) & mask) val = value; \
  else { \
    rayp = joyray; \
    while (rayp < end) { \
      mid = rayp + ((end - rayp) >> 1); \
      if (*mid & mask) rayp = mid + 1; \
      else end = mid; \
      } \
    val = end - joyray; \
    } 

  BSEARCH(joy->js_ax, 0x01);
  BSEARCH(joy->js_ay, 0x02);
  BSEARCH(joy->js_bx, 0x04);
  BSEARCH(joy->js_by, 0x08);
  joy->js_but = ~joyray[0] & 0xf0;
  return(0);
}

void
js_init(joy)
  struct joystick *joy;
{
  if ((joy->js_ax_nrm =
    (int *) malloc(4 * (JS_TIMEOUT + 1) * sizeof(int)))
    == NULL) {
    fprintf(stderr, "Can't alloc memory in js_init\n");
    exit(1);
    }
  joy->js_ay_nrm = joy->js_ax_nrm + JS_TIMEOUT + 1;
  joy->js_bx_nrm = joy->js_ay_nrm + JS_TIMEOUT + 1;
  joy->js_by_nrm = joy->js_bx_nrm + JS_TIMEOUT + 1;
}

void
js_free(joy)
  struct joystick *joy;
{
  if (joy->js_ax_nrm != NULL)
    free (joy->js_ax_nrm);
  joy->js_ax_nrm = joy->js_ay_nrm =
  joy->js_bx_nrm = joy->js_by_nrm = NULL;
}

void
js_calibrate(joy)
  struct joystick *joy;
{
  int butnotpushed = 1;

  fprintf
  (stderr, "\nPlease move joystick(s) to all four corners,\n");
  fprintf
  (stderr, "then center the joystick(s) and press a button.\n");
  js_get(joy);  /* start everything in center */ 
  joy->js_ax_min = joy->js_ax_max = joy->js_ax;
  joy->js_ay_min = joy->js_ay_max = joy->js_ay;
  joy->js_bx_min = joy->js_bx_max = joy->js_bx;
  joy->js_by_min = joy->js_by_max = joy->js_by;

#define MINMAX(x,y,z) \
  y = (y <= x) ? y : x; z = (z >= x) ? z : x

/* this will get the data until the button is released */ 
  while (butnotpushed || joy->js_but) {
    if (joy->js_but) butnotpushed = 0;
    /* get mins and maxes */
    MINMAX(joy->js_ax, joy->js_ax_min, joy->js_ax_max);
    MINMAX(joy->js_ay, joy->js_ay_min, joy->js_ay_max);
    MINMAX(joy->js_bx, joy->js_bx_min, joy->js_bx_max);
    MINMAX(joy->js_by, joy->js_by_min, joy->js_by_max);
    js_get(joy);
    }
  /* this makes it more accurate, see PS. */ 
  js_get(joy); 
  /* get centers and adjust min and max values */
  joy->js_ax_ctr = joy->js_ax;
  joy->js_ay_ctr = joy->js_ay;
  joy->js_bx_ctr = joy->js_bx;
  joy->js_by_ctr = joy->js_by;
  fprintf(stderr, "Thank you.\n");
}

void
js_dotable(joy, flag, min, ctr, max, ctrwth, endwth)
  struct joystick *joy;
  int flag, min, ctr, max, ctrwth, endwth;
{
  double d1, d2, val;
  register int loop;

  /* half the center flat spot on each side of ctr */
  ctrwth >>= 1;

  /* fill in the lookup tables */
  /* no joystick attached is defined as centered */
#define DOTABLE(minv,ctrv,maxv,nrma) \
  if (minv == maxv) \
    for (loop = 0; loop <= JS_TIMEOUT; ++loop) \
      nrma[loop] = ctr; \
  else { \
    minv += endwth; maxv -= endwth; \
    d1 = 1.0 * (ctr - min) / (ctrv - ctrwth - minv); \
    d2 = 1.0 * (max - ctr) / (maxv - ctrwth - ctrv); \
    nrma[0] = nrma[JS_TIMEOUT] = ctr; \
    for (loop = 1; loop <= minv; ++loop) \
      nrma[loop] = min; \
    for (val = min + d1; loop < ctrv - ctrwth; ++loop, val += d1) \
      nrma[loop] = (int) val; \
    for (; loop <= ctrv + ctrwth; ++loop) \
      nrma[loop] = ctr; \
    for (val = ctr + d2; loop < maxv; ++loop, val += d2) \
      nrma[loop] = (int) val; \
    for (; loop < JS_TIMEOUT ; ++loop) \
      nrma[loop] = max; \
  }

  if (flag & 0x01) {
    DOTABLE(joy->js_ax_min,joy->js_ax_ctr,
      joy->js_ax_max,joy->js_ax_nrm);
    }
  if (flag & 0x02) {
    DOTABLE(joy->js_ay_min,joy->js_ay_ctr,
      joy->js_ay_max,joy->js_ay_nrm);
    }
  if (flag & 0x04) {
    DOTABLE(joy->js_bx_min,joy->js_bx_ctr,
      joy->js_bx_max,joy->js_bx_nrm);
    }
  if (flag & 0x08) {
    DOTABLE(joy->js_by_min,joy->js_by_ctr,
      joy->js_by_max,joy->js_by_nrm);
    }
}
