/*
    Function: BCD2Bin
    Description:
        Module local function used in RestoreDOSTime to convert the
        BCD number format returned from the real-time clock to
        binary.  Argument is a single byte.
*/
static int BCD2Bin(UINT8 Arg)
{
    return (((Arg >> 4) & 0xF) * 10) + (Arg & 0xF);
}

/*
    Function: RestoreDOSTime
    Description:
        Restores the DOS time-of-day counter from the real-time clock.
        This alleviates the burden on the programmer to have to call
        the system timer vector (the old INT 8 vector) 18.2 times per
        second when the vector has been hooked for some reason.  This
        is particularly useful if the system timer rate has been
        reprogrammed to a non-integer multiple of the 18.2 Hz rate.
*/
void RestoreDOSTime(void)
{
    union REGS Regs;
    UINT8 Century;
    UINT8 Year;
    UINT8 Month;
    UINT8 Day;
    UINT8 Hours;
    UINT8 Mins;
    UINT8 Secs;

    /* NOTE: be sure to read both date and time from the RTC before */
    /*   setting the DOS date or time.  DOS will set the RTC date */
    /*   AND time whenever either DOS set routine is called (from
    /*   its stored copies).  This will temporarily corrupt the RTC */
    /*   date or time (whichever is not being set) so we have to make */
    /*   a copy for ourselves first so we can set the second one */
    /*   from the correct information. */

    /* get RTC date */
    Regs.h.ah = READ_RTC_DATE;
    int86(CLOCK_INT, &Regs, &Regs);

    Century = Regs.h.ch;
    Year    = Regs.h.cl;
    Month   = Regs.h.dh;
    Day     = Regs.h.dl;

    /* get RTC time */
    Regs.h.ah = READ_RTC_TIME;
    int86(CLOCK_INT, &Regs, &Regs);

    Hours   = Regs.h.ch;
    Mins    = Regs.h.cl;
    Secs    = Regs.h.dh;

    /* set DOS date */
    Regs.x.cx = BCD2Bin(Century) * 100 + BCD2Bin(Year);
    Regs.h.dh = BCD2Bin(Month);
    Regs.h.dl = BCD2Bin(Day);
    Regs.h.ah = SET_DOS_DATE;
    int86(DOS_INT, &Regs, &Regs);

    /* set DOS time */
    Regs.h.ch = BCD2Bin(Hours);
    Regs.h.cl = BCD2Bin(Mins);
    Regs.h.dh = BCD2Bin(Secs);
    Regs.h.dl = 0; /* hunredths */
    Regs.h.ah = SET_DOS_TIME;
    int86(DOS_INT, &Regs, &Regs);
}

