/* File: SETMODEX.C
** Description:
**   Routines for putting the VGA into Mode X (and Y).
** Copyright:
**   Copyright 1994, David G. Roberts
** Acknowledgements:
**   Table driven method inspired by source from Themie Gouthas.
*/

#include <assert.h>
#include <dos.h>
#include <stdio.h>
#include "gamedefs.h"
#include "setmodex.h"
#include "vga.h"

/* constants */

static UINT8 ModeXParams[] = {
	VERT_TOTAL_INDEX,			0x0D,
    OVERFLOW_INDEX,				0x3E,
    MAX_SCAN_LINE_INDEX,		0x41,
    VERT_RETRACE_START_INDEX,	0xEA,
    VERT_RETRACE_END_INDEX,		0xAC,
    VERT_DISPLAY_END_INDEX,		0xDF,
    UNDERLINE_LOCATION_INDEX,	0x00,
    START_VERT_BLANK_INDEX,		0xE7,
    END_VERT_BLANK_INDEX,		0x06,
    MODE_CONTROL_INDEX,			0xE3
};

static UINT8 ModeYParams[] = {
	UNDERLINE_LOCATION_INDEX,	0x00,
    MODE_CONTROL_INDEX,			0xE3
};

/*
	Function: SetUndocMode
    Description:
    	Put the VGA into an undocumented mode.
*/
void SetUndocMode(int NumParams, UINT8 ModeParams[], UINT8 DotClock)
{
	union REGS regs;
    UINT8 VertRetraceEnd;
    unsigned i;
    UINT8 Index;
    UINT8 Data;
    UINT8 far * GraphicsBase;

    assert(ModeParams != NULL);

	/* set Mode 13h using VGA BIOS */
    regs.x.ax = 0x13;
    int86(VGA_BIOS_INT, &regs, &regs);

    /* disable Chain 4 */
    outportb(SEQ_INDEX_REG, MEMORY_MODE_INDEX);
    outportb(SEQ_DATA_REG, 0x06);

    /* enable synchronous reset */
    /* note: this avoids problems while twiddling MISC_OUTPUT_REG */
    outportb(SEQ_INDEX_REG, RESET_INDEX);
    outportb(SEQ_DATA_REG, 0x01);

    /* select 25 MHz dot clock and 60 Hz scan rate */
    if (DotClock != 0) {
		outportb(MISC_OUTPUT_REG, DotClock);
    }

    /* undo reset */
    outportb(SEQ_INDEX_REG, RESET_INDEX);
    outportb(SEQ_DATA_REG, 0x03);

    /* remove write protect on CRTC registers */
    outportb(CRTC_INDEX_REG, VERT_RETRACE_END_INDEX);
    VertRetraceEnd = inportb(CRTC_DATA_REG);
    VertRetraceEnd &= 0x7F;	/* clear high bit */
    outportb(CRTC_DATA_REG, VertRetraceEnd);


    /* alter CRTC registers */
    for (i = 0; i < NumParams; i++) {
    	Index = ModeParams[2 * i];
        Data = ModeParams[(2 * i) + 1];
        outportb(CRTC_INDEX_REG, Index);
        outportb(CRTC_DATA_REG, Data);
    }

    /* clear display memory to zero */
    outportb(SEQ_INDEX_REG, MAP_MASK_INDEX);
    outportb(SEQ_DATA_REG, 0xF);	/* enable all planes */
    GraphicsBase = (UINT8 far *) MK_FP(VIDEO_MEM_SEGMENT, 0);
    for (i = 0; i != 0xFFFF; i++) { /* note i must be unsigned */
    	GraphicsBase[i] = 0;
    }
    GraphicsBase[0xFFFF] = 0;
}

/*
	Function: SetModeX
    Description:
    	Puts the VGA into Mode X.
*/
void SetModeX(void)
{
	SetUndocMode(DIM(ModeXParams) / 2, ModeXParams, 0xE3);
	GScreenWidth = MODEX_WIDTH;
	GScreenHeight = MODEX_HEIGHT;
	GScreenVirtualWidth = MODEX_WIDTH;
	GVGAMode = MODE_X;
}

/*
	Function: SetModeY
    Description:
    	Puts the VGA into Mode Y.
*/
void SetModeY(void)
{
	SetUndocMode(DIM(ModeYParams) / 2, ModeYParams, 0);
	GScreenWidth = MODEY_WIDTH;
	GScreenHeight = MODEY_HEIGHT;
	GScreenVirtualWidth = MODEY_WIDTH;
	GVGAMode = MODE_X;
}
