/* File: RSYNCDMO.C
** Description:
**   Demonstrates synchronization of the system timer with the VGA
**   retrace.  This is a modification of the FLIPDEMO program
**   that runs at 30 fps no matter how few balls are drawn.
** Copyright:
**   Copyright 1994, David G. Roberts
*/

#include <alloc.h>
#include <assert.h>
#include <conio.h>
#include <stdio.h>
#include <stdlib.h>
#include "gamedefs.h"
#include "animate.h"
#include "bitblt.h"
#include "retrace.h"
#include "setmodex.h"
#include "timer.h"
#include "vga.h"
#include "vrsync.h"

UINT8 BallBitmap[] = {
	10, 0,			/* Width (little endian) */
    10, 0,			/* Height (little endian) */
    0, 0,			/* X and Y origin */
    0, 0,
    0, 0, 0, 2, 2, 2, 2, 0, 0, 0, /* color 2 = green w/ default palette */
    0, 2, 2, 2, 2, 2, 2, 2, 2, 0,
    0, 2, 2, 0, 0, 2, 2, 2, 2, 0,
    2, 2, 0, 0, 2, 2, 2, 2, 2, 2,
    2, 2, 0, 2, 2, 2, 2, 2, 2, 2,
    2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
    2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
    0, 2, 2, 2, 2, 2, 2, 2, 2, 0,
    0, 2, 2, 2, 2, 2, 2, 2, 2, 0,
    0, 0, 0, 2, 2, 2, 2, 0, 0, 0
};

UINT8 BallEraseBitmap[] = {
	10,  0,			/* Width (little endian) */
    10,  0,			/* Height (little endian) */
    0, 0,			/* X and Y origin */
    0, 0,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16, /* color 16 = black */
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16, /* with default palette */
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16,
    16, 16, 16, 16, 16, 16, 16, 16, 16, 16
};

/* globals used in RetraceInt as well as main function */
int RetraceCount;
int HiddenPage;
UINT16 Offset[2];
BOOL FrameReady;

/* note: make these track width and height of bitmap, above */
#define BITMAP_HEIGHT	(10)
#define BITMAP_WIDTH	(10)

#define PAGE0_OFFSET	(0)
#define PAGE1_OFFSET	(0x4B00)

#define NUM_BALLS		(5)

#define FRAME_SKIP		1	/* 0 = 60 fps, 1 = 30 fps, 2 = 20 fps */

/*
	Function: RetraceProc
    Description:
    	Replacement system timer interrupt called after the timer
        is sync'd with the VGA vertical retrace.
*/
void far interrupt RetraceProc(void)
{
	asm sti;	/* reenable other interrupts */

    if (RetraceCount == 0) {
        if (FrameReady) {
	        /* flip page */
        	PageFlip(Offset[HiddenPage]);
        	RetraceTimerReinit();
        	HiddenPage ^= 1; /* flip HiddenPage to other state */
        	RetraceCount = FRAME_SKIP;
            FrameReady = FALSE;
        }
        else {
        	/* we weren't quite ready */
        	WaitVerticalRetraceStart();
            RetraceTimerReinit();
            RetraceCount = 0;
        }
    }
    else {
    	/* simply do this again */
	    RetraceCount--;
	    WaitVerticalRetraceStart();
        RetraceTimerReinit();
    }

	/* acknowledge the int to the PIC */
    outportb(PIC, NONSPECIFIC_EOI);
}

int main()
{
	int x[NUM_BALLS], y[NUM_BALLS];
    int OldX[NUM_BALLS][2], OldY[NUM_BALLS][2];
    int vx[NUM_BALLS], vy[NUM_BALLS];
    PLANAR_BITMAP far * PlanarBallBitmap;
    PLANAR_BITMAP far * PlanarBallEraseBitmap;
    BOOL Erase[NUM_BALLS][2];
    int i;

    /* detect VGA */
    if (!DetectVGA()) {
    	printf("You must have a VGA to run this program.\n");
        return 1;
    }

    Offset[0] = PAGE0_OFFSET;
    Offset[1] = PAGE1_OFFSET;
    HiddenPage = 1;

    PlanarBallBitmap = LinearToPlanar((LINEAR_BITMAP far *) BallBitmap);
    PlanarBallEraseBitmap =
		LinearToPlanar((LINEAR_BITMAP far *) BallEraseBitmap);

    /* set Mode X and display page 0 */
    SetModeX();

    for (i = 0; i < NUM_BALLS; i++) {
    	x[i] = random(MODEX_WIDTH - BITMAP_WIDTH);
    	y[i] = random(MODEX_HEIGHT - BITMAP_HEIGHT);
    	vx[i] = random(2) + 1;
    	vy[i] = random(2) + 1;
    	Erase[i][0] = Erase[i][1] = FALSE;
    }

    /* start up synchronized interrupt */
    FrameReady = FALSE;
    RetraceCount = FRAME_SKIP;
    RetraceTimerInit(RetraceProc);

    while (!kbhit()) {
        /* update new position */
        for (i = 0; i < NUM_BALLS; i++) {
        	if (Erase[i][HiddenPage]) {
        		x[i] += vx[i];
        		y[i] += vy[i];
        		if ((x[i] < 0) || ((x[i] + BITMAP_WIDTH) >= MODEX_WIDTH)) {
            		vx[i] = -vx[i];
            		x[i] += 2 * vx[i];
        		}
        		if ((y[i] < 0) || ((y[i] + BITMAP_HEIGHT) >= MODEX_HEIGHT)) {
            		vy[i] = -vy[i];
            		y[i] += 2 * vy[i];
        		}
        	}
        }

        /* stall until retrace procedure flips page */
        while (FrameReady);

        /* erase */
        for (i = 0; i < NUM_BALLS; i++) {
        	if (Erase[i][HiddenPage]) {
    			BltPlanar(PlanarBallEraseBitmap,
					OldX[i][HiddenPage], OldY[i][HiddenPage],
					Offset[HiddenPage]);
            }
        }

    	/* draw */
        for (i = 0; i < NUM_BALLS; i++) {
    		BltPlanar(PlanarBallBitmap, x[i], y[i], Offset[HiddenPage]);
        	OldX[i][HiddenPage] = x[i];
        	OldY[i][HiddenPage] = y[i];
        	Erase[i][HiddenPage] = TRUE;
        }

        /* flip page */
        FrameReady = TRUE;
    }

    RetraceTimerStop();

    getch();

    farfree(PlanarBallBitmap);
    farfree(PlanarBallEraseBitmap);

    SetVGAMode(0x3);

    return 0;
}
