/* File: RETRACE.C
** Description:
**   Routines used to look for vertical retrace.
** Author:
**   David G. Roberts
** Copyright:
**   Copyright 1994, David G. Roberts
*/

#include <assert.h>
#include <dos.h>
#include <stdio.h>
#include "gamedefs.h"
#include "retrace.h"
#include "vga.h"

/* constants */
#define VR_BIT (0x08)
#define DISPLAY_ENABLE (0x01)

/*
	Function: WaitVerticalRetraceStart
    Description:
    	Waits for the VGA to begin vertical retrace.  Note that if the
        VGA is already in vertical retrace, the routine will wait all
        during the next display frame until vertical retrace begins
        again.
*/
void WaitVerticalRetraceStart(void)
{
	UINT8 InputStatus1;

    /* make sure we aren't in retrace already */
    do {
    	InputStatus1 = inportb(INPUT_STAT_1_REG);
    } while (InputStatus1 & VR_BIT);

    /* wait for retrace to start */
    do {
    	InputStatus1 = inportb(INPUT_STAT_1_REG);
    } while (!(InputStatus1 & VR_BIT));
}

/*
	Function: WaitVerticalRetraceEnd
    Description:
    	Waits for vertical retrace to complete.  If the VGA is not
        in vertical retrace, it waits until the VGA enters vertical
        retrace again and the retrace subsequently completes.
*/
void WaitVerticalRetraceEnd(void)
{
	UINT8 InputStatus1;

    /* make sure we aren't in display state (= not retrace) */
    do {
    	InputStatus1 = inportb(INPUT_STAT_1_REG);
    } while (!(InputStatus1 & VR_BIT));

    /* wait for retrace to end */
    do {
    	InputStatus1 = inportb(INPUT_STAT_1_REG);
    } while (InputStatus1 & VR_BIT);
}

/*
	Function: WaitDisplayMode
    Description:
    	If the VGA is currently in display mode, the function returns
        immediately, else it waits until diplay mode is entered.
*/
void WaitDisplayMode(void)
{
	UINT8 InputStatus1;

    /* wait for retrace to end and display mode to start */
    do {
    	InputStatus1 = inportb(INPUT_STAT_1_REG);
    } while (InputStatus1 & DISPLAY_ENABLE);
}

/*
	Function: WaitRetraceMode
    Description:
    	If the VGA is currently in retrace mode, the function returns
        immediately, else it waits until retrace mode is entered.
*/
void WaitRetraceMode(void)
{
	UINT8 InputStatus1;

    /* wait for display mode to end and retrace to start */
    do {
    	InputStatus1 = inportb(INPUT_STAT_1_REG);
    } while (!(InputStatus1 & VR_BIT));
}

/*
	Function: VerticalRetraceOccuring
    Description:
    	Returns the current state of the VGA vertical retrace.  If
        retrace is occuring, the function returns TRUE.
*/
BOOL VerticalRetraceOccuring(void)
{
	return (inportb(INPUT_STAT_1_REG) & VR_BIT) >> 3;
}
