/* File: BITBLT.C
** Description:
**   Module for bitblt'ing bitmaps to the screen.
** Copyright:
**   Copyright 1994, David G. Roberts
*/

#include <alloc.h>
#include <assert.h>
#include <dos.h>
#include <mem.h>
#include <stdio.h>
#include "gamedefs.h"
#include "vga.h"
#include "bitblt.h"

/*
	Function: BltLinear
    Description:
    	Draws a linear bitmap to the screen.
        The base of the screen is specified by the ScreenBase
        parameter and the position on that screen specified by the
        x and y parameters.
*/
void BltLinear
	(
	LINEAR_BITMAP far * LinearBM,
	int x,
	int y,
	UINT8 far * ScreenBase
	)
{
	int Top;	/* coordinate values of bitmap top-left corner */
    int Left;
    int BltWidth;	/* width of bitmap so we don't dereference pointers */
    int BltHeight;  /* height of bitmap so we don't dereference pointers */
    UINT16 TempOffset;	/* temp variable to calc far pointer offsets */
    UINT8 far * Screen; /* pointer to current screen position */
    UINT8 far * Bitmap; /* pointer to current bitmap position */
    /* unsigned WidthCounter; used for C version */
    /* unsigned HeightCounter; used for C version */
    unsigned ScreenIncrement;

    assert(LinearBM != NULL);
    assert(ScreenBase != NULL);

    /* calculate top-left corner position */
    Left = x - LinearBM->OriginX;
    Top  = y - LinearBM->OriginY;

    /* calculate screen pointer starting position */
    TempOffset = Top * GScreenVirtualWidth + Left;
    Screen = ScreenBase + TempOffset;

    /* calculate bitmap pointer starting position */
    Bitmap = &(LinearBM->Data);

    /* blt to screen */
    BltWidth = LinearBM->Width;
    BltHeight = LinearBM->Height;
    ScreenIncrement = GScreenVirtualWidth - BltWidth;

    /*
    This bit of C code is equivalent to the inline assembly
    language, below, and is shown here for reference.

    for (HeightCounter = 0; HeightCounter < BltHeight; HeightCounter++) {
    	for (WidthCounter = 0; WidthCounter < BltWidth; WidthCounter++) {
        	if (*Bitmap != 0) {
				*Screen = *Bitmap;
            }
            Screen++;
            Bitmap++;
        }
        Screen += ScreenIncrement;
    }
    */
    asm {
    	push ds
        mov dx,BltWidth
        mov bx,BltHeight
        lds si,Bitmap
        les di,Screen
    }
    rowloop:
    asm {
    	mov cx,dx
    }
    columnloop:
    asm {
        mov al,[si]
        inc si
        or al,al
        jz transparent
        mov es:[di],al
        inc di
        dec cx
        jnz columnloop
        add di,ScreenIncrement
        dec bx
        jnz rowloop
        jmp cleanup
    }
    transparent:
    asm {
    	inc di
        dec cx
        jnz columnloop
        add di,ScreenIncrement
        dec bx
        jnz rowloop
    }
    cleanup:
    asm {
        pop ds
    }
}

/*
	Function: BltLinearNoTransparent
    Description:
    	Draws a linear bitmap to the screen. Ignores transparent
        pixels.
*/
void BltLinearNoTransparent
	(
	LINEAR_BITMAP far * LinearBM,
	int x,
	int y,
	UINT8 far * ScreenBase
	)
{
	int Top;	/* coordinate values of bitmap top-left corner */
    int Left;
    int BltWidth;	/* width of bitmap so we don't dereference pointers */
    int BltHeight;  /* height of bitmap so we don't dereference pointers */
    UINT16 TempOffset;	/* temp variable to calc far pointer offsets */
    UINT8 far * Screen; /* pointer to current screen position */
    UINT8 far * Bitmap; /* pointer to current bitmap position */
    /* unsigned WidthCounter; used for C version */
    /* unsigned HeightCounter; used for C version */
    unsigned ScreenIncrement;

    assert(LinearBM != NULL);
    assert(ScreenBase != NULL);

    /* calculate top-left corner position */
    Left = x - LinearBM->OriginX;
    Top  = y - LinearBM->OriginY;

    /* calculate screen pointer starting position */
    TempOffset = Top * GScreenVirtualWidth + Left;
    Screen = ScreenBase + TempOffset;

    /* calculate bitmap pointer starting position */
    Bitmap = &(LinearBM->Data);

    /* blt to screen */
    BltWidth = LinearBM->Width;
    BltHeight = LinearBM->Height;
    ScreenIncrement = GScreenVirtualWidth - BltWidth;

    /*
    This bit of C code is equivalent to the inline assembly
    language, below, and is shown here for reference.

    for (HeightCounter = 0; HeightCounter < BltHeight; HeightCounter++) {
    	for (WidthCounter = 0; WidthCounter < BltWidth; WidthCounter++) {
			*Screen++ = *Bitmap++;
        }
        Screen += ScreenIncrement;
    }
    */
    asm {
    	push ds
        mov dx,BltWidth
        mov bx,BltHeight
        lds si,Bitmap
        les di,Screen
    }
    rowloop:
    asm {
    	mov cx,dx

        shr cx,1
        rep movsw
        adc cx,cx
        rep movsb

        add di,ScreenIncrement
        dec bx
        jnz rowloop
        pop ds
    }
}

/*
	Function: LinearToPlanar
    Description:
    	Converts a bitmap from linear format to planar format.
        The function returns a new block of memory allocated with
        farmalloc which should be farfreed when the program is finished
        with it.  Returns NULL if memory could not be allocated.  If
        the source bitmap width is not evenly divisible by four,
        the function rounds the planar bitmap width up to an even
        multiple of four and fills the new bitmap columns with 0 pixels.
*/
PLANAR_BITMAP far * LinearToPlanar(LINEAR_BITMAP far * LinearBM)
{
	unsigned PlanarWidth;
    UINT16 PlanarBMSize;
    PLANAR_BITMAP far * PlanarBM;
    unsigned PlaneCounter;
    unsigned WidthCounter;
    unsigned HeightCounter;
    UINT8 far * LinearData;
    UINT8 far * PlanarData;

    assert(LinearBM != NULL);

    /* make PlanarWidth even multiple of 4, rounding up */
    if ((LinearBM->Width % 4) != 0) {
    	PlanarWidth = LinearBM->Width + (4 - (LinearBM->Width % 4));
    }
    else {
    	PlanarWidth = LinearBM->Width;
    }

    /* calculate needed memory and allocate */
    /* sizeof(UINT8) correct for the dummy Data field of the */
    /* PLANAR_BITMAP structure which is already included in the */
    /* width*height term */
    PlanarBMSize = PlanarWidth * LinearBM->Height +
		sizeof(PLANAR_BITMAP) - sizeof(UINT8);
    PlanarBM = (PLANAR_BITMAP far *) farmalloc(PlanarBMSize);
    if (PlanarBM == NULL) {	/* error! */
    	return NULL;
    }

    /* fill in width and height info */
    PlanarBM->Width = PlanarWidth / 4;
    PlanarBM->Height = LinearBM->Height;
    PlanarBM->OriginX = LinearBM->OriginX;
    PlanarBM->OriginY = LinearBM->OriginY;

    /* store data for each plane consecutively */
    PlanarData = &(PlanarBM->Data);
    for (PlaneCounter = 0; PlaneCounter < 4; PlaneCounter++) {

        for (HeightCounter = 0; HeightCounter < PlanarBM->Height;
			HeightCounter++) {

            /* reset LinearData pointer to start of next row */
	    	LinearData = &(LinearBM->Data) +
				HeightCounter * LinearBM->Width + PlaneCounter;

            /* iterate over row, stepping by four */
            for (WidthCounter = PlaneCounter; WidthCounter < PlanarWidth;
            	WidthCounter += 4) {

                if (WidthCounter < LinearBM->Width) {
                	*PlanarData = *LinearData;
	                LinearData += 4;
                }
                else {
					/* fill 0's to create new rows */
                	*PlanarData = 0;
                }
                PlanarData++;
            }
        }
    }

    return PlanarBM;
}

/*
	Function: BltPlanar
    Description:
    	Draws a planar bitmap to a mode X/Y screen.  The bitmap
        is positioned at (x,y) and is drawn on the mode X page
        starting at the address specified by the PageOffset parameter.
*/
void BltPlanar
	(
	PLANAR_BITMAP far * PlanarBM,
	int x,
	int y,
	UINT16 PageOffset
	)
{
	int Top;	/* coordinate values of bitmap top-left corner */
    int Left;
    unsigned BltWidth;	/* width of clipped bitmap */
    unsigned BltHeight; /* height of clipped bitmap */
    UINT16 TempOffset;	/* temp variable to calc far pointer offsets */
    /* UINT8 far * Screen; ptr to screen position - used in C version */
    UINT8 far * ScreenInit; /* used to reload screen ptr between planes */
    UINT8 far * Bitmap; /* pointer to current bitmap position */
    /* unsigned WidthCounter; used for C version */
    /* unsigned HeightCounter; used for C version */
    unsigned PlaneCounter;
    unsigned ScreenIncrement;
    UINT8 MMRValue;

    assert(PlanarBM != NULL);

    /* calculate top-left corner position */
    Left = x - PlanarBM->OriginX;
    Top  = y - PlanarBM->OriginY;

    /* setup */
    BltWidth = PlanarBM->Width;
    BltHeight = PlanarBM->Height;
    ScreenIncrement = GScreenVirtualWidth / 4 - BltWidth;
    Bitmap = &(PlanarBM->Data);
    if (Left >= 0) {
    	MMRValue = 1 << (Left % 4);
    }
    else {
    	/* correct for when bitmap moves offscreen to left */
    	MMRValue = 1 << ((4 - (-Left % 4)) % 4);
    }
    TempOffset = (Top * (GScreenVirtualWidth / 4)) + PageOffset;
    if (Left >= 0) {
    	TempOffset += (Left / 4);
    }
    else {
    	/* correct for when bitmap moves offscreen to left */
    	TempOffset -= ((-Left - 1) / 4) + 1;
    }
    ScreenInit = MK_FP(VIDEO_MEM_SEGMENT, TempOffset);

    /*
    This bit of C code is equivalent to the inline assembly
    language, below, and is shown here for reference.

    for (PlaneCounter = 0; PlaneCounter < 4; PlaneCounter++) {
        SetMMR(MMRValue);
	    Screen = ScreenInit;

    	for (HeightCounter = 0; HeightCounter < BltHeight;
			HeightCounter++) {
    		for (WidthCounter = 0; WidthCounter < BltWidth;
				WidthCounter++) {
        		if (*Bitmap != 0) {
					*Screen = *Bitmap;
            	}
            	Screen++;
            	Bitmap++;
        	}
        	Screen += ScreenIncrement;
    	}

        MMRValue <<= 1;
        if (!(MMRValue & 0x0F)) {
        	MMRValue = 1;
            ScreenInit++;
        }
    }
    */

    /* bitblt */
    asm {
    	push ds
    	mov PlaneCounter,4
        lds si,Bitmap
    }
    planeloop:
    asm {
        mov ah,MMRValue
        mov al,MAP_MASK_INDEX
        mov dx,SEQ_INDEX_REG
        out dx,ax
        mov dx,BltWidth
        mov bx,BltHeight
        les di,ScreenInit
    }
    rowloop:
    asm {
    	mov cx,dx
    }
    columnloop:
    asm {
        mov al,[si]
        inc si
        or al,al
        jz transparent
    }
    nontransparent:
    asm {
        mov es:[di],al
        inc di
        dec cx
        jnz columnloop
    }
    lineend:
    asm {
        add di,ScreenIncrement
        dec bx
        jnz rowloop
        jmp cleanup
    }
    transparent:
    asm {
    	inc di
        dec cx
        jz lineend
		mov al,[si]
        inc si
        or al,al
        jnz nontransparent
        jmp transparent
    }
    cleanup:
    asm {
    	dec PlaneCounter
        jz exit
    	mov al,MMRValue
        shl al,1
        and al,0xF
        jnz noreinit
        mov al,1
        inc word ptr ScreenInit
    }
    noreinit:
    asm {
    	mov MMRValue,al
        jmp planeloop
    }
    exit:
    asm {
        pop ds
    }

}

/*
	Function: BltPlanarNoTransparent
    Description:
    	Draws a planar bitmap to a mode X/Y screen.  The routine
        ignores transparent pixels.
*/
void BltPlanarNoTransparent
	(
	PLANAR_BITMAP far * PlanarBM,
	int x,
	int y,
	UINT16 PageOffset
	)
{
	int Top;	/* coordinate values of bitmap top-left corner */
    int Left;
    unsigned BltWidth;	/* width of clipped bitmap */
    unsigned BltHeight; /* height of clipped bitmap */
    UINT16 TempOffset;	/* temp variable to calc far pointer offsets */
    /* UINT8 far * Screen; ptr to screen position - used in C version */
    UINT8 far * ScreenInit; /* used to reload screen ptr between planes */
    UINT8 far * Bitmap; /* pointer to current bitmap position */
    /* unsigned WidthCounter; used for C version */
    /* unsigned HeightCounter; used for C version */
    unsigned PlaneCounter;
    unsigned ScreenIncrement;
    UINT8 MMRValue;

    assert(PlanarBM != NULL);

    /* calculate top-left corner position */
    Left = x - PlanarBM->OriginX;
    Top  = y - PlanarBM->OriginY;

    /* setup */
    BltWidth = PlanarBM->Width;
    BltHeight = PlanarBM->Height;
    ScreenIncrement = GScreenVirtualWidth / 4 - BltWidth;
    Bitmap = &(PlanarBM->Data);
    if (Left >= 0) {
    	MMRValue = 1 << (Left % 4);
    }
    else {
    	/* correct for when bitmap moves offscreen to left */
    	MMRValue = 1 << ((4 - (-Left % 4)) % 4);
    }
    TempOffset = (Top * (GScreenVirtualWidth / 4)) + PageOffset;
    if (Left >= 0) {
    	TempOffset += (Left / 4);
    }
    else {
    	/* correct for when bitmap moves offscreen to left */
    	TempOffset -= ((-Left - 1) / 4) + 1;
    }
    ScreenInit = MK_FP(VIDEO_MEM_SEGMENT, TempOffset);

    /*
    This bit of C code is equivalent to the inline assembly
    language, below, and is shown here for reference.

    for (PlaneCounter = 0; PlaneCounter < 4; PlaneCounter++) {
        SetMMR(MMRValue);
	    Screen = ScreenInit;

    	for (HeightCounter = 0; HeightCounter < BltHeight;
			HeightCounter++) {
    		for (WidthCounter = 0; WidthCounter < BltWidth;
				WidthCounter++) {
				*Screen = *Bitmap;
            	Screen++;
            	Bitmap++;
        	}
        	Screen += ScreenIncrement;
    	}

        MMRValue <<= 1;
        if (!(MMRValue & 0x0F)) {
        	MMRValue = 1;
            ScreenInit++;
        }
    }
    */

    /* bitblt */
    asm {
    	push ds
    	mov PlaneCounter,4
        lds si,Bitmap
    }
    planeloop:
    asm {
        mov ah,MMRValue
        mov al,MAP_MASK_INDEX
        mov dx,SEQ_INDEX_REG
        out dx,ax
        mov dx,BltWidth
        mov bx,BltHeight
        les di,ScreenInit
    }
    rowloop:
    asm {
    	mov cx,dx
        rep movsb
        add di,ScreenIncrement
        dec bx
        jnz rowloop
    	dec PlaneCounter
        jz exit
    	mov al,MMRValue
        shl al,1
        and al,0xF
        jnz noreinit
        mov al,1
        inc word ptr ScreenInit
    }
    noreinit:
    asm {
    	mov MMRValue,al
        jmp planeloop
    }
    exit:
    asm {
        pop ds
    }
}

/*
	Function: LinearToVideoMemory
    Description:
    	Converts a linear bitmap into a video memory bitmap.  The first
        parameter points to the original linear bitmap structure while
        the second points to the address in video memory that should be
        used to hold the video bitmap.  The function returns NULL if
        no video bitmap data structure could be allocated and a
        far pointer to the structure otherwise.
*/
VIDEO_MEM_BITMAP far * LinearToVideoMemory
	(
	LINEAR_BITMAP far * LinearBM,
	UINT16 Storage,
    UINT16 * Length
	)
{
	unsigned VideoWidth;	/* width of video memory bitmap */
    unsigned VideoBMSize;	/* size of video BM structure + map mask data */
    VIDEO_MEM_BITMAP far * VideoBM;
    unsigned WidthCounter;	/* width counter for video memory */
    unsigned LinearWidthCounter; /* width counter for linear bitmap */
    unsigned HeightCounter;
    unsigned PlaneCounter;
    UINT8 far * LinearData;
    UINT8 far * VideoData;
    UINT8 far * MapMaskData;
    UINT8 Pixel;


    assert(LinearBM != NULL);
    assert(Storage != NULL);

    /* make PlanarWidth even multiple of 4, rounding up */
    if ((LinearBM->Width % 4) != 0) {
    	VideoWidth = LinearBM->Width + (4 - (LinearBM->Width % 4));
    }
    else {
    	VideoWidth = LinearBM->Width;
    }

    /* calculate length of video memory bitmap data structure */
    VideoBMSize = (VideoWidth / 4) * LinearBM->Height +
		sizeof(VIDEO_MEM_BITMAP) - 1;

    /* allocate video memory bitmap data structure */
    VideoBM = (VIDEO_MEM_BITMAP far *) farmalloc(VideoBMSize);
    if (VideoBM == NULL) { /* error! */
    	return NULL;
    }

    /* fill in fields */
    VideoBM->Width = VideoWidth / 4;
    VideoBM->Height = LinearBM->Height;
    VideoBM->OriginX = LinearBM->OriginX;
    VideoBM->OriginY = LinearBM->OriginY;
    VideoBM->VideoDataOffset = Storage;

    /* copy linear data to video memory */
    LinearData = &(LinearBM->Data);
    VideoData = MK_FP(VIDEO_MEM_SEGMENT, Storage);
    MapMaskData = &(VideoBM->MapMaskData);
    for (HeightCounter = 0; HeightCounter < LinearBM->Height;
		HeightCounter++) {
        LinearWidthCounter = 0;
		for (WidthCounter = 0; WidthCounter < (VideoWidth / 4);
        	WidthCounter++) {
            *MapMaskData = 0;
            for (PlaneCounter = 0; PlaneCounter < 4; PlaneCounter++) {
            	if (LinearWidthCounter < LinearBM->Width) {
                	Pixel = *LinearData;
                    LinearData++;
                }
                else {
                	Pixel = 0;	/* transparent pixel */
                }
                SetMMR(1 << PlaneCounter);
                *VideoData = Pixel;
                if (Pixel != 0) {
                	*MapMaskData |= 1 << PlaneCounter;
                }
                LinearWidthCounter++;
            }
            MapMaskData++;
            VideoData++;
        }
    }

    if (Length != NULL) {
    	*Length = VideoBM->Width * VideoBM->Height;
    }

    return VideoBM;
}

/*
	Function: BltVideoMem
    Description:
    	Performs a bitblt of a video memory bitmap to the screen.
        This routines does NOT perform clipping.  The X coordinate
        given to the routine should be a multiple of four.  If it
        is not, the routine rounds it down to the nearest multiple
        of four.
*/
void BltVideoMem
	(
	VIDEO_MEM_BITMAP far * VideoBM,
	int x,
	int y,
	UINT16 PageOffset
	)
{
	int Top;	/* coordinate values of bitmap top-left corner */
    int Left;
	UINT8 MMRCache;
    UINT8 far * MapMaskData;
    UINT8 far * Source;
    /* UINT8 far * Dest; needed in C version */
    unsigned DestOffset;
    /* unsigned WidthCounter; needed in C version */
    unsigned HeightCounter;
    unsigned BltWidth;
    unsigned BltHeight;
    UINT8 ModeRegTemp;
    /* UINT8 Dummy; used in version */
    unsigned DestIncrement;

    assert(VideoBM != NULL);

    Left = x - VideoBM->OriginX;
    Top  = y - VideoBM->OriginY;

    /* initialize the various pointers */
    Source = MK_FP(VIDEO_MEM_SEGMENT, VideoBM->VideoDataOffset);
    DestOffset = (Top * (GScreenVirtualWidth / 4)) +
		(Left / 4) + PageOffset;
    /* Dest = MK_FP(VIDEO_MEM_SEGMENT, DestOffset); needed in C version */
    MapMaskData = &(VideoBM->MapMaskData);

    /* initialize MMRCache with a value that cannot occur normally */
    /* (normal values are 0x00-0x0F) which will force a reload */
    MMRCache = 0xFF;

    /* set write mode 1 for fast copying using the display latches */
    outportb(GC_INDEX_REG, MODE_INDEX);
	ModeRegTemp = inportb(GC_DATA_REG);
    ModeRegTemp = (ModeRegTemp & 0xFC) | 0x01;
    outportb(GC_DATA_REG, ModeRegTemp);

    /* setup */
    DestIncrement = (GScreenVirtualWidth / 4) - VideoBM->Width;
    BltWidth = VideoBM->Width;
    BltHeight = VideoBM->Height;

    /*
    This bit of C code is equivalent to the inline assembly
    language, below, and is shown here for reference.

    for (HeightCounter = 0; HeightCounter < BltHeight;
    	HeightCounter++) {
        for (WidthCounter = 0; WidthCounter < BltWidth;
        	WidthCounter++) {
            // if current MMR value == last MMR value, don't do this
            if (MMRCache != *MapMaskData) {
            	MMRCache = *MapMaskData;
                SetMMR(MMRCache);
            }
            Dummy = *Source;	// since we're using write mode 1
            *Dest = Dummy;		//   the Dummy value is ignored
            Source++;
            Dest++;
            MapMaskData++;
        }
        Dest += DestIncrement;
    }
    */

    /* copy data */
    asm {
    	push ds
        lds si,Source		/* ds:si = source */
        mov di,DestOffset   /* ds:di = dest */
        les bx,MapMaskData	/* es:bx = map mask data */
        mov dx,BltWidth
        mov ax,BltHeight
        mov HeightCounter,ax
    }
    rowloop:
    asm {
    	mov cx,dx
    }
    lineloop:
    asm {
    	mov ah,es:[bx]		/* get map mask data */
        inc bx
        cmp ah,MMRCache
        je mmrok
        mov MMRCache,ah		/* if its not the same as last time */
        push dx             /* set it and update cache */
        mov dx,SEQ_INDEX_REG
        mov al,MAP_MASK_INDEX
        out dx,ax
        pop dx
    }
    mmrok:
    asm {
        mov al,[si]			/* dummy read */
        mov [di],al			/* dummy write */
        inc si
        inc di
        dec cx
        jnz lineloop
        add di,DestIncrement
        dec HeightCounter
        jnz rowloop
        pop ds
    }

    /* reset write mode back to write mode 0 */
    ModeRegTemp &= 0xFC;
    outportb(GC_INDEX_REG, MODE_INDEX);
    outportb(GC_DATA_REG, ModeRegTemp);
}

/*
	Function: BltVideoMemNoTransparent
    Description:
    	Performs a bitblt of a video memory bitmap to the screen.
        This routines does NOT perform clipping.  The X coordinate
        given to the routine should be a multiple of four.  If it
        is not, the routine rounds it down to the nearest multiple
        of four.  This routine ignores the MapMaskData field of
        the video memory bitmap structure.  Transparent pixels are
        ignored and drawn normally as color 0.
*/
void BltVideoMemNoTransparent
	(
	VIDEO_MEM_BITMAP far * VideoBM,
	int x,
	int y,
	UINT16 PageOffset
	)
{
	int Top;	/* coordinate values of bitmap top-left corner */
    int Left;
    UINT8 far * Source;
    /* UINT8 far * Dest; needed in C version */
    unsigned DestOffset;
    /* unsigned WidthCounter; needed in C version */
    /* unsigned HeightCounter; needed in C version */
    unsigned BltWidth;
    unsigned BltHeight;
    UINT8 ModeRegTemp;
    /* UINT8 Dummy; used in C version */
    unsigned DestIncrement;

    assert(VideoBM != NULL);

    Left = x - VideoBM->OriginX;
    Top  = y - VideoBM->OriginY;

    /* initialize the various pointers */
    Source = MK_FP(VIDEO_MEM_SEGMENT, VideoBM->VideoDataOffset);
    DestOffset = (Top * (GScreenVirtualWidth / 4)) +
		(Left / 4) + PageOffset;
    /* Dest = MK_FP(VIDEO_MEM_SEGMENT, DestOffset); needed in C version */

    /* set write mode 1 for fast copying using the display latches */
    outportb(GC_INDEX_REG, MODE_INDEX);
	ModeRegTemp = inportb(GC_DATA_REG);
    ModeRegTemp = (ModeRegTemp & 0xFC) | 0x01;
    outportb(GC_DATA_REG, ModeRegTemp);

    /* setup */
    DestIncrement = (GScreenVirtualWidth / 4) - VideoBM->Width;
    BltWidth = VideoBM->Width;
    BltHeight = VideoBM->Height;
    SetMMR(0x0F);

    /*
    This bit of C code is equivalent to the inline assembly
    language, below, and is shown here for reference.

    for (HeightCounter = 0; HeightCounter < BltHeight;
    	HeightCounter++) {
        for (WidthCounter = 0; WidthCounter < BltWidth;
        	WidthCounter++) {
            Dummy = *Source;	// since we're using write mode 1
            *Dest = Dummy;		//   the Dummy value is ignored
            Source++;
            Dest++;
            MapMaskData++;
        }
        Dest += DestIncrement;
    }
    */

    /* copy data */
    asm {
    	push ds
        lds si,Source		/* ds:si = source */
        mov di,DestOffset   /* ds:di = dest */
        mov bx,BltHeight
        mov dx,BltWidth
    }
    rowloop:
    asm {
    	mov cx,dx
    }
    lineloop:
    asm {
        mov al,[si]			/* dummy read */
        mov [di],al			/* dummy write */
        inc si
        inc di
        dec cx
        jnz lineloop
        add di,DestIncrement
        dec bx
        jnz rowloop
        pop ds
    }

    /* reset write mode back to write mode 0 */
    ModeRegTemp &= 0xFC;
    outportb(GC_INDEX_REG, MODE_INDEX);
    outportb(GC_DATA_REG, ModeRegTemp);
}


/*
	Function: BltLinearClipRect
    Description:
    	Draws a linear bitmap to the screen, clipping as appropriate.
        The base of the screen is specified by the ScreenBase
        parameter and the position on that screen specified by the
        x and y parameters.
*/
void BltLinearClipRect
	(
	LINEAR_BITMAP far * LinearBM,
	int x,
	int y,
	UINT8 far * ScreenBase,
    RECT * ClipRect
	)
{
	int Top;	/* coordinate values of bitmap top-left corner */
    int Left;
	unsigned BMOffsetX;	/* starting offset into clipped bitmap */
    unsigned BMOffsetY;
    unsigned ClippedLeft;/* top-left corner position of clipped bitmap */
    unsigned ClippedTop;
    int BltWidth;	/* width of clipped bitmap */
    int BltHeight; /* height of clipped bitmap */
    UINT16 TempOffset;	/* temp variable to calc far pointer offsets */
    UINT8 far * Screen; /* pointer to current screen position */
    UINT8 far * Bitmap; /* pointer to current bitmap position */
    unsigned WidthCounter;
    unsigned HeightCounter;
    unsigned ScreenIncrement;
    unsigned BitmapIncrement;

    assert(LinearBM != NULL);

    Left = x - LinearBM->OriginX;
    Top  = y - LinearBM->OriginY;

    if (Left >= (int) ClipRect->Right || Top >= (int) ClipRect->Bottom ||
    	(Left + (int) LinearBM->Width) < ClipRect->Left ||
		(Top + (int)LinearBM->Height) < ClipRect->Top)
        return;

    /* clip bitmap to upper left edge of rect */

	/* if Left < ClipRect->Left, calculate starting x offset in bitmap */
    if (Left < ClipRect->Left) {
    	BMOffsetX = ClipRect->Left - Left;
        ClippedLeft = ClipRect->Left;
        BltWidth = LinearBM->Width - BMOffsetX;
    }
    else {
    	BMOffsetX = 0;
        ClippedLeft = Left;
        BltWidth = LinearBM->Width;
	}

    /* if Top < ClipRect->Top, calculate starting y offset in bitmap */
    if (Top < ClipRect->Top) {
    	BMOffsetY = ClipRect->Top - Top;
        ClippedTop = ClipRect->Top;
        BltHeight = LinearBM->Height - BMOffsetY;
    }
    else {
    	BMOffsetY = 0;
        ClippedTop = Top;
        BltHeight = LinearBM->Height;
    }

    /* clip bitmap to lower right edge of rect */

    /* if Left + bitmap width > rect width, calc ending x point */
    if ((ClippedLeft + BltWidth) > ClipRect->Right) {
    	BltWidth -= ClippedLeft + BltWidth - ClipRect->Right;
    }

    /* if Top + bitmap height > rect height, calc ending y point */
    if ((ClippedTop + BltHeight) > ClipRect->Bottom) {
    	BltHeight -= ClippedTop + BltHeight - ClipRect->Bottom;
    }

    /* calculate screen pointer starting position based on */
    /* clipped bitmap location */
    TempOffset = ClippedTop * GScreenVirtualWidth + ClippedLeft;
    Screen = ScreenBase + TempOffset;

    /* calculate bitmap pointer starting position based on */
    /* clipped bitmap location */
    TempOffset = BMOffsetY * LinearBM->Width + BMOffsetX;
    Bitmap = &(LinearBM->Data);
    Bitmap += TempOffset;

    /* blt to screen */
    ScreenIncrement = GScreenVirtualWidth - BltWidth;
    BitmapIncrement = LinearBM->Width - BltWidth;
    for (HeightCounter = 0; HeightCounter < BltHeight; HeightCounter++) {
    	for (WidthCounter = 0; WidthCounter < BltWidth; WidthCounter++) {
        	if (*Bitmap != 0) {
				*Screen = *Bitmap;
            }
            Screen++;
            Bitmap++;
        }
        Screen += ScreenIncrement;
        Bitmap += BitmapIncrement;
    }
}

/*
	Function: BltPlanarClipRect
    Description:
    	Draws a planar bitmap to a mode X/Y screen.  The routine
        clips the bitmap to the screen.
*/
void BltPlanarClipRect
	(
	PLANAR_BITMAP far * PlanarBM,
	int x,
	int y,
	UINT16 PageOffset,
    RECT * ClipRect
	)
{
	int Top;	/* coordinate values of bitmap top-left corner */
    int Left;
    unsigned PixelWidth;
	unsigned BMOffsetX[4];	/* starting offset into clipped bitmap */
    unsigned BMOffsetY;
    unsigned ClippedLeft;/* upper left corner position of clipped bitmap */
    unsigned ClippedTop;
    unsigned BltWidth[4];	/* width of clipped bitmap */
    unsigned BltHeight; /* height of clipped bitmap */
    UINT16 TempOffset;	/* temp variable to calc far pointer offsets */
    UINT8 far * Screen; /* pointer to current screen position */
    UINT8 far * Bitmap; /* pointer to current bitmap position */
    unsigned WidthCounter;
    unsigned HeightCounter;
    unsigned PlaneCounter;
    unsigned LeftClipWidth;
    unsigned RightClipWidth;
    unsigned ScreenIncrement;
    unsigned BitmapIncrement;
    int i;

    assert(PlanarBM != NULL);

    Left = x - PlanarBM->OriginX;
    Top  = y - PlanarBM->OriginY;

    PixelWidth = PlanarBM->Width * 4;

    if (Left >= (int) ClipRect->Right || Top >= (int) ClipRect->Bottom ||
    	(Left + (int) PixelWidth) < ClipRect->Left ||
		(Top + (int) PlanarBM->Height) < ClipRect->Top)
        return;

    /* clip bitmap width */

    LeftClipWidth = (Left < ClipRect->Left) ? (ClipRect->Left - Left) : 0;
    ClippedLeft = Left + LeftClipWidth;
	RightClipWidth = ((Left + PixelWidth) > ClipRect->Right) ?
    	(Left + PixelWidth - ClipRect->Right) : 0;
    for (i = 0; i < 4; i++) {
    	BMOffsetX[i] = LeftClipWidth / 4;
        BltWidth[i] = PlanarBM->Width - BMOffsetX[i] - (RightClipWidth / 4);
        if (i < LeftClipWidth % 4) {
        	BMOffsetX[i]++;
            BltWidth[i]--;
        }
        if (i >= (4 - (RightClipWidth % 4))) {
        	BltWidth[i]--;
        }
    }

    /* clip bitmap height */
    /* if Top < 0, calculate starting y offset in bitmap */
    if (Top < ClipRect->Top) {
    	BMOffsetY = ClipRect->Top - Top;
        ClippedTop = ClipRect->Top;
        BltHeight = PlanarBM->Height - BMOffsetY;
    }
    else {
    	BMOffsetY = 0;
        ClippedTop = Top;
        BltHeight = PlanarBM->Height;
    }

    /* if Top + bitmap height > screen height, calc ending y point */
    if ((ClippedTop + BltHeight) > ClipRect->Bottom) {
    	BltHeight -= ClippedTop + BltHeight - ClipRect->Bottom;
    }

    /* bitblt data for each plane */
    ScreenIncrement = GScreenVirtualWidth / 4;
    BitmapIncrement = PlanarBM->Width;
    for (PlaneCounter = 0; PlaneCounter < 4; PlaneCounter++) {
    	/* calculate screen pointer starting position based on */
    	/* clipped bitmap and current plane */
    	TempOffset = (ClippedTop * (GScreenVirtualWidth / 4)) +
			((ClippedLeft + PlaneCounter - (LeftClipWidth % 4)) / 4) +
			PageOffset;
        if (PlaneCounter < (LeftClipWidth % 4)) {
        	TempOffset++;
        }
    	Screen = MK_FP(VIDEO_MEM_SEGMENT, TempOffset);
        SetMMR(1 << ((Left + PlaneCounter) % 4));

    	/* calculate bitmap pointer starting position based on */
    	/* clipped bitmap and current plane */
    	TempOffset = (PlanarBM->Width * PlanarBM->Height * PlaneCounter) +
			(BMOffsetY * PlanarBM->Width) + BMOffsetX[PlaneCounter];
    	Bitmap = &(PlanarBM->Data);
    	Bitmap += TempOffset;

    	/* blt to screen */
    	for (HeightCounter = 0; HeightCounter < BltHeight;
			HeightCounter++) {
    		for (WidthCounter = 0; WidthCounter < BltWidth[PlaneCounter];
				WidthCounter++) {
        		if (*Bitmap != 0) {
					*Screen = *Bitmap;
            	}
            	Screen++;
            	Bitmap++;
        	}
        	Screen += ScreenIncrement - BltWidth[PlaneCounter];
        	Bitmap += BitmapIncrement - BltWidth[PlaneCounter];
    	}
    }
}
