{ SKEDDEMO.PAS }
{
Description:  This simple Turbo Vision application demonstrates
              the TScheduleManager object, including a simple
              event monitor.

Author:       Don Taylor
Date:         2 September 1994
Last Revised: 25 October 1994  21:15
Application:  IBM PC and compatibles; BP/TP 7.0; Turbo Vision 2.0
}

program SkedDemo;

{.$DEFINE Debug}

uses
 Dos,
 Crt,
 App,
 Objects,
 Dialogs,
 Menus,
 Drivers,
 Views,
 SCollect,
 Utils,
 {$IFDEF Debug}
 Gadgets,
 {$ENDIF}
 Schedule;

const
 { Program constants }
 MaxEvents   = 10;           { Maximum number of events/messages }
 MinEventNum = 6000;         { Starting event number }
 MaxEventNum = MinEventNum + MaxEvents - 1;

 { Menu command constants }
 cmAddEvent         =  200;
 cmEventStatus      =  201;

{ Program defaults }
const
 OnScreenDelaySecs : Word = 20;   { Number of seconds a message is displayed }

type
 SchedListBoxRec = record   { A default record for listboxes }
  List   : PPStrCollection;
  RecNum : Word;
 end; { record }

 PBalloon = ^TBalloon;
 TBalloon = object(TDialog)
  CloseID : Word;
  constructor Init(var Bounds : TRect; ATitle : TTitleStr; ID : Word);
  procedure HandleEvent(var Event : TEvent); virtual;
 end; { object }

 PScheduleMonitor = ^TScheduleMonitor;
 TScheduleMonitor = object(TDialog)
  EventRec : SchedListBoxRec;
  ListBox  : PListBox;
  constructor Init(var Bounds : TRect; ATitle : TTitleStr);
  destructor Done; virtual;
  procedure FillList; virtual;
  procedure HandleEvent(var Event : TEvent); virtual;
 end; { object }

 TApp = object(TApplication)
  EventNum   : Word;
  {$IFDEF Debug}
  HeapViewer : PHeapView;
  {$ENDIF}
  constructor Init;
  destructor Done; virtual;
  procedure InitMenubar; virtual;
  procedure AddBalloon(ID : Word); virtual;
  procedure AddEventToQueue(ID : Word); virtual;
  procedure ShowEvents;
  procedure Idle; virtual;
  procedure HandleEvent(var Event : TEvent); virtual;
 end; { object }

var
 DemoApp    : TApp;
 Scheduler  : PScheduleManager;

{-------------------------------}
{        Global Routines        }
{-------------------------------}
{---------------------
  LPadStr returns a string padded with spaces on
  the left to make it the specified length.
----------------------}
function LPadStr(S : String; Len : Byte) : String;
var
 s1 : String;
begin
 s1 := s;
 while Length(s1) < Len do s1 := ' ' + s1;
 LPadStr := s1;
end; { LPadStr }

{--------------------
  The DateMMDDStr function formats the date
  in "MM/DD/Year" format.
---------------------}
function DateMMDDStr(DT : DateTime) : String;
var
 S     : String;
 NS    : String;
begin
 with DT do
  begin
   Str(Month, S);
   if Length(S) < 2 then S := '0' + S;

   Str(Day, NS);
   if Length(NS) < 2 then NS := '0' + NS;
   S := S + '/' + NS + '/';

   Str(Year, NS);
   DateMMDDStr := S + NS;
  end; { with }
end; { DateMMDDStr }

{--------------------
  The Time24Str function formats the time
  in 24-hour "HH:MM:SS" format.
---------------------}
function Time24Str(DT : DateTime) : String;
var
 S  : String;
 NS : String;
begin
 with DT do
  begin
   Str(Hour, S);
   if Length(S) < 2 then S := '0' + S;

   Str(Min, NS);
   if Length(NS) < 2 then NS := '0' + NS;
   S := S + ':' + NS + ':';

   Str(Sec, NS);
   if Length(NS) < 2 then NS := '0' + NS;
   S := S + NS;

   Time24Str := S;
  end; { with }
end; { Time24Str }


{-------------------------------}
{    TScheduleMonitor Methods   }
{-------------------------------}

{---------------------
  Init constructs a specialized dialog that
  lets the user browse a list of scheduled
  events.
----------------------}
constructor TScheduleMonitor.Init(var Bounds : TRect; ATitle : TTitleStr);
begin
 inherited Init(Bounds, ATitle);
 ListBox         := nil;
 EventRec.List   := nil;
 EventRec.RecNum := 0;
end; { TScheduleMonitor.Init }

{--------------------
  Done extends the inherited destructor
  to dispose of the object's collection.
---------------------}
destructor TScheduleMonitor.Done;
begin
 if EventRec.List <> nil then Dispose(EventRec.List, Done);
 inherited Done;
end; { TScheduleMonitor.Done }

{--------------------
  FillList fills the listbox with
  data from the scheduler.
---------------------}
procedure TScheduleMonitor.FillList;
var
 NList   : PPStrCollection;
 P       : PString;
 s       : String;
 i       : Integer;
 LTime   : Longint;
 SEvent  : ScheduleEventRec;
 DT      : DateTime;
begin
 { Create a new list of events }
 NList := New(PPStrCollection, Init(10, 5));
 for i := 0 to Scheduler^.NumEvents - 1 do
  begin
   LTime := Scheduler^.At(i, SEvent);
   if LTime <> 0
    then begin
          Str(SEvent.EventID, s);
          s := LPadStr(s, 6);
          LongintToDate(LTime, DT);
          s := s + LPadStr(DateMMDDStr(DT), 16);
          s := s + LPadStr(Time24Str(DT), 10);
          if SEvent.Recurring
           then s := s + '    Y'
           else s := s + '    N';

          P := NewStr(s);
          NList^.Insert(P);
         end;
  end; { for }

 { Give the new list to the listbox }
 ListBox^.NewList(NList);
 EventRec.RecNum := 0;
 EventRec.List   := NList;
end; { TScheduleMonitor.FillList }

{--------------------
  HandleEvent extends the ancestor's ability
  to handle special keys and messages.
---------------------}
procedure TScheduleMonitor.HandleEvent(var Event : TEvent);
begin
 case Event.What of
  evBroadcast :
   case Event.Command of
    cmScheduleChanged :
     begin { Force an update of the listbox }
(*      ClearEvent(Event); *)
      FillList;
     end;
   end; { case }

  evCommand :
   if Event.Command = cmCancel
    then begin
          ClearEvent(Event);
          Message(@Self, evCommand, cmClose, nil);
         end;

  evKeyDown :
   case Event.KeyCode of
    kbEsc :
     begin
      ClearEvent(Event);
      Message(@Self, evCommand, cmClose, nil);
     end;
   end; { case }
 end; { case }

 inherited HandleEvent(Event);
end; { TScheduleMonitor.HandleEvent }


{-------------------------------}
{        TBalloon Methods       }
{-------------------------------}

{---------------------
  Init constructs a specialized dialog -
  called a "balloon" - that displays
  its event number.
----------------------}
constructor TBalloon.Init(var Bounds : TRect; ATitle : TTitleStr; ID : Word);
begin
 inherited Init(Bounds, ATitle);
 CloseID := ID;
end; { TBalloon.Init }

{--------------------
  HandleEvent extends the ancestor's ability
  to handle special keys.
---------------------}
procedure TBalloon.HandleEvent(var Event : TEvent);
begin
 case Event.What of
  evBroadcast :
   if Event.Command = CloseID
    then begin { "Pop" the balloon if the command is its ID }
          ClearEvent(Event);
          Message(@Self, evCommand, cmClose, nil);
         end;

  evKeyDown :
   case Event.KeyCode of
    kbEsc :
     begin
      ClearEvent(Event);
      Message(@Self, evCommand, cmClose, nil);
     end;

   end; { case }
 end; { case }
 inherited HandleEvent(Event);
end; { TBalloon.HandleEvent }


{=========================}
{      TApp Methods       }
{=========================}

{--------------------
  Init constructs a new application
  object.
---------------------}
constructor TApp.Init;
{$IFDEF Debug}
var
 R : TRect;
{$ENDIF}
begin
 inherited Init;
 EventNum := MinEventNum;

 {$IFDEF Debug}
 { Install the Heap Viewer }
 GetExtent(R);
 Dec(R.B.X);
 R.A.X := R.B.X - 9;
 R.B.X := R.A.X + 9;
 R.A.Y := R.B.Y - 1;
 HeapViewer := New(PHeapView, Init(R));
 HeapViewer^.GrowMode := gfGrowLoY + gfGrowHiX + gfGrowHiY;
 Insert(HeapViewer);
 {$ENDIF}

 Scheduler := New(PScheduleManager, Init);
end; { TApp.Init }

{--------------------
  Done disposes of any objects not
  owned by the Desktop.
---------------------}
destructor TApp.Done;
begin
 Dispose(Scheduler, Done);
 inherited Done;
end; { TApp.Done }

{--------------------
  InitMenuBar initializes a special
  menu bar for this application.
---------------------}
procedure TApp.InitMenuBar;
var
 R : TRect;
begin
 GetExtent(R);
 R.B.Y := R.A.Y + 1;
 MenuBar := New(PMenuBar, Init(R, NewMenu(
  NewSubMenu('~S~chedule Demo', hcNoContext, NewMenu(
   NewItem('~A~dd balloon', 'F3', kbF3, cmAddEvent, hcNoContext,
   NewItem('~E~vent status', 'F5', kbF5, cmEventStatus, hcNoContext,
   NewLine(
   NewItem('E~x~it', 'Alt-X', kbAltX, cmQuit, hcNoContext,
   nil))))),
  nil)
 )));
end; { TApp.InitMenuBar }

{--------------------
  AddBalloon places a new event
  message (balloon) at a random
  location on the screen.
---------------------}
procedure TApp.AddBalloon(ID : Word);
var
 PDlg    : PBalloon;
 R       : TRect;
 Control : PView;
 HScroll : PView;
 Message : PStaticText;
 NumStr  : String[20];
 X       : Word;
 Y       : Word;
begin
 Randomize;
 X := Random(55) + 1;
 Y := Random(15) + 1;
 R.Assign(X, Y, X+24, Y+5);
 PDlg := New(PBalloon, Init(R, 'Balloon', ID));

 R.Assign(2,2,22,3);
 Str(ID, NumStr);
 NumStr := #3 + 'Event #' + NumStr;
 Message := New(PStaticText, Init(R, NumStr));
 PDlg^.Insert(Message);

 PDlg^.SelectNext(False);
 Desktop^.Insert(PDlg);
end; { TApp.AddBalloon }

{--------------------
  AddEventToQueue places a new event
  in the Scheduler's queue.
---------------------}
procedure TApp.AddEventToQueue(ID : Word);
var
 ScheduleRec : ScheduleEventRec;
begin
 with ScheduleRec do
  begin
   EventID     := ID;
   EventType   := scTimeDelay;
   Time        := OnScreenDelaySecs;
   MessageType := scBroadcast;
   Command     := ID; { The balloon will recognize its own ID
                        as a command to "pop" }
   Recurring   := False;
  end; { with }

 if Scheduler <> nil then Scheduler^.Add(ScheduleRec);
end; { TApp.AddEventToQueue }

{--------------------
  ShowEvents brings up a dialog with
  a listbox that displays the events
  currently in the scheduler's queue.
---------------------}
procedure TApp.ShowEvents;
var
 R         : TRect;
 Control   : PView;
 HScroll   : PView;
 EventDlg  : PScheduleMonitor;

begin
 { Create the dialog }
 R.Assign(15, 2, 64, 15);
 EventDlg := New(PScheduleMonitor, Init(R, 'Scheduled Events'));

 R.Assign(45, 3, 46, 9);
 Control := New(PScrollbar, Init(R));
 EventDlg^.Insert(Control);

 R.Assign(3, 3, 45, 9);
 Control := New(PListBox, Init(R, 1, PScrollbar(Control)));
 EventDlg^.Insert(Control);
 EventDlg^.ListBox := PListBox(Control);

 R.Assign(3, 2, 11, 3);
 Control := New(PStaticText, Init(R, 'Event ID'));
 EventDlg^.Insert(Control);

 R.Assign(19, 2, 31, 3);
 Control := New(PStaticText, Init(R, 'Trigger Time'));
 EventDlg^.Insert(Control);

 R.Assign(36, 2, 45, 3);
 Control := New(PStaticText, Init(R, 'Recurring'));
 EventDlg^.Insert(Control);

 R.Assign(20, 10, 30, 12);
 Control := New(PButton, Init(R, '~D~one', cmCancel, bfDefault));
 EventDlg^.Insert(Control);

 EventDlg^.SelectNext(False);
 EventDlg^.SelectNext(False);

 { Fill the list box and insert it }
 EventDlg^.FillList;
 EventDlg^.HelpCtx := hcNoContext;
 Desktop^.Insert(EventDlg);
end; { TApp.ShowEvents }

{--------------------
  The Idle procedure calls the Update
  method of the Scheduler.
---------------------}
procedure TApp.Idle;
begin
 inherited Idle;
 {$IFDEF Debug}
 HeapViewer^.Update;
 {$ENDIF}
 Scheduler^.Update;
end; { TApp.Idle }

{--------------------
  HandleEvent processes the menu
  commands.
---------------------}
procedure TApp.HandleEvent(var Event : TEvent);
begin
 inherited HandleEvent(Event);
 if Event.What = evCommand
  then case Event.Command of
        cmAddEvent :
         begin
          if EventNum <= MaxEventNum
           then begin
                 AddBalloon(EventNum);
                 AddEventToQueue(EventNum);
                 Inc(EventNum);
                 if EventNum > MaxEventNum
                  then DisableCommands([cmAddEvent]);
                end;
          ClearEvent(Event);
         end;

        cmEventStatus :
         begin
          ShowEvents;
          ClearEvent(Event);
         end;
       end; { case }
end; { TApp.HandleEvent }

{====================}
begin { Program SkedDemo }
 DemoApp.Init;
 DemoApp.Run;
 DemoApp.Done;
end. { Program SkedDemo }
