{ SCHEDULE.PAS }

{
Description:  This unit manages a linked list of scheduled "events" -
              Turbo Vision messages which are tied to the times they
              are to be transmitted.

Author:       Don Taylor
Date:         4 January 1994
Last Revised: 25 October 1994 18:05
Application:  IBM PC and compatibles; BP/TP 7.0; Turbo Vision 2.0
}

unit Schedule;

{--------------------}
     INTERFACE
{--------------------}

uses
 Objects,
 Drivers,
 Dos,
 Views,
 Utils,
 App;

type
 ScheduleMessageType = (scCommand, scBroadcast);
 ScheduleEventType   = (scSchedule, scTimeDelay);

 PScheduleEventRec = ^ScheduleEventRec;
 ScheduleEventRec = record
  EventID     : Word;                   { Unique identifier assigned by user }
  EventType   : ScheduleEventType;      { Scheduled event or time-delayed event }
  Time        : Longint;                { Absolute date/time or delay time }
  MessageType : ScheduleMessageType;    { Transmit a command or a broadcast? }
  Command     : Word;                   { The command to send }
  Recurring   : Boolean;                { If time delay, should it repeat? }
 end; { record }

 PTriggerCollection = ^TTriggerCollection;
 TTriggerCollection = object(TCollection)
  constructor Init(ALimit, ADelta : Integer);
  function GetItem(var S : TStream) : Pointer; virtual;
  procedure PutItem(var S : TStream; Item : Pointer); virtual;
  procedure FreeItem(Item : Pointer); virtual;
 end; { object }

 PScheduleManager = ^TScheduleManager;
 TScheduleManager = object(TObject)
  Disabled : Boolean;      { *** READ ONLY *** }
  constructor Init;
  destructor Done; virtual;
  function Add(EventData : ScheduleEventRec) : Boolean; virtual;
  function Kill(ID : Word) : Boolean; virtual;
  function Get(ID : Word; var EventData : ScheduleEventRec) : Longint; virtual;
  function At(Idx : Word; var EventData : ScheduleEventRec) : Longint; virtual;
  procedure Update; virtual;
  procedure Enable; virtual;
  procedure Disable; virtual;
  procedure CorrectTime(DeltaTime : Longint); virtual;
  function NumEvents : Word; virtual;
  function Clear : Boolean; virtual;
  constructor Load(var S : TStream);
  procedure Store(var S : TStream); virtual;

 private
  List : PTriggerCollection;
 end; { object }

procedure RegisterScheduler;

{ Command/Message Constants }
const
 cmScheduleChanged = 65001;

const
 RTriggerCollection : TStreamRec = (
  ObjType : 1570;
  VmtLink : Ofs(TypeOf(TTriggerCollection)^);
  Load : @TTriggerCollection.Load;
  Store : @TTriggerCollection.Store
 );

 RScheduleManager : TStreamRec = (
  ObjType : 1571;
  VmtLink : Ofs(TypeOf(TScheduleManager)^);
  Load : @TScheduleManager.Load;
  Store : @TScheduleManager.Store
 );

{--------------------}
   IMPLEMENTATION
{--------------------}

type
 PScheduleTriggerRec = ^ScheduleTriggerRec;
 ScheduleTriggerRec = record
  EventRec    : ScheduleEventRec;
  TriggerTime : Longint;
 end; { record }


{  Global Routines }

{--------------------
  RegisterScheduler registers the objects
  for this unit.
---------------------}
procedure RegisterScheduler;
begin
 RegisterType(RTriggerCollection);
 RegisterType(RScheduleManager);
end; { RegisterScheduler }

{-----------------------------------}
{  TTriggerCollection Methods  }
{-----------------------------------}

{--------------------
  Init constructs a TCollection
  derivative that holds records
  containing trigger information
  for scheduled events.
  Override : Seldom.
---------------------}
constructor TTriggerCollection.Init(ALimit, ADelta : Integer);
begin
 inherited Init(ALimit, ADelta);
end; { TTriggerCollection.Init }

{--------------------
  GetItem overrides the standard
  version to provide for retreiving
  schedule trigger records from a
  stream. Override : Seldom.
---------------------}
function TTriggerCollection.GetItem(var S : TStream) : Pointer;
var
 P : PScheduleTriggerRec;
begin
 New(P);
 with P^ do
  begin
   S.Read(EventRec.EventID, SizeOf(Word));
   S.Read(EventRec.EventType, SizeOf(ScheduleEventType));
   S.Read(EventRec.Time, SizeOf(Longint));
   S.Read(EventRec.MessageType, SizeOf(ScheduleMessageType));
   S.Read(EventRec.Command, SizeOf(Word));
   S.Read(EventRec.Recurring, SizeOf(Boolean));
   S.Read(TriggerTime, SizeOf(Longint));
  end;
 GetItem := P;
end; { TTriggerCollection.GetItem }

{--------------------
  PutItem overrides the standard
  version to provide for saving
  schedule trigger records to a
  stream. Override : Seldom.
---------------------}
procedure TTriggerCollection.PutItem(var S : TStream; Item : Pointer);
var
 TR : ScheduleTriggerRec;
begin
 TR := ScheduleTriggerRec(Item^);
 with TR do
  begin
   S.Write(EventRec.EventID, SizeOf(Word));
   S.Write(EventRec.EventType, SizeOf(ScheduleEventType));
   S.Write(EventRec.Time, SizeOf(Longint));
   S.Write(EventRec.MessageType, SizeOf(ScheduleMessageType));
   S.Write(EventRec.Command, SizeOf(Word));
   S.Write(EventRec.Recurring, SizeOf(Boolean));
   S.Write(TriggerTime, SizeOf(Longint));
  end; { with }
end; { TTriggerCollection.PutItem }

{--------------------
  FreeItem overrides the standard
  version to provide for disposing
  of individual trigger records from
  the collection. Override : Seldom.
---------------------}
procedure TTriggerCollection.FreeItem(Item : Pointer);
begin
 Dispose(PScheduleTriggerRec(Item));
end; { TTriggerCollection.FreeItem }


{----------------------------}
{  TScheduleManager Methods  }
{----------------------------}

{--------------------
  Init constructs an object that
  manages a TTriggerCollection,
  providing an active scheduling
  "machine". Override : Seldom.
---------------------}
constructor TScheduleManager.Init;
begin
 inherited Init;
 List     := New(PTriggerCollection, Init(30, 5));
 Disabled := False;
end;

{--------------------
  Done overrides the inherited
  destructor to dispose of the
  managed collection.
  Override : Seldom.
---------------------}
destructor TScheduleManager.Done;
begin
 inherited Done;
 Dispose(List, Done);
end; { TScheduleManager.Done }

{--------------------
  The Add method attempts to schedule
  the specified event, by placing it
  in the collection. The event ID number
  must not match any currently scheduled
  events. Add returns TRUE if the operation
  is successful. It also broadcasts a message
  to the application, indicating the schedule
  has changed. Override : Seldom.
---------------------}
function TScheduleManager.Add(EventData : ScheduleEventRec) : Boolean;
var
 P        : PScheduleTriggerRec;
 NewEvent : PScheduleTriggerRec;

  function Matches(Item : Pointer) : Boolean; far;
  begin
   Matches := Word(Item^) = EventData.EventID;
  end; { Matches }

  function ItemIsGreaterThan(Item : Pointer) : Boolean; far;
  begin
   ItemIsGreaterThan := PScheduleTriggerRec(Item)^.TriggerTime > NewEvent^.TriggerTime;
  end; { ItemIsGreaterThan }

begin
 if not Disabled
  then begin
        Disable;
        P := List^.FirstThat(@Matches);
        if P = nil { The ID is unique }
         then begin  { Add the new schedule event to the list }
               New(NewEvent);
               NewEvent^.EventRec := EventData;

               if NewEvent^.EventRec.EventType = scSchedule
                then begin
                      NewEvent^.TriggerTime := NewEvent^.EventRec.Time;
                     end
                else begin
                      NewEvent^.TriggerTime := Now + NewEvent^.EventRec.Time;
                     end;

               P := List^.FirstThat(@ItemIsGreaterThan);
               if P <> nil
                then List^.AtInsert(List^.IndexOf(P), NewEvent)
                else List^.Insert(NewEvent);
               Add := True;
               Message(Application, evBroadcast, cmScheduleChanged, nil);
              end
         else begin { The ID was a duplicate }
               Add := False;
              end;
        Enable;
       end
  else Add := False;
end; { TScheduleManager.Add }

{--------------------
  The Kill method attempts to remove
  the specified event from the collection.
  It returns TRUE and broadcasts a change
  notification if the operation is successful.
  Override : Seldom.
---------------------}
function TScheduleManager.Kill(ID : Word) : Boolean;
var
 P : PScheduleTriggerRec;

  function Matches(Item : Pointer) : Boolean; far;
  begin
   Matches := PScheduleTriggerRec(Item)^.EventRec.EventID = ID;
  end; { Matches }

begin
 if not Disabled
  then begin
        Disable;
        P := List^.FirstThat(@Matches);
        if P <> nil then List^.Free(P);
        Kill := P <> nil;
        if P <> nil
         then Message(Application, evBroadcast, cmScheduleChanged, nil);
        Enable;
       end
  else Kill := False;

end; { TScheduleManager.Kill }

{--------------------
  Get returns a Longint containing the
  trigger time data for the specified
  event ID, along with the complete
  data record. If unsuccessful, it returns
  zero. Override : Seldom.
---------------------}
function TScheduleManager.Get(ID : Word; var EventData : ScheduleEventRec) : Longint;
var
 P : PScheduleTriggerRec;

  function Matches(Item : Pointer) : Boolean; far;
  begin
   Matches := PScheduleTriggerRec(Item)^.EventRec.EventID = ID;
  end; { Matches }

begin
 P := PScheduleTriggerRec(List^.FirstThat(@Matches));
 if P <> nil
  then begin
        EventData := P^.EventRec;
        Get := P^.TriggerTime;
       end
  else Get := 0;
end; { TScheduleManager.Get }

{--------------------
  At returns a Longint containing the
  trigger time data for the collection
  record at the specified index, along
  with the complete data record. If
  unsuccessful, it returns zero.
  Override : Seldom.
---------------------}
function TScheduleManager.At(Idx : Word; var EventData : ScheduleEventRec) : Longint;
var
 P : PScheduleTriggerRec;
begin
 if Idx < List^.Count
  then begin
        P := PScheduleTriggerRec(List^.At(Idx));
        EventData := P^.EventRec;
        At := P^.TriggerTime;
       end
  else At := 0;
end; { TScheduleManager.At }

{--------------------
  Update is the workhorse method. It checks
  the first trigger event, to see if its
  trigger time is less than the current
  time. If so, it sends an appropriate
  message to the application and then removes
  or updates the trigger record.
  Override: Seldom.
---------------------}
procedure TScheduleManager.Update;
var
 P     : PScheduleTriggerRec;
 CTime : Longint;
begin
 if not Disabled and (List^.Count > 0)
  then begin
        Disable;
        CTime := Now;

        P := List^.At(0);
        if CTime >= P^.TriggerTime
         then begin { Send the appropriate message }
               case P^.EventRec.MessageType of
                scCommand :
                 Message(Application, evCommand, P^.EventRec.Command, nil);
                scBroadcast :
                 Message(Application, evBroadcast, P^.EventRec.Command, nil);
               end; { case }

               if (P^.EventRec.Recurring)
                then { Reschedule the event }
                 P^.TriggerTime := CTime + P^.EventRec.Time
                else { Remove the event from the list }
                 List^.AtFree(0);

               Message(Application, evBroadcast, cmScheduleChanged, nil);
              end;
        Enable;
       end;
end; { TScheduleManager.Update }

{--------------------
  Enable clears the Disabled flag,
  which enables the schedule manager.
  Override: Seldom.
---------------------}
procedure TScheduleManager.Enable;
begin
 Disabled := False;
end; { TScheduleManager.Enable }

{--------------------
  Disable sets the Disabled flag,
  which disables the schedule manager.
  Override: Seldom.
---------------------}
procedure TScheduleManager.Disable;
begin
 Disabled := True;
end; { TScheduleManager.Disable }

{--------------------
  The CorrectTime method provides a
  way to update the entire list of scheduled
  events, to accommodate a change in the
  Real Time Clock. Events which are scheduled
  to occur at a specific time are not affected,
  but those events which are time delays are
  adjusted by the amount (+ or -) specified.
  This permits a correction of the Real Time
  Clock without messing up active time delay
  events. Override: Seldom.
---------------------}
procedure TScheduleManager.CorrectTime(DeltaTime : Longint);
var
 DisableState : Boolean;

  procedure Correct(P : PScheduleTriggerRec); far;
  begin
   if P^.EventRec.EventType <> scSchedule
    then P^.TriggerTime := P^.TriggerTime + DeltaTime;
  end; { Correct }

begin
 DisableState := Disabled;
 if not DisableState then Disable;
 List^.ForEach(@Correct);
 if not DisableState then Enable;
end; { TScheduleManager.CorrectTime }

{--------------------
  NumEvents returns the number of events
  currently in the list. Override: Seldom.
---------------------}
function TScheduleManager.NumEvents : Word;
begin
 NumEvents := List^.Count;
end; { TScheduleManager.NumEvents }

{--------------------
  Clear attempts to remove all scheduled
  events from the list. If successful,
  it returns TRUE, and a change notification
  is sent to the application. Override: Seldom.
---------------------}
function TScheduleManager.Clear : Boolean;
begin
 if not Disabled
  then begin
        Disable;
        List^.FreeAll;
        Clear := True;
        Message(Application, evBroadcast, cmScheduleChanged, nil);
        Enable;
       end
  else Clear := False;
end; { TScheduleManager.Clear }

{--------------------
  Load constructs a schedule manager object
  from a stream. Override: Seldom.
---------------------}
constructor TScheduleManager.Load(var S : TStream);
begin
 S.Read(Disabled, SizeOf(Boolean));
 List := PTriggerCollection(S.Get);
end; { TScheduleManager.Load }

{--------------------
  Store writes a schedule manager object to
  a stream. Override: Seldom.
---------------------}
procedure TScheduleManager.Store(var S : TStream);
begin
 S.Write(Disabled, SizeOf(Boolean));
 S.Put(List);
end; { TScheduleManager.Store }

end. { unit Schedule }