unit TrafLite;

interface {=== TrafLite Unit ===}

  uses
    Controls, Classes, Forms, DsgnIntf, StdCtrls, Graphics;

  const                      { Special Colors for Traffic Light }
    clDullRed       = TColor( $000040 );
    clDullYellow    = TColor( $408080 );
    clDullGreen     = TColor( $004000 );
    clTrafficYellow = TColor( $00C0C0 );

  type
    TActiveLight = ( alNone, alRed, alYellow, alGreen, alAll );
    TLightArray = array[ TActiveLight ] of TColor;

  const       { Light color must change to reflect active light }
    RedLightColor    : TLightArray = ( clDullRed, clRed,
                                       clDullRed, clDullRed,
                                       clRed );
    YellowLightColor : TLightArray = ( clDullYellow, clDullYellow,
                                       clYellow, clDullYellow,
                                       clYellow );
    GreenLightColor  : TLightArray = ( clDullGreen, clDullGreen,
                                       clDullGreen, clLime,
                                       clLime );

  type
    TTrafficLight = class( TGraphicControl )
    private
      FActiveLight : TActiveLight;
      FCaseColor   : TColor;
    protected
      procedure SetActiveLight( Value : TActiveLight );
      procedure SetCaseColor( Value : TColor );
      procedure Paint; override;
    public
      constructor Create(AOwner: TComponent); override;
    published
      property ActiveLight : TActiveLight read FActiveLight
                                          write SetActiveLight;
      property CaseColor : TColor read FCaseColor
                                  write SetCaseColor;
      property OnClick;            { Make OnClick event visible }
    end;

  procedure Register;

implementation {=== TrafLite Unit ===}

  uses
    Messages, WinTypes, WinProcs;


  {===========================}
  {== TTrafficLight Methods ==}
  {===========================}

  constructor TTrafficLight.Create( AOwner : TComponent );
  begin
    inherited Create( AOwner );
    FActiveLight := alAll;           { Set Default Active Light }
    FCaseColor := clTrafficYellow;     { Set Default Case Color }
    Width := 48;                            { Set Default Width }
    Height := 60;                          { Set Default Height }
  end; {= TTrafficLight.Create =}


  procedure TTrafficLight.SetActiveLight( Value : TActiveLight );
  begin
    if Value <> FActiveLight then
    begin
      FActiveLight := Value;
      Repaint;                  { Repaint with new Active Light }
    end;
  end; {= TTrafficLight.SetActiveLight =}


  procedure TTrafficLight.SetCaseColor( Value : TColor );
  begin
    if Value <> FCaseColor then
    begin
      FCaseColor := Value;
      Repaint;
    end;
  end; {= TTrafficLight.SetCaseColor =}


  procedure TTrafficLight.Paint;
  var
    RegionX, CenterX, RegionY, DeltaY, Radius : Integer;

    {= DrawCovers - Draws the black triangular covers on sides =}
    procedure DrawCovers( X, CY, R : Integer );
    begin
      Canvas.Polygon( [ Point( X, CY - R ),
                        Point( 0, CY - R ),
                        Point( X, CY ) ] );
      Canvas.Polygon( [ Point( 3 * X, CY - R ),
                        Point( 4 * X, CY - R ),
                        Point( 3 * X, CY ) ] );
    end; {= DrawCovers =}

    {= DrawLight - Draws Red, Yellow, or Green light on front =}
    procedure DrawLight( CX, CY, R : Integer );
    begin
      { Draw the actual light }
      Canvas.Ellipse( CX - R, CY - R, CX + R, CY + R );
      { Draw a straight-on view of cover over the light }
      Canvas.Arc( CX - R, CY - R + 1, CX + R, CY + R,
                  CX + R, CY, CX - R, CY );
      Canvas.Arc( CX - R, CY - R + 2, CX + R, CY + R,
                  CX + R, CY, CX - R, CY );
    end; {= DrawLight =}

  begin {= TTrafficLight.Paint =}
    RegionX := Round( Width / 4 );            { Position Values }
    CenterX := Round( Width / 2 );
    RegionY := Round( Height / 3 );
    DeltaY  := Round( Height / 6 );
    Radius  := Round( Height / 8 );

    with Canvas do
    begin
      Brush.Color := FCaseColor; { Draw case using FCaseColor }
      RoundRect( RegionX, 0, 3 * RegionX, Height, 6, 6 );

      { Draw all three sets of light covers }
      Brush.Color := clBlack;      { Covers are painted black }
      DrawCovers( RegionX, DeltaY, Radius );
      DrawCovers( RegionX, RegionY + DeltaY, Radius );
      DrawCovers( RegionX, 2 * RegionY + DeltaY, Radius );

      { Drawing the Lights:
        The appropriate light color is determined by the
        Active Light and the corresponding Color Array }

      { Red Light }
      Brush.Color := RedLightColor[ FActiveLight ];
      DrawLight( CenterX, DeltaY, Radius );

      { Yellow Light }
      Brush.Color := YellowLightColor[ FActiveLight ];
      DrawLight( CenterX, RegionY + DeltaY, Radius );

      { Green Light }
      Brush.Color := GreenLightColor[ FActiveLight ];
      DrawLight( CenterX, 2 * RegionY + DeltaY, Radius );
    end; { with }

  end; {= TTrafficLight.Paint =}


  {= Register the TrafficLight component with Delphi and place =}
  {= under Blazing Pascal tab in component palette.            =}
  procedure Register;
  begin
    RegisterComponents( 'Blazing Pascal', [ TTrafficLight ] );
  end;

end. {=== TrafLite Unit ===}

