// BITMANIP.C   Michael A. Covington 1994
// Geometric transformation of Windows BMP files
// Borland C for Windows 3.1

#include <alloc.h>   
#include <stdlib.h>
#include <stdio.h>
#include <windows.h>
#include <commdlg.h>
#include <sys\stat.h> 
#include <string.h>

// --- Globals -----------------------------------------------

typedef struct {                 // Part of Windows .BMP header 
  char           id1, id2;       // Must be 'BM' 
  unsigned long  filesize;       // May be 0 when read in 
  int            junk1, junk2;
  unsigned long  ofsdata;        // Offset of bitmap in file 
  unsigned long  bmisize;        // Must be 40 in Windows BMP
  unsigned long  width, height;
  unsigned int   planes;         // Must be 1 
  unsigned int   bitsperpixel;   // 1, 4, 8, or 24
  unsigned long  compression;    // Must be 0 
  // Ignore rest of header; all will get copied anyway.
} HEADER;

typedef struct {                 // 2 useful numbers not in hdr
  int            bytesperpixel;  // 1, 1, 1, or 3 respectively
  int            widthbytes;     // Affected by rounding
} LAYOUT;

unsigned char   huge *OldBMP = NULL, huge *NewBMP = NULL;
HEADER          huge *OldHeader,     huge *NewHeader;
LAYOUT          OldLayout,           NewLayout;
 
// --- User interface functions ------------------------------

void Quit(char* text1, char* text2){
  // Print 2 msgs, free dynamic memory, and end program 
  printf("%s %s \n",text1,text2);
  if (OldBMP != NULL) farfree(OldBMP);
  if (NewBMP != NULL) farfree(NewBMP);
  puts("Program terminated.");
  exit(0);
}

void GetFileName(char* filename, int fnlength, char rw){
  // Calls a Windows common file dialog to choose a file.
  // filename = string in which to return full path to file
  // fnlength = length of filename
  // rw       = 'r' to read, 'w' to write

  OPENFILENAME    ofn;   // Data structure for file dialog

  // Initialize filename to invalid value (containing '*').
  // User will be forced to give an actual file name.
  strcpy(filename,"*.bmp"); 

  // Initialize ofn to all zeroes, then change some fields
  memset(&ofn,0,sizeof(OPENFILENAME));  
  ofn.lStructSize  =  sizeof(OPENFILENAME);
  ofn.hwndOwner    = GetActiveWindow();
  ofn.lpstrFilter  = "Bitmap Files\000*.bmp\000";
  ofn.nFilterIndex = 1;
  ofn.lpstrFile    = filename;  // pointer assignment
  ofn.nMaxFile     = fnlength;
  if (rw == 'w') {
    ofn.lpstrTitle = "File to Write";
    ofn.Flags = OFN_PATHMUSTEXIST | OFN_OVERWRITEPROMPT |
		OFN_HIDEREADONLY | OFN_NOREADONLYRETURN;
  }
  else
  {
    ofn.lpstrTitle = "File To Read";
    ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST |
                OFN_READONLY;
  }

  // Call Windows and check that result is nonzero
  if (!GetOpenFileName(&ofn))
    Quit(filename,"is not a valid filename.");
}

// --- Bitmap handling functions -----------------------------

void LoadOldBMP(char *infilename){
  // Obtains huge memory object, reads entire file into it,
  // makes OldBMP and OldHeader point to it, validates contents,
  // initializes layout data.  Aborts program if unsuccessful.

  int		r; 
  HFILE		infile;
  struct stat	infilestat;
  long		bytecount;

  r = stat(infilename,&infilestat);
  if (r == -1) Quit("Can't find file",infilename);

  OldBMP = (char _huge *) farmalloc(infilestat.st_size);
  if (OldBMP == NULL) Quit("Insufficient memory","");

  infile = _lopen(infilename,READ);
  if (infile == HFILE_ERROR) Quit("Can't open",infilename);

  printf("Loading %s...",infilename);
  bytecount = _hread(infile,OldBMP,infilestat.st_size);
  if (bytecount != infilestat.st_size)
    Quit("Can't read",infilename);

  printf("Read %ld bytes.\n",bytecount);
  _lclose(infile);

  OldHeader = (HEADER _huge *) OldBMP;
  // Header = first part of file, interpreted differently 

  if ( OldHeader->id1         != 'B' ||
       OldHeader->id2         != 'M' ||
       OldHeader->bmisize     != 40  ||
       OldHeader->planes      != 1   ||
       OldHeader->compression != 0     )
          Quit("Not an uncompressed Windows BMP file","");
        
  OldHeader->filesize = bytecount;  // Ensure correct value

  OldLayout.bytesperpixel = (OldHeader->bitsperpixel + 7) / 8;
  OldLayout.widthbytes =
    (OldHeader->filesize-OldHeader->ofsdata)/OldHeader->height;
      // Width of row in bytes has been rounded up;
      // exact value is computed from file size.

  printf("Successfully loaded %ld x %ld %ld-color bitmap.\n",
	 OldHeader->width,
	 OldHeader->height,
	 (1L << OldHeader->bitsperpixel));
}

void InitializeNewBMP(void){
  // Creates a NewBMP identical in size to OldBMP.
  // Copies entire header and color table into it.
  // Remainder of it remains uninitialized.

  NewBMP = (char _huge *) farmalloc(OldHeader->filesize);
  if (OldBMP == NULL) Quit("Insufficient memory","");

  NewHeader = (HEADER _huge *) NewBMP;
  _fmemcpy(NewBMP,OldBMP,OldHeader->ofsdata);

  NewLayout.bytesperpixel = OldLayout.bytesperpixel;
  NewLayout.widthbytes    = OldLayout.widthbytes;
}

void SaveNewBMP(char* outfilename){
  // Writes NewBMP on a file. 

  HFILE	outfile;
  long	bytecount;

  outfile = _lcreat(outfilename,0);
  if (outfile == HFILE_ERROR) Quit("Can't open",outfilename);
  printf("Writing %s...",outfilename);

  bytecount = _hwrite(outfile,NewBMP,NewHeader->filesize);
  if (bytecount != NewHeader->filesize)
    Quit("Can't write complete file",outfilename);
  printf("Wrote %ld bytes.\n",bytecount);

  _lclose(outfile);
}

// --- Pixel access functions --------------------------------

unsigned long ReadPixel(int x, int y){
  // Retrieves value of one pixel, rightmost in a 32-bit dword.
  // If coordinates out of range, returns nearest actual pixel.

  unsigned long  byteoffset;
  int  		 bitoffset, i;
  unsigned long  result = 0L;

  // Bring within range
  if (x < 0) x = 0;
  if (y < 0) y = 0;
  if (x > OldHeader->width - 1) x = OldHeader->width - 1;
  if (y > OldHeader->height - 1) y = OldHeader->height - 1;

  // Figure out where we are
  byteoffset = (long)y * OldLayout.widthbytes 
       + x * OldHeader->bitsperpixel / 8
       + OldHeader->ofsdata;
  bitoffset =  (x * OldHeader->bitsperpixel) % 8;

  // Grab appropriate number of bytes into a 32-bit dword
  for (i = 0; i < OldLayout.bytesperpixel; i++)
    result =
      result | ((unsigned long)OldBMP[byteoffset+i] << (i*8));

  // Shift whole thing to high end of dword, then to low end
  result =
    (result << (32 - 8*OldLayout.bytesperpixel + bitoffset))
      >> (32 - OldHeader->bitsperpixel);
  return result; 
}

void WritePixel(int x, int y, unsigned long v){
  // Inverse of ReadPixel.  Stores v in NewBMP at (x,y).
  // Coordinates must be within range.

  unsigned long  byteoffset, mask;
  int            bitoffset, i, shift;
  unsigned char  bytev, bytemask;

  // Figure out where we are
  byteoffset = (long)y * NewLayout.widthbytes 
       + x * NewHeader->bitsperpixel / 8
       + NewHeader->ofsdata;
  bitoffset =  (x * NewHeader->bitsperpixel) % 8;

  // Shift bits to appropriate position in 32-bit dword.
  // Create a mask with 1's in those positions, 0's elsewhere.
  shift = (8 - NewHeader->bitsperpixel) % 8 - bitoffset;
  v = v << shift;
  mask = (0xFFFFFFFFL >> (32 - NewHeader->bitsperpixel)) << shift;

  // Twiddle the appropriate number of bytes
  for (i = 0; i < NewLayout.bytesperpixel; i++){
     bytev = 0xFF & (v >> (i*8));
     bytemask = 0xFF & (mask >> (i*8));
     NewBMP[byteoffset+i] =
       (NewBMP[byteoffset+i] & ~bytemask) | bytev;
  }
}

// --- Function to compute the transformation ----------------

#include "tiling.c"   // defines TITLE and ComputePixel(x,y)

// --- Main control functions --------------------------------

void ComputeNewBMP(void){
  int x, y;
  printf("Computing...     ");
  for (y=0; y<OldHeader->height; y++){
    if (y%4 == 0)
      printf("\b\b\b\b\b%4.0f%%",100.0 * y / OldHeader->height);
    for (x=0; x<OldHeader->width; x++)
      WritePixel(x,y,ComputePixel(x,y));
  }
  printf("...Done.\n");
}

void main(){
  char filename[256],cmd[256];
  unsigned long x, y;

  // Get information from user
  _InitEasyWin();   
  SetWindowText(GetActiveWindow(),TITLE);
  GetFileName(filename,sizeof(filename),'r');
  LoadOldBMP(filename);

  // Compute
  InitializeNewBMP();
  ComputeNewBMP();

  // Save result
  GetFileName(filename,sizeof(filename),'w');
  SaveNewBMP(filename);

  // Launch Paintbrush to view the results
  strcat(cmd,"pbrush ");
  strcat(cmd,filename);
  WinExec(cmd,SW_SHOWMAXIMIZED);
  Quit("All done.","");
}
