/****
	ADDATTRC.C -- test ADDATTR.OBJ.
	R. Zigler
****/

#include <bios.h>
#include <conio.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include "addattr.h"

/**** Data Type Synonyms ****/

typedef unsigned char	BYTE;
typedef unsigned short	WORD;
typedef unsigned long	DWORD;

/**** Keyboard Macros ****/

#define KEYREAD			(bioskey(0))	/* read keyboard buffer				*/
#define KEYREADY			(bioskey(1))	/* check for key in buffer			*/
#define KEYSTATUS			(bioskey(2))	/* get k/b shift key flags			*/

/**** BIOS Video Services ****/

#define GET_ATTR			0x08				/* reads char/attr at cursor		*/
#define WRITE_MANY		0x09				/* write char/attr n times			*/
#define BIOS_VIDEO		0x10

/**** BIOS Cursor Services ****/

#define ACT_PAGE			0x00				/* active video page					*/
#define CUR_SIZE			0x01				/* set cursor size					*/
#define CUR_SET			0x02				/* set cursor position				*/
#define CUR_GET			0x03				/* read cursor type/position		*/
#define CUR_OFF			0x2000			/* turn off cursor					*/

/**** Switches for Cursor() ****/

static enum					CurSwitch		/* cursor switch						*/
								{
								OFF		,		/* switch off							*/
								ON			,		/* switch on							*/
								SAVE		,		/* save location						*/
								RESTORE	,		/* restore to location				*/
								}			;

/**** Data Macros ****/

#define LOBYTE(x)			((x) & 0x00FF)	/* get low byte of int				*/
#define HIBYTE(x)			((x) & 0xFF00)	/* get high byte of int				*/

/**** Some Background Text ****/

static char CgaText[] =
{
"Borland's Turbo C 2.0 and later compilers maintain a table of video state\r\n"
"data at the public address _video.  At offset 10 is an undocumented flag\r\n"
"that the video initialization routine sets to 1 when it detects a CGA\r\n"
"adapter, in which case direct video i/o routines in the library access\r\n"
"video memory only during the raster horizontal retrace.  That eliminates\r\n"
"the \"snow\" that plagued the earliest CGAs but slows video i/o considerably.\r\n"
"\r\n"
"Though no longer a hot topic, you can insure that your text-mode programs\r\n"
"perform well on older hardware by clearing the CGA flag.  Routines such as\r\n"
"puttext() perform noticeably better, cprintf() and cputs() enormously so.\r\n"
"If you've never had the pleasure of seeing the latter routines at work on a\r\n"
"CGA, they move text smartly until they hit the bottom of the screen.  After\r\n"
"that, their performance can only be termed \"glacial.\"  That's because the\r\n"
"video routines must read and write 2000 words of video data each time the\r\n"
"screen srolls, which results in 4000 trips through the snow-control code.\r\n"
"\r\n"
"You can step through the video initialization code by running a program in\r\n"
"Turbo Debugger's assembler mode (with the -l switch).  The last step taken\r\n"
"by the start-up code, prior to pushing the arguments to main() on the\r\n"
"stack, is video initialization.  Note that the CGA flag is not the same as\r\n"
"the documented directvideo flag, though the latter must be set to 1 (the\r\n"
"default) for the CGA flag to affect video performance.\r\n"
};

/**** The Help Screen

ͻ
                                                                      
                      A B O U T    A D D A T T R                      
                                                                      
  Ĵ1ĿĴ2Ŀ  
   AddAttr is useful for adding   You can make any number of      
   video attributes to text that  atttribute changes to any       
   you want to move directly      number of characters in the     
   into video memory.             text.                           
    
                                                                      
  Ĵ3ĿĴ4Ŀ  
   Attribute changes are marked   When AddAttr exhausts the       
   by flag characters in the      list of attributes, it uses     
   text.  The flag may be any     the last for the remainder      
   character you choose.          of the text.                    
    
                                                                      
   Use Esc to quit program; any other key to toggle this screen...    
                                                                      
ͼ

****/

enum {X1 = 5,X2 = 76,Y1 = 3,Y2 = 23};	/* window coordinates				*/

#define HELP_WIN			X1,Y1,X2,Y2		/* help window coordinates			*/

static char HelpStr[] =
{
"ͻ"
"                                                                      "
"                    ~  A B O U T    A D D A T T R  ~                    "
"                                                                      "
"  Ĵ~1~ĿĴ~2~Ŀ  "
"   AddAttr is useful for adding   You can make ~any number~ of      "
"   ~video attributes~ to text that  ~atttribute changes~ to ~any~       "
"   you want to move directly      ~number~ of ~characters~ in the     "
"   into ~video memory~.             text.                           "
"    "
"                                                                      "
"  Ĵ~3~ĿĴ~4~Ŀ  "
"   Attribute changes are marked   When AddAttr ~exhausts~ the       "
"   by ~flag characters~ in the      ~list of attributes~, it uses     "
"   text.  The flag may be ~any~     the ~last~ for the ~remainder~      "
"   ~character~ you choose.          of the text.                    "
"    "
"                                                                      "
"   Use ~Esc~ to ~quit~ program; ~any other key~ to ~remove~ this screen...    "
"                                                                      "
"ͼ"
};

#define NUM_ATTRS			46

static char					AttrList[NUM_ATTRS];

#define BUFFERSIZE		((X2-X1+1)*(Y2-Y1+1)*2)		/* screen buffers		*/

/**** Local Routines ****/

static void		pascal	Cursor			( short									);
static int		pascal	GetVideoAttr	( void									);
static int		pascal	HelpScreen		( int										);

/************************************************************************/
void main( void )
	{
	int			hlp,
					attr,
					help_attr;

#if defined(__TURBOC__)
	extern char _video;
	*((char *)&_video+10) = '\0';
#endif

	clrscr();
	attr = GetVideoAttr();					/* get current video attribute	*/
	help_attr =	((attr & 0x07) << 4)	|	/*  and invert for help window	*/
					((attr & 0x70) >> 4)	|
					( attr & 0x08)			;

	HelpScreen( help_attr | 0x8000 );	/* init help screen colors			*/
	Cursor( OFF );
	cprintf( "%s\r\n  Use Esc to quit program; F1 for info screen...",
																					CgaText );
	Cursor( ON );
	do
		{
		hlp = KEYREAD;
		if ( hlp == 0x3B00 )							/* F1								*/
			hlp = HelpScreen( help_attr );
		else if ( LOBYTE(hlp) == 0x1B )			/* Esc							*/
			break;
		else
			hlp = 0;
		}
	while ( hlp >= 0 && hlp != 0x1B );
	clrscr();
	}

/************************************************************************/
/* Cursor handling.  Call with OFF argument prior to ON, and with			*/
/* SAVE prior to RESTORE.																*/

static void pascal Cursor( short CurSwitch )
	{

	switch ( CurSwitch )
		{
		static short cshape, cpos;			/* cursor shape, position			*/

		case OFF:								/* turn off the cursor				*/
			_AH = CUR_GET;						/* get cursor shape/position		*/
			_BH = ACT_PAGE;					/* on video page 0					*/
			geninterrupt( BIOS_VIDEO );
			cshape = _CX;						/* save shape							*/
			_AH =  CUR_SIZE;
			_CX |= CUR_OFF;					/* set bit 5 in CH					*/
			geninterrupt( BIOS_VIDEO );
			break;
		case ON:									/* turn cursor back on				*/
			_AH = CUR_SIZE;
			_CX = cshape & ~CUR_OFF;		/* clear bit 5 in CH					*/
			geninterrupt( BIOS_VIDEO );
			break;
		case SAVE:								/* save cursor position				*/
			_AH = CUR_GET;
			_BH = ACT_PAGE;
			geninterrupt( BIOS_VIDEO );
			cpos = _DX;
			break;
		case RESTORE:							/* restore cursor position			*/
			_AH = CUR_SET;
			_BH = ACT_PAGE;
			_DX = cpos;
			geninterrupt( BIOS_VIDEO );
			break;
		default:
			break;
		}
	}												/**** Cursor()						****/

/************************************************************************/
/*	GetVideoAttr()																			*/
/*	Get video attribute at cursor on current page.								*/

static int pascal GetVideoAttr( void )
	{

	_AH = GET_ATTR;
	_BH = 0;
	geninterrupt( BIOS_VIDEO );
	return ( _AH );
	}

/************************************************************************/
/* Display help screen.  To initialize AttrList, pass a negative			*/
/* value in Attr (set the high bit); thereafter, pass normal attribute.	*/

static int pascal HelpScreen( register int Attr )
	{
	int		rtn = 0;

	if ( Attr < 0 )							/* init help screen attributes	*/
		{
		int i;
		int high_attr = Attr ^ 0x0F;		/* complement fgnd for highlight	*/

		/* if high_attr is invisible, toggle intensity bit						*/

		if ( ((high_attr & 0x70) >> 4) == (high_attr & 0x0F) )
			high_attr ^= 0x08;
		for ( i = NUM_ATTRS - 1 ; i >= 0 ; i-- )
			AttrList[i] = i & 1 ? (char) Attr : (char) high_attr ;
      }
	else
		{
		char	*	save_scr = NULL;
		char	*	help_scr = NULL;
		if (
			(save_scr = malloc( BUFFERSIZE )) == NULL
			||
			(help_scr = malloc( BUFFERSIZE )) == NULL
			)
			--rtn;
		else
			{
			Cursor( OFF );
			gettext( HELP_WIN, save_scr );
			AddAttr( help_scr, HelpStr, AttrList, NUM_ATTRS, Attr, '~' );
			puttext( HELP_WIN, help_scr );
			rtn = LOBYTE(KEYREAD);						/* get kb entry			*/
			puttext( HELP_WIN, save_scr );
			Cursor( ON );
			}
      if ( help_scr )
			free( help_scr );
		if ( save_scr )
			free( save_scr );
		}
	return( rtn );
	}												/**** HelpScreen()				****/

/**** EOF:  ADDATTRC.C ****/
