UNIT FileBuff;
INTERFACE
Uses
  DOS;

CONST
  MAXALLOC   = 65520;
  MAXBUFFERS = 64;
TYPE
    ListPtr = ^List;
    List = Record               { MRU Doubly Linked List         }
        Prev   : ListPtr;       { Will Chain the most recently   }
        Next   : ListPtr;       { used blocks together           }
        BuffNr : word;
      end;

    BuffArrayPtr = ^BuffArray;  { Buffer for the data            }
    BuffArray = array[0..pred(MAXALLOC)] of char;

    BlockRec = record           { Record for each block          }
        blocknr : word;         { Block stored at this location  }
        Recbuff : BuffArrayPtr; { Pointer to char buffer of data }
      end;

    BufferPtr = ^Buffer;        { arrays for storing blocks     }
    Buffer = array[0..pred(MAXBUFFERS)] of BlockRec;

    FileBuffObjPtr = ^FilebuffObj; {File buffer object definition }
    FileBuffObj = object
        Constructor  Init(FileStr   : pathstr;  ElSize    : word;
                          BuffBytes : word;     NumBuff   : word);
        Destructor   Done;
        Procedure    Read(var dta; Loc : Longint);
        Function     GetNumRecs : longint;
 PRIVATE
        FilePtr     : File;
        Head        : ListPtr;
        Tail        : ListPtr;
        NumRecs     : Longint;
        NumBuffers  : word;
        ElementSize : word;
        NumPBlock   : word;
        NumBlocks   : word;
        BuffSize    : word;
        BufferPool  : BufferPtr;
{$IFDEF DEBUG} DiskAccesses   : word;      {$ENDIF}
{$IFDEF DEBUG} ListIters      : word;      {$ENDIF}
{$IFDEF DEBUG} BlocksNotThere : word;      {$ENDIF}
{$IFDEF DEBUG} Procedure    DispDebugInfo; {$ENDIF}
        Function     BlockPresent(blocknum : word;
                                  var buffnum : word) : boolean;
        Procedure    MemoryRoutine(Alloc : boolean);
        Procedure    GetBlock(block : word; buffnum : word);
      end;
   IMPLEMENTATION

Constructor FileBuffObj.Init(FileStr   : pathstr; ElSize  : word;
                             BuffBytes : word;    NumBuff : word);
var
  TotalSize : Longint;
  i         : word;
begin
{$IFDEF DEBUG}  DiskAccesses   := 0; {$ENDIF}
{$IFDEF DEBUG}  BlocksNotThere := 0; {$ENDIF}
{$IFDEF DEBUG}  ListIters      := 0; {$ENDIF}
  ElementSize := ElSize;  NumRecs := 0;
  Head := NIL; Tail := NIL; BufferPool := NIL;
  assign(FilePtr, FileStr);
{$I-}  reset(FilePtr, ElementSize); {$I+}
  if IOResult = 0 then
  begin
    NumBuffers  := NumBuff;
    TotalSize   := FileSize(FilePtr) * ElementSize;
    if TotalSize = 0 then            {if no records        }
    begin                            {close file           }
      Close(FilePtr);
    end
    else
    begin
      BuffSize := BuffBytes;
      if (BuffSize = 0) or (BuffSize > MAXALLOC)      or
                           (BuffSize < ElementSize) then
        BuffSize := MAXALLOC;
      NumRecs   := TotalSize div ElementSize; {# recs total }
      NumPBlock := BuffSize  div ElementSize; {# recs per block }
      BuffSize  := NumPBLock * ElementSize;   {readjust buffsize }
      NumBlocks := (TotalSize + pred(BuffSize)) div BuffSize;
      if (NumBuffers > MAXBUFFERS) or (NumBuffers = 0) then
        {value was too high or all buffers wanted }
        NumBuffers := MAXBUFFERS;
      if NumBuffers > NumBlocks then
        { value is set to more than needed }
        NumBuffers := NumBlocks;
      MemoryRoutine(TRUE);
      for i := 0 to pred(NumBuffers) do
        GetBlock(i, i);
    end;
  end;
end;

Destructor FileBuffObj.Done;
begin
{$IFDEF DEBUG} DispDebugInfo; {$ENDIF}
{$I-}  Close(FilePtr); {$I+}
  {If File Was Open, then memory was allocated}
  if IOResult = 0 then
    MemoryRoutine(FALSE);
end;

Procedure FileBuffObj.Read(var dta; Loc : Longint);
var
  buffnr,
  blocknr,
  offset   : word;
begin
  if loc > NumRecs then          {if tried to read too far      }
    exit;
  dec(loc);                      {filebuff is zero based        }

  blocknr := Loc div NumPBlock;  {calculate block  number       }
  offset  := Loc mod NumPBlock;  {calculate offset into buffer  }
  if not BlockPresent(blocknr, buffnr) then
    GetBlock(blocknr, buffnr);
  move(BufferPool^[buffnr].RecBuff^[offset * ElementSize],
       dta, ElementSize);
end;

Function FileBuffObj.BlockPresent(blocknum : word;
                                  var buffnum : word) : boolean;
var
  Current : ListPtr;
  Found   : boolean;
begin
  Found := False; current := head;
  while (Current <> NIL) and not Found do
  begin
    Found := BlockNum = BufferPool^[Current^.BuffNr].blocknr;
    If not Found then
    begin
{$IFDEF DEBUG}  inc(ListIters); {$ENDIF}
      Current := Current^.Next;  { go to next LRU list element  }
    end;
  end;
  If Not Found then
  begin
{$IFDEF DEBUG}  inc(BlocksNotThere); {$ENDIF}
    Current := tail;             { make current point at end    }
  end;
  if Current <> Head then        { don't adjust list if at head }
  begin
    Current^.Prev^.Next :=
      Current^.Next;             { take current out of link }
    if current = tail then       { if tail make new tail }
      tail := current^.prev
    else  { else set next to previous    }
      Current^.Next^.Prev := Current^.Prev;

    Current^.Next := Head;       { make current head            }
    Head^.Prev    := Current;    { and set old head to current  }
    Head          := Current;    { still making current head    }
    Head^.Prev    := NIL;        { head's prev is nil           }
  end;
  BuffNum := Head^.Buffnr;       {set BuffNumber                }
  BlockPresent := Found;
end;

Function  FileBuffObj.GetNumRecs : longint;
begin  GetNumRecs := NumRecs; end;

Procedure FileBuffObj.MemoryRoutine(Alloc : boolean);
var
  Current,
  PrevPtr : ListPtr;
  i       : word;
begin
  if Alloc then
  begin
    GetMem(BufferPool, sizeof(BlockRec) * NumBuffers);
    for i := 0 to pred(NumBuffers) do
    {loop through buffer array and allocate}
    begin
      if MaxAvail >= BuffSize then
        GetMem(BufferPool^[i].RecBuff, BuffSize)
      else
      begin
        NumBuffers := i;  break; {exit out with new BuffNumber}
      end;
    end;
    New(Head); Current :=  Head;  Head^.Prev := NIL;
    PrevPtr := Head;  Head^.BuffNr := 0;
    for i := 1 to pred(NumBuffers) do
    begin
      New(Current);
      Current^.Buffnr := i;
      PrevPtr^.Next := Current;
      Current^.prev := PrevPtr;
      Prevptr := Current;
    end;
    tail := current; tail^.Next:= NIL;
  end
  else
  begin
    for i := 0 to Pred(NumBuffers) do { deallocate buffer pools }
      FreeMem(BufferPool^[i].RecBuff, BuffSize);
    { deallocate buffer array: }
    FreeMem(BufferPool, sizeof(BlockRec) * NumBuffers);
    Prevptr := Tail;           {Dispose List in reverse}
    while PrevPtr <> NIL do
    begin
      current := PrevPtr^.Prev;
      dispose(prevptr);
      Prevptr := current;
    end;
  end;
end;

Procedure FileBuffObj.GetBlock(block : word; buffnum : word);
var
  result : word;
  Pos    : longint;
  test   : longint;
begin
{$IFDEF DEBUG}  inc(DiskAccesses); {$ENDIF}
  Pos := longint(block) * longint(NumPBlock);
  seek(FilePtr, Pos);
  test := FilePos(FilePtr);
  blockread(FilePtr, BufferPool^[buffnum].RecBuff^,
            NumPBlock, result);
  BufferPool^[buffnum].blocknr := block;
end;

{$IFDEF DEBUG}
Procedure FileBuffObj.DispDebugInfo;
begin
  writeln('Number Records Total     : ', NumRecs);
  writeln('Buffer Size              : ', BuffSize);
  writeln('Number Buffers Actual    : ', NumBuffers);
  writeln('Number Blocks Total      : ', NumBlocks);
  writeln('Number Recs/Block        : ', NumPBlock);
  writeln('Record Size In File      : ', ElementSize);
  writeln('Number Disk Accesses     : ', DiskAccesses);
  writeln('Blocks Not Found in List : ', BlocksNotThere);
  writeln('List Iterations          : ', ListIters);
end;
{$ENDIF}

END.  {end UNIT FileBuff}
