{ SCRNSAVE.PAS }

{
Description:  This unit is provides a "basic black" screen saver
              for Turbo Vision applications. After waiting a
              predetermined time, the screen is blanked; if an
              event is detected by the Application program, it
              can call the ResetTimer method provided here to
              restore the screen.

Author:       Don Taylor
Date:         31 December 1993
Last Revised: 10 April 1994 23:15
Application:  IBM PC and compatibles; BP/TP 7.0; Turbo Vision 2.0
}

unit ScrnSave;

{--------------------}
     INTERFACE
{--------------------}

uses
 Objects,
 Drivers,
 App;

const
 MaxTimeout     : Word = 60;  { Maximum timeout set at 60 min }
 DefaultTimeout : Word =  5;  { Default timeout is 5 min }

type
 PScrnSaver = ^TScrnSaver;
 TScrnSaver = object(TObject)
  BlankingEnabled : Boolean;
  Blanked         : Boolean;

  constructor Init;
  procedure Update; virtual;
  procedure ResetTimer; virtual;
  procedure EnableBlanking(BDelay : Word); virtual;
  procedure DisableBlanking; virtual;
  procedure InitScreen; virtual;
  procedure UpdateScreen; virtual;
  procedure DoneScreen; virtual;

 private
  BlankingTime  : Longint;
  BlankingDelay : Word;

  procedure StartBlanking; virtual;
  procedure StopBlanking; virtual;
 end; { object }


{--------------------}
   IMPLEMENTATION
{--------------------}

uses
 Utils,
 Memory,
 Dos,
 Crt;

{----------------------}
{  TScrnSaver Methods  }
{----------------------}

{--------------------
  Init constructs a screen saver object
  and it enables the blanking process,
  setting the timeout to the default.
  Override : Seldom.
---------------------}
constructor TScrnSaver.Init;
begin
 inherited Init;
 Blanked := False;
 EnableBlanking(DefaultTimeout)
end; { TScrnSaver.Init }


{--------------------
  Update expects to be called often, by
  the application's Idle method. Update
  checks to make sure blanking is enabled,
  then it checks to see if the target
  blanking time has been met or exceeded.
  If so, it calls StartBlanking to initiate
  blanking. If blanking is already in progress,
  Update calls UpdateScreen to enable any
  special screen drawing. Override: Seldom.
---------------------}
procedure TScrnSaver.Update;
const
 KeyMask : Byte = $0F;
var
 Regs  : Registers;
begin
 if BlankingEnabled and not Blanked
  then if Now >= BlankingTime then StartBlanking;

{ Reset timer if Ctrl, Alt or Shift pressed
  - otherwise, update the screen }
 if Blanked
  then begin
        Regs.AH := $02;
        Intr($16,Regs);
        if (Regs.AL and KeyMask) <> 0
         then ResetTimer
         else UpdateScreen;
       end;
end; { TScrnSaver.Update }


{--------------------
  ResetTimer gets the current date and time
  from DOS and saves it in BlankingTime, for
  use by the Update method in determining
  when to initiate blanking. If blanking
  has been taking place, it is stopped
  by calling StopBlanking. Override: Seldom.
---------------------}
procedure TScrnSaver.ResetTimer;
begin
 if Blanked then StopBlanking;
 BlankingTime := Now + MinutesToLongint(BlankingDelay);
end; { TScrnSaver.ResetTimer }


{--------------------
  EnableBlanking enables the blanking
  countdown process, setting the timeout
  to the specified value (or to the max
  or min value, if the specified value is
  not in the range. EnableBlanking then
  resets the timer, which starts the countdown
  from scratch. Override: Seldom.
---------------------}
procedure TScrnSaver.EnableBlanking(BDelay : Word);
begin
 BlankingEnabled := True;
 if BDelay = 0
  then BlankingDelay := DefaultTimeout
  else if BDelay <= MaxTimeout
        then BlankingDelay := BDelay
        else BlankingDelay := MaxTimeout;
 ResetTimer;
end; { TScrnSaver.EnableBlanking }


{--------------------
  DisableBlanking turns off the blanking
  countdown, which effectively disables
  the Update method as well. Override: Seldom.
---------------------}
procedure TScrnSaver.DisableBlanking;
begin
 if Blanked then StopBlanking;
 BlankingEnabled := False;
end; { TScrnSaver.DisableBlanking }


{--------------------
  StartBlanking is called by Update when
  it detects the timeout has been met or
  exceeded. StartBlanking shuts down the
  Turbo Vision video manager, and then calls
  the InitScreen method to initialize the screen.
  Override: Seldom.
---------------------}
procedure TScrnSaver.StartBlanking;
begin
 if not Blanked
  then begin
        DoneVideo;     { Shut down the TV video system }
        InitScreen;    { Initialize the screen }
        Blanked := True;
       end;
end; { TScrnSaver.StartBlanking}


{--------------------
  StopBlanking is called by ResetTimer when
  blanking is taking place when a timer reset
  is activated. StopBlanking re-enables the
  Turbo Vision video manager, and then it
  restores the TV screen by requesting the
  Application to redraw it. Override: Seldom.
---------------------}
procedure TScrnSaver.StopBlanking;
begin
 DoneScreen;
 InitVideo;
 Blanked := False;
 Application^.Redraw;
end; { TScrnSaver.StopBlanking }


{--------------------
  InitScreen sets the display to its initial
  condition before starting whatever activity
  is programmed during the "blanked" period.
  The default InitScreen does nothing in addition
  to what StartBlanking did. Override: Often.
---------------------}
procedure TScrnSaver.InitScreen;
begin
end; { TScrnSaver.InitScreen }


{--------------------
  UpdateScreen performs the next action on
  the screen during the "blanked" period.
  (This may be moving a sprite, drawing a
  graphic, or whatever. Any additional
  variables needed should be added to the
  ScreenSaver object itself.) The default
  UpdateScreen does nothing. Override: Often.
---------------------}
procedure TScrnSaver.UpdateScreen;
begin
end; { TScrnSaver.UpdateScreen }


{--------------------
  DoneScreen performs any cleanup that
  needs to be done to end the "blanked" period.
  (Typically, this would be disposing of any
  objects that have been constructed.)
  The default DoneScreen does nothing.
  Override: Often.
---------------------}
procedure TScrnSaver.DoneScreen;
begin
end; { TScrnSaver.DoneScreen }


end. { unit ScrnSave }
