/******************************************************************
 * EXAMPLE.C
 *
 *	This program contrasts Bresenham's (implemented in LINE.C) 
 *	and Wu's (implemented in ANTILINE.C) line-drawing
 *	algorithms.
 *
 *	This program requires vanilla VGA support.  It goes into
 *		320x200x256 mode, sets the DAC to a gray ramp
 *		function, and draws randomly selected lines at
 *		1-second intervals until the user presses a key.
 *
 *	Invoke with the -a command-line argument to enable
 *		antialiasing of the lines.
 *	Invoke with the -g command-line argument to enable
 *		gamma correction.	
 *
 *	Copyright (C) 1993 by Nicholas Wilt.  All rights reserved.
 *****************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>

#include <dos.h>
#include <conio.h>

void
Plot(int x, int y, unsigned char clr)
{
    unsigned char *pix = MK_FP(0xa000, y*320+x);
    *pix = clr;
}

extern void antiline(int x0, int y0, int x1, int y1, int clr);
extern void drawline(int x0, int y0, int x1, int y1, int clr);

void 
usage(char *launchname)
{
    fprintf(stderr, "Usage: %s [-a] [-g]\n", launchname);
    fprintf(stderr, "\t-a: antialias.\n");
    fprintf(stderr, "\t-g: perform gamma correction.\n");
    fprintf(stderr, "\nThis program requires VGA support.\n");
    exit(0);
}

int
main(int argc, char *argv0[])
{
    int i;
    unsigned char dac[256];
    union REGS regs;
    int gamma, anti;
    void (*line)(int, int, int, int, int);
    char **argv = argv0;

    // By default, no gamma correction or antialiasing.
    gamma = anti = 0;

    argc -= 1;
    argv += 1;
    while (argc--) {
	char *arg = *argv;
	if (arg[0] != '-')
	    usage(argv0[0]);
	if (arg[1] == 'a')
	    anti = 1;
	else if (arg[1] == 'g')
	    gamma = 1;
	else
	    usage(argv0[0]);
	argv += 1;
    }

    for (i = 0; i < 256; i++) {
	double r = pow(i/255., 1/2.222);
	if (gamma)
	    dac[i] = (unsigned char) (r * 255.);
	else
	    dac[i] = (unsigned char) i;
    }

    // If antialiasing, do Wu's algorithm.  Otherwise,
    // do Bresenham's algorithm.
    line = (anti) ? antiline : drawline;

    // Enter standard VGA 320x200x256 mode.
    regs.x.ax = 0x13;
    int86(0x10, &regs, &regs);

    // Set up lookup table.
    rampdac(dac);

    randomize();
    do {
	int x0 = random(320);
	int x1 = random(320);
	int y0 = random(200);
	int y1 = random(200);
	(*line)(x0, y0, x1, y1, 255);
	delay(1000);
	(*line)(x0, y0, x1, y1, 0);
    } while (! kbhit());

    regs.x.ax = 7;
    int86(0x10, &regs, &regs);

    (void) getch();

    return 0;
}

