/*   SCREEN.C - A private-library display-control module for OS/2
* 2.x in text mode.  Function prototypes and access to the global
* data are in the header file: screen.h.
*    Note:  The cursor is hidden from the first time clearit() is
* called until the program exits, except during KeyInput()
* operations. 
*    Written for C/C++ Tools 2.01 compiler and IBM Toolkit 2.1
*          (C) Copyright Murray L. Lesser 1994
*/
#define INCL_NOPMAPI
#define INCL_VIO
#include <os2.h>
#include <conio.h>
#include <string.h>

// Global variables
    BYTE Attr[2] = {' ',0x7};   // Default: Blank, white on black
    USHORT row = 0;             // Default: Full screen window
    USHORT col = 0;
    USHORT RowTop = 0;
    USHORT ColLft = 0;
    USHORT RowBot = 24;
    USHORT ColRgt = 79;
// Local Variables
    static USHORT BufLen;       // Length of Logical Video Buffer
    static ULONG Buf16;         // (ULONG)16:16 pointer to LVB
    static PVOID BufPtr;        // 0:32 flat pointer to LVB
    static PVOID MemBase;       // Pointer to allocated memory
    static BYTE ExitFlag = 0;
    static BYTE MemFlag = 0;
    static VIOCURSORINFO CursorInfo;
/*   VIOCURSORINFO is a structure typedefined in bsesub.h, with
*     members:  USHORT   yStart;   [Cursor "start" line]
*               USHORT   cEnd;     [Cursor "stop"  line]
*               USHORT   cx;       [Cursor width (text mode = 1)]
*               USHORT   attr;     [-1 hides cursor]
*/

static void closit();              // Auto-execute at end

/*  clearit() clears portion of display bounded by Row..., Col...
*  values to Attr characters and attributes. 
*/
void clearit()
{
    if (ExitFlag == 0)  {          // First time through
        ExitFlag = 1;
        atexit(closit);
        VioGetBuf(&Buf16, &BufLen, 0);      // Set Logical Buffer
        BufPtr = (PVOID)DosSelToFlat(Buf16);  // "Thunk" pointer
        VioGetCurType(&CursorInfo, 0);        // Load CursorInfo
        CursorInfo.attr = -1;                 //  and hide cursor
        VioSetCurType(&CursorInfo, 0);
    }
    VioScrollUp(RowTop,ColLft,RowBot,ColRgt,0xffff,Attr,0);
}
/*  scrollit() scrolls the defined screen area up one row,
*  clearing the bottom row to the value defined in Attr. 
*/
void scrollit()
{
    VioScrollUp(RowTop,ColLft,RowBot,ColRgt,1,Attr,0);
}
/*  pwindow(*string), cwindow(*string), and dwindow(*string),
*  write string to the window defined by the Row...,Col...
*  values, starting at position row,col.  pwindow() sets col to
*  ColLft before writing.  cwindow() sets col to center the
*  string between ColLft and ColRgt.  dwindow() does the actual
*  writing.  If row is less than RowTop, it is changed to RowTop.
*  If row is below RowBot, the window is scrolled up and the
*  string is written in the cleared row.  After writing, row will
*  be increased by one, and col will have the value: col +
*  strlen(string).  If the string will not fit in the row, the
*  error return is '1' and the string is not written.
*/
int dwindow(char *string)
{
    if (col + strlen(string) - 1 > ColRgt)
        return 1;                       // String too long to fit
    if ((short)row < RowTop)            // Is row in window?
        row = RowTop;
    else if (row > RowBot) {
        scrollit();
        row = RowBot;
    }
    VioWrtCharStr(string,strlen(string),row++,col,0);
    col += strlen(string);
    return 0;
}
int pwindow(char *string)
{
    col = ColLft;
    return dwindow(string);
}
int cwindow(char *string)
{
    col = (ColLft + ColRgt + 1 - strlen(string))/2;
    if ((short)col < (short)ColLft)
        return 1;
    return dwindow(string);
}
/*  KeyInput() accepts keyed input, displaying values starting at
*  row-1,col and capturing input in the string array.  Length of
*  input string is limited by MaxLen, or the length between col
*  and ColRgt, whichever is least.  Row is initialized to row-1;
*  after return it is restored to its original value.
*/
void KeyInput(char *string, short MaxLen)
{
    char c;
    short i;
// Setup
    CursorInfo.attr = 0;               // "Unhide" cursor
    VioSetCurType(&CursorInfo, 0);
    row -= 1;                          // Move row up a line
    i = ColRgt - col + 1;              // Maximum value of MaxLen
    if (i <= MaxLen)                   //   to fit in window
        MaxLen = i;
    else                               // Else mark end of input
        VioWrtNChar("]",1,row,col+MaxLen,0);    // line with "]"
/* Read in the line */
    for (i = 0; i <= MaxLen; i++) {
        VioSetCurPos(row, col+i, 0);   // Show cursor for input
        c = getch();
        if (c == '\x0d') {             // If "Enter" key, input
            string[i] = '\0';          //   done; terminate
            CursorInfo.attr = -1;      //   string, hide cursor
            VioSetCurType(&CursorInfo, 0);
            row += 1;                  // Restore original row
            return;                    //  and return to caller
        }
        else if (c == '\x08')  {       // If "Backspace", and not
            if (i != 0)                //   row start, clear char
                VioWrtNChar(" ",1,row,col + --i,0);
            i -= 1;                    // Restore "i" counter
        }       
        else if (c == '\x1b')  {              // If "Escape" key'
            VioWrtNChar(" ",MaxLen,row,col,0);  //  clear line
            i = -1;                           //  and start over
        }
        else if (c == '\0')  {        // If old "Extended code",
            getch();                  //   get rid of it
            i -= 1;                   //   and restore i
        }
        else if (i == MaxLen)  {      // If past end of row,
            DosBeep(1380,500);        //    "beep"
            i -= 1;                   //    and restore i
        }
        else  {                       // Real input character!
            VioWrtNChar(&c,1,row,col+i,0);
            string[i] = c;
        }
    }
}
/*  putscrn() copies the logical video buffer to a different
*  memory page.  getscrn() restores the original saved copy. 
*/
void putscrn()
{
    if (MemFlag == 0) {                 // Get space for LVB copy
        DosAllocMem(&MemBase,(ULONG)BufLen,
                        PAG_COMMIT | PAG_WRITE);
        MemFlag = 1;
    }
    memcpy(MemBase,BufPtr,(size_t)BufLen);      // Copy LVB
}
void getscrn()
{
    memcpy(BufPtr,MemBase,(size_t)BufLen);
    VioShowBuf(0, BufLen, 0);
}
//  Replace divots on program exit:
void closit()             // Auto-called when program terminates
{
     if (MemFlag != 0)    // If memory space has been allocated,
         DosFreeMem(MemBase); //   restore it.
     Attr[1] = 0x7;       // Clear bottom row to default values
     RowTop = RowBot = 24;
     ColLft = 0;
     ColRgt = 79;
     clearit();
     CursorInfo.attr = 0;               // "Unhide" cursor and
     VioSetCurType(&CursorInfo, 0);
     VioSetCurPos(23, 0, 0);	        // set at row 23, col 0
}
