;-----< serial.asm >----------------------------------------------
;
; Description: C function that reads a serial number from the boot
;               sector of any disk that is formatted with DOS 4, 5
;               or 6 by calling DOS INT 25h.
;
;                   Example C declaration:
;
;                       extern long int serial_num(int drive);
;
;
;                   Example call from C:
;
;                       rval = serial_num(drive);
;
;                   Where the function serial_num returns a
;                   long int and drive is an int specifying the
;                   drive to read. Drive can only range from 0 to
;                   26 for drives a-z. If an error occurs, the
;                   returned value will be 0. This can signify a
;                   DOS read error or that the disk was not
;                   formatted by DOS 4, 5 or 6 so no serial
;                   number existed.
;
;                   Drives are specified as 0 = A, 1 = B, etc.
;
;   Written and tested with QuickC 2.5 and MASM 5.1
;-----------------------------------------------------------------
;
; Version: 1.01   8/30/93
;
; By: Steven C. Hageman  8/14/93
;
;-----------------------------------------------------------------

.MODEL SMALL, C         ;Directive for C Procedure call

;-- Start of data area -------------------------------------------
.DATA
sec_buf DB 2048 DUP(?)  ;Plenty of room for any drive

;-- Start of code area -------------------------------------------
.CODE

serial_num  PROC    drive:WORD  ;Drive number: 0=A, 1=B, etc.

;-- Remember: C procedures should not change SI,DI,CS,DS
;-- or SS All others are fair game!

;-- Set up sector buffer address in DS:BX, DS is already
;-- set on entry
lea bx,sec_buf      ;bx gets offset of buffer

;-- Set up other parameters for INT 25h
mov al,byte ptr (drive) ;Set drive to get S/N from
mov cx,1            ;Number of sectors to read
mov dx,0            ;First sector to read

int 25h             ;INT 25h, make sure to pop flags after
pop cx              ;Discard flags into cx

jc @@error          ;If carry set, then INT 25h failed

;-- Byte 8 of boot sector has the DOS version number.
;-- Check to be sure that this number is 4 or 5 for DOS 4
;-- or 5 or 6. DOS 6 currently (strangely enough) puts a 5
;-- in byte 8, identifying it as a DOS 5 disk, this may
;-- need to be changed for a future DOS release!!!!!
cmp [sec_buf+8],034h    ;Check if DOS 4
je @@get_sn             ;If DOS 4, jump to get serial #

cmp [sec_buf+8],035h    ;Check if DOS 5 (or 6)
jne @@error             ;If not DOS 5 (or 6), jump error

@@get_sn:
;-- Boot sector bytes 39-42 have the disk serial number.
;-- Load these bytes to DX, AX to return long int.
mov dh,[sec_buf+42]
mov dl,[sec_buf+41]
mov ah,[sec_buf+40]
mov al,[sec_buf+39]
jmp short xit       ;Done, exit function

@@error:
;-- Error occurred, set serial number to 0 for error code
xor dx,dx
xor ax,ax

xit:
;-- Restore pushed reg's, return to calling function
ret                 ;Return to calling C function

serial_num  ENDP

END     ;-- End of C procedure -----------------------------------
