unit SpinBttn;

interface {=== SpinBttn Unit ===}

uses
  Views, Dialogs, Objects, Drivers, Validate, RJInput;

type
  PSpinInputLine = ^TSpinInputLine;
  TSpinInputLine = object( TRJInputLine )
    procedure HandleEvent( var Event : TEvent ); virtual;
    function Valid( Command : Word ) : Boolean; virtual;
  end;

  PSpinButton = ^TSpinButton;
  TSpinButton = object( TGroup )
  private
    InputLine : PSpinInputLine;
    Value, Min, Max, Step : Longint;
    AllowEdit : Boolean;    { Allow keyboard editing of numbers }

  public
    constructor Init( var Bounds : TRect;
                      LowLimit, UpLimit, StepSize : Longint;
                      EditFlag : Boolean );
    function DataSize : Word; virtual;
    procedure Decrement; virtual;
    procedure Draw; virtual;
    procedure GetData( var Rec ); virtual;
    function GetValue : Longint; virtual;
    procedure HandleEvent( var Event : TEvent ); virtual;
    procedure Increment; virtual;
    procedure InitInputLine;
    procedure Synchronize; virtual;
    procedure SetData( var Rec ); virtual;
    procedure SetRange( LoVal, HiVal : Longint ); virtual;
    procedure SetStep( AStep : Longint ); virtual;
    procedure SetValue( N : Longint ); virtual;
    procedure SetState( AState : Word; Enable : Boolean ); virtual;
  end;

implementation {=== SpinBttn Unit ===}

{============================}
{== TSpinInputLine Methods ==}
{============================}

procedure TSpinInputLine.HandleEvent( var Event : TEvent );
begin
  if ( Event.What = evKeyDown ) and
     (( Event.KeyCode = kbUp ) or ( Event.KeyCode = kbDown )) then
  begin
    PSpinButton( Owner )^.Synchronize;
    if Event.KeyCode = kbUp then
      PSpinButton( Owner )^.Increment
    else
      PSpinButton( Owner )^.Decrement;
    ClearEvent( Event );
  end;
  inherited HandleEvent( Event );
end; {= TSpinInputLine.HandleEvent =}

function TSpinInputLine.Valid( Command : Word ) : Boolean;
begin
  PSpinButton( Owner )^.Synchronize;   { Sync Value & InputLine }
  Valid := inherited Valid( Command );
end; {= TSpinInputLine.Valid =}

{=========================}
{== TSpinButton Methods ==}
{=========================}

constructor TSpinButton.Init(var Bounds : TRect;
                             LowLimit, UpLimit, StepSize : Longint;
                             EditFlag : Boolean );
begin
  Bounds.B.Y := Bounds.A.Y + 1;    { Make sure only 1 line high }
  inherited Init( Bounds );
            { Allow processing all mouse clicks, even selection }
  Options := Options or OfFirstClick;
  AllowEdit := EditFlag;
  InitInputLine;
  if InputLine <> nil then
    Insert( InputLine );
                         { Set disable state based on AllowEdit }
  InputLine^.SetState( sfDisabled, not AllowEdit );
  SetRange( LowLimit, UpLimit );
  SetStep( StepSize );
end; {= TSpinButton.Init =}

function TSpinButton.DataSize : Word;
begin
  DataSize := SizeOf( Value );
end; {= TSpinButton.DataSize =}

procedure TSpinButton.Decrement;
begin
  if ( Value > Max ) or ( Value - Step < Min ) then
    SetValue( Max )
  else
    SetValue( Value - Step );
end; {= TSpinButton.Decrement =}

procedure TSpinButton.Draw;
var
  Color : Byte;
  B : TDrawBuffer;
begin
  inherited Draw;             { Make sure Input Line gets drawn }
  if GetState( sfSelected ) then
    Color := GetColor( 12 )      { Get Color of Selected Button }
  else if GetState( sfDisabled ) then
    Color := GetColor( 13 )      { Get Color of Disabled Button }
  else
    Color := GetColor( 10 );       { Get Color of Normal Button }
  MoveStr( B, ' ' + Chr( 25 ) + Chr( 24 ) + ' ', Color );
  WriteBuf( Size.X - 4, 0, 5, 1, B );
end;

procedure TSpinButton.GetData( var Rec );
begin
  Longint( Rec ) := Value;
end; {= TSpinButton.GetData =}

function TSpinButton.GetValue : Longint;
begin
  GetValue := Value;
end; {= TSpinButton.GetValue =}

procedure TSpinButton.HandleEvent( var Event : TEvent );
var
  NumStr : String;
  astate : Boolean;

  procedure ProcessMouseClick;
  var
    Mouse : TPoint;
  begin
    MakeLocal( Event.Where, Mouse );
    if Mouse.X >= Size.X - 4 then
      Synchronize;            { Synchronize if button pressed }
    if ( Mouse.X = Size.X - 4 ) or ( Mouse.X = Size.X - 3 ) then
      Decrement
    else if (Mouse.X = Size.X - 2) or (Mouse.X = Size.X - 1) then
      Increment;
  end; {= ProcessMouseClick =}

begin {= TSpinButton.HandleEvent =}
  inherited HandleEvent( Event );
  case Event.What of
    evMouseDown:
    begin
      ProcessMouseClick;
      repeat
        if event.What = evMouseAuto then
          ProcessMouseClick;
      until not MouseEvent( event, evMouseMove + evMouseAuto );
    end; { evMouseDown }

    evKeyDown :
    begin          { InputLine disabled, must handle kbd myself }
      case Event.KeyCode of
        kbRight, kbUp :
          Increment;

        kbLeft, kbDown :
          Decrement;

        kbHome :                           { Set to max value }
          SetValue( Max );

        kbEnd :                            { Set to min value }
          SetValue( Min );
      end; { case }
    end; { evKeyDown : }

  end; { case Event.What }
end; {= TSpinButton.HandleEvent =}

procedure TSpinButton.Increment;
begin
  if ( Value < Min ) or ( Value + Step > Max ) then
    SetValue( Min )
  else
    SetValue( Value + Step );
end; {= TSpinButton.Increment =}

procedure TSpinButton.InitInputLine;
var
  R : TRect;
begin
  GetExtent( R );
  R.B.X := R.B.X - 4;  { Width of InputLine is 4 short of whole }
                        { Lenght = 11 chars (eg. '-2147483648') }
  InputLine := New( PSpinInputLine, Init( R, 11 ) );
  InputLine^.GrowMode := gfGrowHiX;
end; {= TSpinButton.InitInputLine =}

procedure TSpinButton.Synchronize;
var
  Num : Longint;
  CD : Integer;
  S : String;
begin
  if AllowEdit then
  begin                    { Check first if user can edit value }
    InputLine^.GetData( S );         { Get value from InputLine }
    Val( S, Num, CD );                      { Convert to Number }
    if Num <> Value then            { If different, synchronize }
      SetValue( Num );
  end;
end; {= TSpinButton.Synchronize =}

procedure TSpinButton.SetData( var Rec );
var
  TmpStr : String;
begin
  SetValue( Longint( Rec ) );
end; {= TSpinButton.SetData =}

procedure TSpinButton.SetRange( LoVal, HiVal : Longint );
begin
  Min := LoVal;
  Max := HiVal;           { Create new validator with new range }
  InputLine^.SetValidator( New( PRangeValidator,
                                Init( LoVal, HiVal ) ) );
end; {= TSpinButton.SetRange =}

procedure TSpinButton.SetStep( AStep : Longint );
begin
  Step := AStep;
end; {= TSpinButton.SetStep =}

procedure TSpinButton.SetValue( N : Longint );
var
  ValueStr : string[ 12 ];
begin
  Value := N;
  Str( Value, ValueStr );
  InputLine^.SetData( ValueStr );
end; {= TSpinButton.SetValue =}

procedure TSpinButton.SetState( AState : Word; Enable : Boolean );
begin
  inherited SetState( AState, Enable );
  InputLine^.SetState( sfSelected, TRUE );          { Select it }
  if not Enable then
    InputLine^.SetState( sfDisabled, TRUE )
  else
    InputLine^.SetState( sfDisabled, not AllowEdit );
  DrawView;
end; {= TSpinButton.SetState =}

end. {=== SpinBttn Unit ===}
