#ifndef DISKETTE_HPP
#define DISKETTE_HPP

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE.HPP - IBM PC Diskette Services Class Module Header File
//
//  Version:    0.9     *** NOT FOR GENERAL RELEASE ***
//
//  History:    91/07/31 - Created.
//
//  Compiler:   Microsoft C/C++ V7.0
//
//  Author:     Ian Ashdown, P.Eng.
//              byHeart Software Ltd.
//              620 Ballantree Road
//              West Vancouver, B.C.
//              Canada V7S 1W3
//              Tel. (604) 922-6148
//              Fax. (604) 987-7621
//
//  Copyright:  Public Domain
//
//////////////////////////////////////////////////////////////////////////

// .... INCLUDE FILES

#include "general.hpp"

// .... DEFINITIONS

const int FD_INTERRUPT = 0x13;  // Diskette Services interrupt number

// System type constants

const int FD_XT_CLASS = 0x00;   // XT-class system
const int FD_AT_CLASS = 0x01;   // AT-class system

// System identification byte constants

const byte FD_PC = 0xff;        // PC
const byte FD_XE = 0xfe;        // XT (early models)
const byte FD_JR = 0xfd;        // PC Junior
const byte FD_AT = 0xfc;        // AT, PC/XT-286, PS/2 Models 50 and 60
const byte FD_XL = 0xfb;        // XT (later models)
const byte FD_30 = 0xfa;        // PS/2 Model 30
const byte FD_CO = 0xf9;        // PC Convertible
const byte FD_PS = 0xf8;        // PS/2 Models 70 and 80

// In some BIOS implementations, a Diskette Services request may fail
// due to the diskette motor being off when the request is made.  The
// caller should reset the diskette drive and retry three times to
// ensure that the error is valid.

const int FD_MAXTRY = 4;        // Maximum number of request tries

// Maximum number of attempts to allocate a suitably aligned sector
// transfer buffer (see Diskette::allocate_buffer() for details).

const int FD_MAXBUFTRY = 32;

// BIOS Diskette Service error codes

const int FD_OK =  0x00;        // No error
const int FD_IFR = 0x01;        // Invalid function request
const int FD_NAM = 0x02;        // Address mark not found
const int FD_WPD = 0x03;        // Write-protected disk
const int FD_SNF = 0x04;        // Sector not found
const int FD_REM = 0x06;        // Diskette removed
const int FD_DOR = 0x08;        // DMA overrun
const int FD_DBE = 0x09;        // DMA boundary error
const int FD_UMT = 0x0c;        // Media type not available
const int FD_CRC = 0x10;        // Bad cyclic redundancy check
const int FD_DCF = 0x20;        // Diskette controller failed
const int FD_BSK = 0x40;        // Seek failed
const int FD_TMO = 0x80;        // Time-out

// Sector sizes

const int FD_BPS_128 = 0x00;    // 128 bytes per sector
const int FD_BPS_256 = 0x01;    // 256 bytes per sector
const int FD_BPS_512 = 0x02;    // 512 bytes per sector
const int FD_BPS_1024 = 0x03;   // 1024 bytes per sector

// Change line support codes

const int FD_NODRIVE = 0x00;    // No drive installed
const int FD_NOCHANGE = 0x01;   // Diskette drive with no change line
const int FD_CHANGE = 0x02;     // Diskette drive with change line

// Drive identifiers

const int FD_DRV_360 = 0x01;    // 5.25", 360K
const int FD_DRV_120 = 0x02;    // 5.25", 1.2M
const int FD_DRV_720 = 0x03;    // 3.5", 720K
const int FD_DRV_144 = 0x04;    // 3.5", 1.44M
const int FD_DRV_288 = 0x05;    // 3.5", 2.88M

// MS-DOS diskette identifiers

const int FD_DSK_320 = 0x00;    // 5.25", 320K
const int FD_DSK_360 = 0x01;    // 5.25", 360K
const int FD_DSK_120 = 0x02;    // 5.25", 1.2M
const int FD_DSK_720 = 0x03;    // 3.5", 720K
const int FD_DSK_144 = 0x04;    // 3.5", 1.44M
const int FD_DSK_288 = 0x05;    // 3.5", 2.88M

// Diskette controller data transfer rates

const int FD_RATE_250 = 0x00;   // 250 kbps
const int FD_RATE_300 = 0x01;   // 300 kbps
const int FD_RATE_500 = 0x02;   // 500 kbps

// .... STRUCTURE DECLARATIONS

struct fd_dprm          // Diskette Parameters Table
{
  byte scmd_1;          // First byte of diskette controller Specify
                        // command, where:
                        //
                        //   Bits 7 - 4 = step rate interval
                        //   Bits 3 - 0 = head unload time
                        //
                        // MS-DOS default values for various drive/media
                        // combinations are:
                        //
                        //   Media Type     Drive Type     Value
                        //   -----------------------------------
                        //   360K           360K           0xdf
                        //   360K           1.2M           0xdf
                        //   1.2M           1.2M           0xdf
                        //   720K           720K           0xaf
                        //   720K           1.44M          0xaf
                        //   1.44M          1.44M          0xaf
                        //
                        // Actual times depend on the selected data rate.
                        // They are (in milliseconds):
                        //
                        //             Step Rate        Head Unload
                        //   Value   500K 300K 250K    500K 300K 250K
                        //   ----------------------------------------
                        //   0x01     16   27   32      16   27   32
                        //   0x02     15   25   30      32   53   64
                        //   0x03     14   23   28      48   80   96
                        //   ....     ..   ..   ..      ..   ..   ..
                        //   0x0e      2  3.3    4     224  373  448
                        //   0x0f      1  1.7    2     240  400  480
                        //
                        // If another read or writer command is issued 
                        // before the head unload timer expires, the head
                        // unload operation is skipped.
                        //
  byte scmd_2;          // Second byte of diskette controller Specify
                        // command, where:
                        //
                        //   Bits 7 - 1 = head load time
                        //
                        // MS-DOS default value is 0x01.
                        //
                        // Actual time depends on the selected data rate.
                        // It is (in milliseconds):
                        //
                        //             Head Load
                        //   Value   500K 300K 250K
                        //   ----------------------
                        //   0x01      2  3.3    4
                        //   0x02      4  6.7    8
                        //   0x03      6   10   12
                        //   ....    ...  ...  ...
                        //   0x7e    252  420  504
                        //   0x7f    254  423  508
                        //
                        // Heads are loaded when the motor is started.
                        // However, since the motor turn-on delay is
                        // much longer, the head load time is not really
                        // needed.
                        //
                        //   Bit 0 = Non-DMA mode flag (always 0).
                        //
                        // The diskette controller always operates in
                        // its DMA mode.
                        //
  byte mtd_time;        // Motor turn-off delay (amount of time in timer
                        // ticks [18.2 ticks per second] that diskette
                        // DSR waits before turning off an active drive
                        // motor).  MS-DOS default value is 0x25 (about
                        // two seconds).
                        //
  byte bps_code;        // Bytes per sector code.  Legal values are:
                        //
                        //   0x00 -  128 bytes/sector
                        //   0x01 -  256 bytes/sector
                        //   0x02 -  512 bytes/sector (MS-DOS default)
                        //   0x03 - 1024 bytes/sector
                        //
  byte sect_trk;        // Number of sectors per track.  MS-DOS default
                        // values are:
                        //
                        //   0x08 - 8 sectors (320K 5.25")
                        //   0x09 - 9 sectors (360K 5.25"and 720K 3.5")
                        //   0x0f - 15 sectors (1.2 MB 5.25")
                        //   0x12 - 18 sectors (1.44 MB 3.5")
                        //
                        // All other values up to 255 are accepted,
                        // although the diskette controller and drive
                        // limit the number of sectors per track that can
                        // be physically written to any particular
                        // diskette type.
                        //
  byte gap_len;         // Gap 3 length (number of bytes that the
                        // voltage-controlled oscillator sync signal will
                        // stay low after two CRC bytes durng read or
                        // write commands).  MS-DOS default values are
                        // 0x1b for 1.2M media in a 1.2M drive or 1.44M
                        // media in a 1.44M drive, and 0x2a for all other
                        // media/drive combinations.
                        //
  byte data_len;        // Data length.  Since the bytes per sector
                        // field is nonzero, this member is meaningless
                        // and is set to 0xff.
                        //
  byte fg_len;          // Gap 3 length for format (length of gap to
                        // maintain between sectors when formatting).
                        // MS-DOS default values are:
                        //
                        //   Media Type   Value
                        //   ------------------
                        //   360K/720K    0x50
                        //   1.2M         0x54
                        //   1.44M        0x6c
                        //
  byte ff_byte;         // Format fill byte.  MS-DOS default value is
                        // is 0xf6.
                        //
  byte hs_time;         // Head settle time (amount of time in milli-
                        // seconds diskette DSR must wait for heads to
                        // settle after doing a seek operation).  MS-DOS
                        // default values are 0x0f for 3.5" and 1.2 MB
                        // 5.25" drives and 0x14 to 0x19 for 360K 5.25"
                        // drives.
                        //
  byte ms_time;         // Motor start time (amount of time in 1/8
                        // second increments that diskette DSR must wait
                        // for motor to come up to speed before doing an
                        // I/O operation).  MS-DOS default value is 0x08.
};

struct fd_afld          // Address Field Table element
{
  byte track_num;       // Track number
  byte head_num;        // Head number
  byte sector_num;      // Sector number
  byte sector_size;     // Sector size indicator.  May be one of:
                        //
                        //   0x00 - 128 bytes/sector
                        //   0x01 - 256 bytes/sector
                        //   0x02 - 512 bytes/sector
                        //   0x03 - 1024 bytes/sector
};

// .... CLASS DECLARATIONS

class Diskette                  // IBM PC Diskette Services
{
  private:

  protected:

    // Data

    boolean obj_flag;           // Object status flag

    byte drive_num;             // Drive number (0 or 1)
    byte drive_error;           // Drive error code status
    byte max_track;             // Maximum usable track number
    byte max_sector;            // Maximum usable sector number
    byte bps_code;              // Number of bytes per sector
    byte dummy_1;               // Ensure WORD alignment

    char _far *st_bufp;         // Sector transfer buffer pointer

    int cl_support;             // Change line support
    int drive_type;             // Drive type

    // Diskette Parameters Table pointers

    fd_dprm _far *curr_dptp;    // Current table pointer
    fd_dprm _far *prev_dptp;    // Previous table pointer

    int system_type;            // System type

    // Functions

    boolean allocate_buffer();
    boolean read_drive_parms();
    boolean set_diskette_type(int);

    int get_change_support();

  public:

    // Functions

    Diskette(int);

    ~Diskette();

    boolean detect_change();
    boolean format_track(int, int, fd_afld _far *);
    boolean install_table(fd_dprm _far *);
    boolean obj_status();
    boolean read_sectors(int, int, int, int, char _far *);
    boolean reset_drive();
    boolean set_media_type(int);
    boolean support_change();
    boolean write_sectors(int, int, int, int, char _far *);
    boolean verify_sectors(int, int, int, int, fd_afld _far *);

    int get_status();
    int sectors();
    int tracks();

    void remove_table();
};

// .... INLINE FUNCTIONS

//////////////////////////////////////////////////////////////////////////
//
//  DISKETTE::OBJ_STATUS - Return Object Status
//
//  Purpose:    To return the object status.
//
//  Setup:      inline boolean Diskette::obj_status()
//
//  Result:     The object status flag "obj_flag" is returned.  It is
//              TRUE if the drive was successfully opened; otherwise it
//              is FALSE.
//
//////////////////////////////////////////////////////////////////////////

inline boolean Diskette::obj_status()
{
  return (obj_flag);
}

#endif

