#include <stdio.h>
#include <malloc.h>     /* Required for "_fcalloc" and "_ffree" */
#include <memory.h>     /* Required for "_fmemcpy"              */
#include <dos.h>        /* Required for "int86" and "int86x"    */

/* Microsoft C/C++ V7.0 compiler-specific functions:            */
/*                                                              */
/*   _fcalloc   - allocate initialized far memory               */
/*   _ffree     - release far memory allocated with _fcalloc    */
/*   _fmemcpy   - copy bytes between far memory buffers (use    */
/*                FOR loop to copy byte by byte if equivalent   */
/*                function not available)                       */
/*   int86      - execute 80x86 interrupt routine               */
/*   int86x     - execute 80x86 interrupt routine (far data)    */

#define FALSE           0
#define TRUE            1

/* NOTE: This example code assumes that a 1.2M 5.25" MS-DOS     */
/*       diskette is to be read.  The Diskette Parameters Table */
/*       (not shown) must be modified if non-standard diskettes */
/*       are to be read, written, formatted or verified.        */

#define SECTOR_SIZE     512     /* 512 bytes per sector         */
#define TRACK_SIZE      15      /* 15 sectors per track         */
#define MAX_BUFTRY      32      /* Buffer allocation attempts   */

typedef unsigned char byte;
typedef unsigned long dword;

static char _far *fd_alloc_buffer(void);

/* FD_READ_SECTORS - Read Diskette Sectors                      */
/*                                                              */
/* This function will read diskettes with 128, 256, 512 and     */
/* bytes per sector and any number of sectors per track if      */
/* SECTOR_SIZE and TRACK_SIZE are defined correctly for the     */
/* diskette and the Diskette Parameter Table (not shown) has    */
/* been properly initialized.                                   */

int fd_read_sectors
(
  byte drive_num,       /* Drive number (0 or 1)                */
  byte num_sectors,     /* Number of sectors to be read         */
  byte track_num,       /* Track number                         */
  byte sector_num,      /* Starting sector number               */
  byte head_num,        /* Head number (0 or 1)                 */
  char _far *bufp       /* User buffer pointer                  */
)
{
  int i;                /* Scratch counter                      */
  char _far *tbp;       /* Transfer buffer pointer              */
  union REGS regs;      /* 80x86 register values                */
  struct SREGS sregs;   /* 80x86 segment register values        */

  /* Allocate transfer buffer (normally done only once during   */
  /* program initialization)                                    */

  if ((tbp = fd_alloc_buffer()) == NULL)
    return (FALSE);

  for (i = 0; i < 4; i++)       /* Try four times               */
  {
    regs.h.ah = 0x02;   /* Specify "Read Diskette Sectors" fcn  */
    regs.h.dl = drive_num;
    regs.h.al = num_sectors;
    regs.h.ch = track_num;
    regs.h.cl = sector_num;
    regs.h.dh = head_num;
    regs.x.bx = FP_OFF(tbp);
    sregs.es = FP_SEG(tbp);
    int86x(0x13, &regs, &regs, &sregs);   /* Call interrupt fcn */

    if (regs.x.cflag == 0)      /* Successful read ?            */
    {
      /* Copy transfer buffer to user buffer                    */
      _fmemcpy(bufp, tbp, (size_t) (num_sectors * SECTOR_SIZE));
      return (TRUE);
    }

    regs.h.ah = 0x00;   /* Specify "Reset Diskette Drive" fcn   */
    regs.h.dl = drive_num;
    int86(0x13, &regs, &regs);  /* Call interrupt fcn           */
  }
  return (FALSE);
}

/* FD_ALLOC_BUFFER - Allocate Transfer Buffer                   */
/*                                                              */
/* Dynamically allocate transfer buffer from the far heap,      */
/* sized to hold an entire track of data.  The buffer is        */
/* aligned such that it does not cross a physical 64K page      */
/* boundary.  If a suitably-aligned buffer cannot be found, a   */
/* NULL pointer is returned.                                    */

static char _far *fd_alloc_buffer(void)
{
  int i;                        /* Scratch counter              */
  int j;                        /* Scratch counter              */
  char _far *sp;                /* Buffer start pointer         */
  char _far *ep;                /* Buffer end pointer           */
  char _far *bufp = NULL;       /* Allocated buffer pointer     */
  char _far *cbp[MAX_BUFTRY];   /* Candidate buffer pointers    */
  dword s_segment;              /* Buffer start segment         */
  dword e_segment;              /* Buffer end segment           */

  for (i = 0; i < MAX_BUFTRY; )
  {
    /* Allocate candidate buffer from far memory                */

    if ((cbp[i] = (char _far *) _fcalloc(SECTOR_SIZE * TRACK_SIZE,
        sizeof(char))) == NULL)
    {
      i++;      /* Increment count to free buffer               */
      break;
    }

    /* Initialize start and end buffer pointers                 */

    sp = cbp[i];
    ep = cbp[i] + SECTOR_SIZE * TRACK_SIZE - 1;

    /* Determine buffer normalized start and end addresses      */

    s_segment = ((((dword) FP_SEG(sp)) << 4) + 
                   (dword) FP_OFF(sp)) >> 16;
    e_segment = ((((dword) FP_SEG(ep)) << 4) + 
                   (dword) FP_OFF(ep)) >> 16;

    /* Check to see whether buffer crosses 64K page boundary    */

    if (s_segment == e_segment)
    {
      bufp = cbp[i];    /* Success                              */
      break;
    }

    i++;        /* Increment buffer counter                     */
  }

  for (j = i - 1; j >= 0; j--)  /* Free other allocated buffers */
    _ffree(cbp[j]);

  return (bufp);
}
