unit  HiMem;

{
  Copyright 1990, Tim Farley, All Rights Reserved.
}

interface

uses Dos;          { For Registers data type, Intr() and GetIntVec() }

const
  XMSpresent : boolean = FALSE;
  GotHMA     : boolean = FALSE;

function  HMAMemAvail : word;

procedure HMAGetMem(  var p : pointer; size : word );

procedure HMAMark(    var p : pointer );

procedure HMARelease( var p : pointer );

implementation

var
  XMSentry : procedure;      { procedure pointer to XMS call interface }

type
  XMSParams = record
    case  integer  of
      0 : ( AX : word; BX : word; DX : word );
      1 : ( AL : byte; AH : byte; BL : byte; BH : byte; DL : byte; DH : byte );
      2 : (
      Success  : boolean;
      FnNum    : byte;
      MovePtr  : pointer     { for function 0Bh only, not used here }
      );
    end;  { XMSParams }

procedure  XMSFunction( var Params : XMSParams );
  begin
    {
      Load up AX, BX and DX with our parameters.
      For function 0Bh, use DS and SI instead of DX and BX.
    }
    inline(
    $1E/               {       push ds           }
    $C4/$7E/<Params/   {       les  di,[Params]  }
    $26/$8B/$05/       {       mov  ax,es:[di]   }
    $26/$8B/$5D/$02/   {       mov  bx,es:[di+2] }
    $26/$8B/$55/$04/   {       mov  dx,es:[di+4] }
    $80/$FC/$0B/       {       cmp  ah,0bh      ; fn 0bh is special }
    $75/$04/           {       jne  short normal }
    $8B/$F3/           {       mov  si,bx        }
    $8E/$DA            {       mov  ds,dx        }
                       { normal:                 }
    );
    {
      Call the XMS driver via a far pointer
    }
    XMSentry;
    {
      Get our results from AX, BX and DX and put them in the param record
    }
    inline(
    $26/$89/$05/       {       mov  es:[di],ax   }
    $26/$89/$5D/$02/   {       mov  es:[di+2],bx }
    $26/$89/$55/$04/   {       mov  es:[di+4],dx }
    $1F                {       pop  ds           }
    );
  end;   { XMSFunction }

procedure  DetectXMS;
  var
    regs : registers;
    vect : pointer;
  begin
    GetIntVec( $2F, vect );       { This check is required for DOS 2.x: the }
    if  ( vect <> NIL )  then     { INT 2Fh vector might not be initialized! }
      begin
        regs.AX := $4300;         { Detect if XMS is present }
        Intr( $2F, regs );
        XMSpresent := ( regs.AL = $80 );
        if  XMSpresent  then
          begin
            regs.AX := $4310;     { Get XMS control function entry point }
            Intr( $2F, regs );
            pointer( @XMSentry ) := Ptr( regs.ES, regs.BX );
          end;
      end;
  end;   { DetectXMS }

const
  HMAGrabbed : boolean = FALSE;
  A20Enabled : boolean = FALSE;

procedure  ReleaseHMA;
  var
    xms  : XMSParams;
  begin
    if  HMAGrabbed  then
      begin
        xms.FnNum := 2;      { XMS FN 2: Release High Memory Area }
        XMSFunction( xms );
        HMAGrabbed := FALSE;
      end;
    if  A20Enabled  then
      begin
        xms.FnNum := 4;      { XMS FN 4: Global Disable A20 Line }
        XMSFunction( xms );
        A20Enabled := FALSE;
      end;
  end;   { ReleaseHMA }

var
  OldExitProc : pointer;

{$F+}                        { Must be FAR! }
procedure  HimemExitProcedure;
  begin
    ExitProc := OldExitProc;
    ReleaseHMA;
  end;   { HimemExitProcedure }
{$F-}

procedure  RequestHMA;
  var
    xms  : XMSParams;
  begin
    xms.FnNum := 1;          { XMS FN 1: Request High Memory Area }
    xms.DX    := $FFFF;      { all of it! }
    XMSFunction( xms );
    HMAGrabbed  := xms.Success;
    if  HMAGrabbed  then
      begin
        { We succeeded, so enable the exit procedure }
        OldExitProc := ExitProc;
        ExitProc    := @HimemExitProcedure;
        xms.FnNum := 3;      { XMS FN 3: Global Enable A20 Line }
        XMSFunction( xms );
        A20Enabled := xms.Success;
      end;
    GotHMA := HMAGrabbed AND A20Enabled;    { Central success flag }
  end;   { RequestHMA }

const
  NextOffset : word = $0010;      { Offset of next allocation }

function  HMAMemAvail : word;
  begin
    if  GotHMA  then
      HMAMemAvail := $FFFF - NextOffset + 1
    else
      HMAMemAvail := 0;
  end;  { HMAMemAvail }

procedure HMAGetMem( var p : pointer; size : word );
  begin
    if  ( NOT GotHMA ) OR ( size > HMAMemAvail )  then
      p := NIL
    else
      begin
        p := Ptr( $FFFF, NextOffset );
        Inc( NextOffset, size );
      end;
  end;  { HMAGetMem }

procedure HMAMark( var p : pointer );
  begin
    p := Ptr( $FFFF, NextOffset );
  end;  { HMAMark }

procedure HMARelease( var p : pointer );
  begin
    NextOffset := Ofs( p^ );
  end;  { HMARelease }

begin
  DetectXMS;
  if  XMSpresent  then
    RequestHMA;
end.
