// Listing 2: bitvect.cpp
// Implementation of BitVect (bit vector) class.
// Copyright (C) 1991 by Nicholas Wilt.  All rights reserved.

#include <mem.h>
#include "bitvect.h"

BitVect::BitVect(const BitVect& x)
{
  numbytes = x.numbytes;
  numbits = x.numbits;
  vect = new unsigned char[numbytes];
  memcpy(vect, x.vect, numbytes);
}

BitVect::BitVect(long NumElms): numbits(NumElms)
{
  numbytes = (numbits+7) >> 3;
  vect = new unsigned char[numbytes];
}

BitVect::BitVect(long NumElms, int val): numbits(NumElms)
{
  numbytes = (numbits+7) >> 3;
  vect = new unsigned char[numbytes];
  memset(vect, (val) ? 0xff : 0, numbytes);
}

BitVect::~BitVect()
{
  delete vect;
}

// BitVect x[inx] returns the inx'th element of x (a 0 or 1).
// No range checking is performed.
int BitVect::operator[] (long inx)
{
  return (vect[inx>>3] & (1 << (inx & 7))) != 0;
}

// Same as operator[].	May be more intuitive at times.
int BitVect::GetBit(long inx)
{
  return (*this)[inx];
}

// Sets the inx'th bit in the vector to 1 or 0.
void BitVect::SetBit(long inx, int on)
{
  if (on)
    vect[inx>>3] |= (1 << (inx & 7));
  else
    vect[inx>>3] &= ~(1 << (inx & 7));
}

// Sets the given range of bits to the given value.
void BitVect::SetRange(long min, long max, int on)
{
  long truemin, truemax;

  truemin = (min < max) ? min : max;
  truemax = (max > min) ? max : min;
  for (long i = truemin; i <= truemax; i++)
    SetBit(i, on);
}

// NOT's the bit vector--all the 0's become 1's and vice versa.
BitVect BitVect::operator~()
{
  BitVect ret(numbits);
  unsigned char *src = vect;
  unsigned char *dst = ret.vect;
  for (int cnt = numbytes; cnt--; src++, dst++)
    *dst = ~*src;
  return ret;
}

// OR's the bit vector with another.  They should be the same size,
// but this is not checked for.
BitVect& BitVect::operator|= (BitVect& x)
{
  unsigned char *src = x.vect;
  unsigned char *dst = vect;
  for (int cnt = numbytes; cnt--; src++, dst++)
    *dst |= *src;
  return *this;
}

// AND's the bit vector with another.  They should be the same
// size, but this is not checked for.
BitVect& BitVect::operator&= (BitVect& x)
{
  unsigned char *src = x.vect;
  unsigned char *dst = vect;
  for (int cnt = numbytes; cnt--; src++, dst++)
    *dst &= *src;
  return *this;
}

// Set difference: if the ith element is in *this, and it's
// also in x, remove it from *this.
BitVect& BitVect::operator-= (BitVect& x)
{
  for (long i = 0; i < numbits; i++)
    if (x[i])
      SetBit(i, 0);
  return *this;
}

// OR's two bit vectors together, generating a third.
BitVect operator| (BitVect& x, BitVect& y)
{
  BitVect ret(x);
  ret |= y;
  return ret;
}

// AND's two bit vectors another, generating a third.
BitVect operator& (BitVect& x, BitVect& y)
{
  BitVect ret(x);
  ret &= y;
  return ret;
}

// Returns set difference x - y.
BitVect operator- (BitVect& x, BitVect& y)
{
  BitVect ret(x);
  ret -= y;
  return ret;
}
