// DEMOAPP.C - Windows Help Demonstration Application
// Copyright (C) 1993 Ray Duncan
// PC Magazine * Ziff Davis Publishing

#define dim(x) (sizeof(x) / sizeof(x[0]))   // returns no. of elements

#include "windows.h"
#include "demoapp.h"

// string used to allocate private message number
// for use by application's windows hook callback 
#define HOOKMSGSTRING "HelpHookMessage"

// context ID for context-sensitive help. must be
// synchronized with the [MAP] section of HPJ file.
#define MENU_INDEX_HELP   1

TCHAR szShortName[] = TEXT("DemoApp");      // short application name
TCHAR szAppName[] = TEXT("Help Demo");      // long application name
TCHAR szMenuName[] = TEXT("DemoAppMenu");   // name of menu resource
TCHAR szIconName[] = TEXT("DemoAppIcon");   // name of icon resource
TCHAR szHelpName[] = TEXT("demoapp.hlp");   // name of help file

//
// Table of window messages supported by FrameWndProc()
// and the functions which correspond to each message.
//
struct decodeUINT messages[] = {
    0, DoHelpContext,
    WM_PAINT, DoPaint,
    WM_COMMAND, DoCommand,
    WM_DESTROY, DoDestroy,
    WM_CLOSE, DoClose, } ;

//
// Table of menubar item IDs and their corresponding functions.
//
struct decodeUINT menuitems[] = {
    IDM_EXIT, DoMenuExit,
    IDM_HELPCONTENTS, DoMenuHelpContents,
    IDM_HELPSEARCH, DoMenuHelpSearch,
    IDM_HELPONHELP, DoMenuHelpHelp,
    IDM_HELPABOUT, DoMenuHelpAbout, } ;

// far pointer to windows hook callback
WNDPROC lpfnWindowsHookProc;    

// handle for frame window
HWND hFrame = NULL;

// program instance handle
HANDLE hInst;                   

// handle for Windows hook function
HHOOK hWindowsHook;             

// handle for menu bar accelerator key table
HANDLE hAccel;

//
// WinMain -- entry point for this application from Windows.
//
INT APIENTRY WinMain(HANDLE hInstance,
    HANDLE hPrevInstance, LPSTR lpCmdLine, INT nCmdShow)
{
    MSG msg;

    hInst = hInstance;                      // save this instance handle

    if(!hPrevInstance)                      // if first instance,
        if(!InitApp(hInstance))             // register window class
            return(FALSE);                  // exit if couldn't register

    if(!InitInstance(hInstance, nCmdShow))  // create this instance's window
        return(FALSE);                      // exit if create failed

    while(GetMessage(&msg, NULL, 0, 0))     // while message != WM_QUIT
    {
        if(!TranslateAccelerator(hFrame, hAccel, &msg))
        {
            TranslateMessage(&msg);         // translate virtual key codes
            DispatchMessage(&msg);          // dispatch message to window
        }
    }

    TermInstance(hInstance);                // clean up for this instance
    return(msg.wParam);                     // return code = WM_QUIT value
}

//
// InitApp --- global initialization code for this application.
//
BOOL InitApp(HANDLE hInstance)
{
    WNDCLASS  wc;

    // set parameters for frame window class
    wc.style = CS_HREDRAW|CS_VREDRAW;       // class style
    wc.lpfnWndProc = FrameWndProc;          // class callback function
    wc.cbClsExtra = 0;                      // extra per-class data
    wc.cbWndExtra = 0;                      // extra per-window data
    wc.hInstance = hInstance;               // handle of class owner
    wc.hIcon = LoadIcon(hInst, szIconName); // application icon
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);       // default cursor
    wc.hbrBackground = GetStockObject(WHITE_BRUSH); // background color 
    wc.lpszMenuName =  szMenuName;          // name of menu resource
    wc.lpszClassName = szShortName;         // name of window class

    return(RegisterClass(&wc));             // register class, return status
}

//
// InitInstance() -- instance initialization code
// 
BOOL InitInstance(HANDLE hInstance, INT nCmdShow)
{
    hFrame = CreateWindow(                  // create frame window
        szShortName,                        // window class name
        szAppName,                          // text for title bar
        WS_OVERLAPPEDWINDOW,                // window style
        CW_USEDEFAULT, CW_USEDEFAULT,       // default position
        CW_USEDEFAULT, CW_USEDEFAULT,       // default size
        NULL,                               // no parent window
        NULL,                               // use class default menu
        hInstance,                          // window owner
        NULL);                              // unused pointer

    if(!hFrame) return(FALSE);              // error, can't create window

    // Load menu bar accelerator key table
    if(!(hAccel = LoadAccelerators (hInst, "DemoAppAcc")))
        return FALSE;

    // allocate private message number for use by 
    // windows hook callback procedure
    messages[0].Code = 
        RegisterWindowMessage((LPSTR) HOOKMSGSTRING);

    // give up if can't allocate private message
    if(messages[0].Code == 0)
        return(FALSE);

    // allocate thunk for windows hook callback
    lpfnWindowsHookProc = 
        MakeProcInstance((WNDPROC) WindowsHookProc, hInst);

    // bail out if thunk allocation failed
    if(lpfnWindowsHookProc == NULL)
        return(FALSE);

    // register windows hook callback and save handle
    hWindowsHook = SetWindowsHookEx(WH_MSGFILTER, 
                   lpfnWindowsHookProc, hInst, GetCurrentTask());

    ShowWindow(hFrame, nCmdShow);           // make frame window visible
    UpdateWindow(hFrame);                   // force WM_PAINT message
    return(TRUE);                           // return success flag
}

//
// TermInstance -- instance termination code for this application.
//
BOOL TermInstance(HANDLE hinstance)
{
    // delete our windows hook function from global chain
    UnhookWindowsHookEx(hWindowsHook);

    // release the thunk for our windows callback
    FreeProcInstance(lpfnWindowsHookProc);

    return(TRUE);                           // return success flag
}

//
// FrameWndProc --- callback function for application frame window.
//
LONG CALLBACK FrameWndProc(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    INT i;                                  // scratch variable

    for(i = 0; i < dim(messages); i++)      // decode window message and
    {                                       // run corresponding function
        if(wMsg == messages[i].Code)
            return((*messages[i].Fxn)(hWnd, wMsg, wParam, lParam));
    }
                                            // or hand off to Windows
    return(DefWindowProc(hWnd, wMsg, wParam, lParam));
}

//
// DoCommand -- process WM_COMMAND message for frame window by
// decoding the menubar item with the menuitems[] array, then
// running the corresponding function to process the command.
// 
LONG DoCommand(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    INT i;                                  // scratch variable

    for(i = 0; i < dim(menuitems); i++)     // decode menu command and
    {                                       // run corresponding function
        if(wParam == menuitems[i].Code)
            return((*menuitems[i].Fxn)(hWnd, wMsg, wParam, lParam));
    }

    return(DefWindowProc(hWnd, wMsg, wParam, lParam));
}

//
// DoDestroy -- process WM_DESTROY message for frame window.
// 
LONG DoDestroy(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    PostQuitMessage(0);                     // force WM_QUIT message to
    return(FALSE);                          // terminate the event loop
}

//
// DoPaint -- process WM_PAINT message for frame window.
// 
LONG DoPaint(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    HDC hdc;
    PAINTSTRUCT ps;
    RECT rect;
    HFONT hfont;

    hdc = BeginPaint(hWnd, &ps);            // get device context
    GetClientRect(hWnd, &rect);             // get client area dimensions
    hfont = CreateFont(-36, 0, 0, 0, 700,   // get handle for pretty font
        TRUE, 0, 0, ANSI_CHARSET,
        OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS,
        DEFAULT_QUALITY, (FF_MODERN << 4) + DEFAULT_PITCH,
        TEXT("Ariel"));
    SelectObject(hdc, hfont);               // realize font in DC
    DrawText(hdc, TEXT("Windows Help Demo App"), -1,     // paint text
        &rect, DT_CENTER | DT_VCENTER | DT_SINGLELINE);  // in window
    EndPaint(hWnd, &ps);                    // release device context
    return(FALSE);                          // terminate the event loop
}

//
// DoClose -- process WM_CLOSE message for frame window.
// 
LONG DoClose(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    // shut down help file viewer if running
    WinHelp(hWnd, szHelpName, HELP_QUIT, 0);   

    // destroy the frame window 
    DestroyWindow(hWnd);          
    return(FALSE);                              
}

//
// DoMenuExit -- process File-Exit command from menu bar.
// 
LONG DoMenuExit(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    SendMessage (hWnd, WM_CLOSE, 0, 0L);    // send close message    
    return(FALSE);                          // to shut down the app
}

//
// DoMenuHelpContents -- process Help-Contents command from menu bar.
// 
LONG DoMenuHelpContents(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    WinHelp(hWnd, szHelpName, HELP_INDEX, 0L);
    return(FALSE);
}

//
// DoMenuHelpSearch -- process Help-Search for... command from menu bar.
// 
LONG DoMenuHelpSearch(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    WinHelp(hWnd, szHelpName, HELP_PARTIALKEY, (LONG) (LPSTR) "");
    return(FALSE);
}

//
// DoMenuHelpHelp -- process Help-How to Use Help command from menu bar.
// 
LONG DoMenuHelpHelp(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    WinHelp(hWnd, "WINHELP.HLP", HELP_HELPONHELP, 0L);
    return(FALSE);
}

//
// DoMenuHelpAbout -- process Help-About command from menu bar.
// 
LONG DoMenuHelpAbout(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    WNDPROC lpProcAbout;                    // scratch pointer

    lpProcAbout = MakeProcInstance((WNDPROC) AboutDlgProc, hInst);
    DialogBox(hInst, TEXT("AboutBox"), hWnd, lpProcAbout);         
    FreeProcInstance(lpProcAbout);
    return(FALSE);                              
}

//
// AboutDlgProc -- callback routine for About... dialog
// 
BOOL CALLBACK AboutDlgProc (HWND hwnd, UINT msg, UINT wParam, LONG lParam)
{
    if((msg == WM_COMMAND) && (wParam == IDOK))
        EndDialog(hwnd, 0);
    else
        return(FALSE);
}

//
// DoHelpContext() -- this routine is called by 
// FrameWndProc() for processing of the private message 
// sent by the windows hook procedure.  It 
// invokes the help file viewer with a context ID
// in order to deliver context-specific help.
// 
LONG DoHelpContext(HWND hWnd, UINT wMsg, UINT wParam, LONG lParam)
{
    WinHelp(hWnd, szHelpName, HELP_CONTEXT, MENU_INDEX_HELP);
    return(FALSE);
}

//
// WindowsHookProc -- windows hook callback function.
// This function must be registered with SetWindowsHookEx()
// and exported in the application DEF file.
//
DWORD CALLBACK WindowsHookProc(INT nCode, UINT wParam, LPMSG lpMsg)
{
    // bail out if callback unknown type or if pointer 
    // to message structure is not valid
    if((nCode < 0) || !lpMsg)
        return(CallNextHookEx(hWindowsHook, nCode, wParam, lpMsg));

    // check if this is F1 keypress during a menu popup
    if(nCode == MSGF_MENU)
        if((lpMsg->message==WM_KEYDOWN) && 
           (lpMsg->wParam == VK_F1))
        {
            // yes, post private message number to 
            // our frame window's message handler
            PostMessage(hFrame, messages[0].Code,
                        nCode, lpMsg->hwnd);
        }

    // pass message on to the other windows hooks procs in chain
    return(CallNextHookEx(hWindowsHook, nCode, wParam, lpMsg));
}



