PROGRAM PCDaemon(INPUT, OUTPUT);
{$I Options}
{$M 8192,2048,8192}
{$I-}

{
  Copyright (C) 1991 Julian Byrne. All rights reserved.

  Title:        PC Daemon
  File name:    PCDAEMON.PAS
  Version:      1.00
  Usage:        pcdaemon [CommandFile]
  Description:  Provides a facility for a PC to act as server, performing
                arbitrary tasks; at a given time, time interval, keystroke
                and when a file comes into existence. Particularly useful
                in a network with shared file access. 
  Dependencies: See USES statement
  Author:       Julian Byrne
  Address:      Electrical and Computer Systems Engineering Department
                Monash University, Wellington Road, Clayton, Victoria, 3168
                Australia
  Internet:     julian.byrne@monash.edu.au.
  Other nets:   Quaterman & Hoskins "Notable Computer Networks"
                CACM Oct'86 pp932-971
  History:      89/10/29 Initial version
  Notes:
}

  USES
    CRT,
    DOS,
    Error,
    Country;

{$I Site}

  CONST
    ExtDefault     = '.DAT';
    MaxCom         = 64;
    MaxTime        = -1; { Unsigned LONGINT }
    MinTime        =  0; { Unsigned LONGINT }

  TYPE
    Scalar         = {$IFOPT N+} SINGLE {$ELSE} REAL {$ENDIF} ;
    STRING8        = STRING[8];
    STRING65       = STRING[65];
    STRING80       = STRING[80];
    pSTRING65      = ^STRING65;
    pSTRING        = ^STRING;
    DOScommand     = RECORD
                       Code      : CHAR;
                       DateTimeAbs,
                       MscVal    : LONGINT;
                       Directory : pSTRING65;
                       Command   : pSTRING;
                     END;
    ComTableType   = ARRAY[1..MaxCom] OF DOScommand;

  VAR
    TopCom         : INTEGER;
    ISDcode        : BYTE;
    CDD            : CDI;
    ComTable       : ComTableType;
    LogName        : pSTRING65;
    LogFile        : TEXT;
    UseLogFile     : BOOLEAN;
    CommandFile    : STRING65;
    MyDir,
    Dir            : DirStr;
    MyName,
    Name           : NameStr;
    MyExt,
    Ext            : ExtStr;

  FUNCTION GetDateTime : LONGINT;

{ Return current system time as a 32 bit integer }

    VAR
      DTtemp : DateTimeWH;

    BEGIN { GetDateTime }
      Country.GetDateTime(DTtemp);
      GetDateTime := SecInd2(DTtemp);
    END { GetDateTime };


  FUNCTION StrF(X : INTEGER) : STRING8;

    VAR
      Result : STRING8;

    BEGIN { StrF }
      Str(X, Result);
      StrF := Result;
    END { StrF };


  PROCEDURE LogMessage(Message : STRING80);

{ Output a time stamped message }

    VAR
      DT   : DateTimeWH;
      Line : STRING[120];

    BEGIN { LogMessage }
      Country.GetDateTime(DT);
      Line := '[' + StrDateTime(CDD, DT) + '] ' + Message + '.';
      WriteLn(Line);
      IF UseLogFile THEN
        BEGIN
          WriteLn(LogFile, Line);
          Flush(LogFile);
        END;
    END { LogMessage };


  PROCEDURE Fatal(Message : STRING80);

{ Output a message and stop the program }

    BEGIN { Fatal }
      LogMessage(Message+', exitting');
      IF LogName <> NIL THEN
        Close(LogFile);
      Halt(0);
    END { Fatal };
 

  PROCEDURE WriteCommand(VAR Com : DOSCommand);

{ Display one interpreted line of command file }

    BEGIN { WriteCommand }
      WITH Com DO
        BEGIN
          Write(Code);
          Write(' ', DateTimeAbs);
          Write(' ', MscVal);
          IF Command <> NIL THEN
            Write(' "', Command^, '"');
          IF Directory <> NIL THEN
            Write(' + "', Directory^, '"');
          WriteLn;
        END;
    END { WriteCommand };

{----------------------------------------------------------------------------}

  PROCEDURE GetCommands(FileName : STRING65; VAR ComTable : ComTableType);

{ Read and parse command file in a table }

    VAR
      Line      : STRING;
      ErrorLine : INTEGER;
      f         : TEXT;
      I, Iold   : INTEGER;
      DTtmp     : LONGINT;

    PROCEDURE Error(Message : STRING);

      VAR
        Dummy : WORD;

{ Output message about error in command file and stop the program }

      BEGIN { Error }
        Dummy := IOResult; { Reset I/O so message will output ok }
        WriteLn(MyName, ': Error in line ', ErrorLine, '; ', Message);
        WriteLn('':Length(MyName)+2, Line);
        WriteLn('':Length(MyName)+1+I, '^');
        Halt(0);
      END { Error };

    PROCEDURE CheckLength;

{ See if we are unexpectedly at the end of a line in the command file }

      BEGIN { CheckLength }
        IF I > Length(Line) THEN
          Error('Unexpected end of line');
      END { CheckLength };

    PROCEDURE SkipSpaces;

      BEGIN { SkipSpaces }
        WHILE (I <= Length(Line)) AND (Line[I] <= ' ') DO
          INC(I);
      END { SkipSpaces };

    PROCEDURE SkipNotSpaces;

      BEGIN { SkipNotSpaces }
        WHILE (I <= Length(Line)) AND (Line[I] >  ' ') DO
          INC(I);
      END { SkipNotSpaces };

    PROCEDURE SkipTail;

{ Skip the tail end of a command word }

      BEGIN { SkipTail }
        SkipNotSpaces;
        SkipSpaces;
      END { SkipTail };

    PROCEDURE SkipDigits;

{ Skip a real number }

      BEGIN { SkipDigits }
        WHILE (I <= Length(Line)) AND (((Line[I] >=  '0') AND (Line[I] <= 
          '9')) OR (Line[I] = '.')) DO
          INC(I);
      END { SkipDigits };

    PROCEDURE ParseNumber(VAR x : LONGINT);

{ Read an integer value }

      VAR
        code,
        Iold : INTEGER;

      BEGIN { ParseNumber }
        SkipSpaces;
        Iold := I;
        IF ((I+2) <= Length(Line)) AND
          (Line[I  ] = '''') AND
          (Line[I+2] = '''') THEN
          BEGIN
            x := ORD(Line[I+1]);
            I := I + 3;
          END
        ELSE
          BEGIN
            SkipDigits;
{$IFOPT R+}
      {$R-} Val(Copy(Line, Iold, I-Iold), x, code); {$R+}
{$ELSE}
            Val(Copy(Line, Iold, I-Iold), x, code);
{$ENDIF}
            IF code <> 0 THEN
              Error('Bad integer');
          END;
      END { ParseNumber };

    PROCEDURE ParseDelTime(VAR x : LONGINT);

{ Read a real time value with appended unit }

      VAR
        code,
        Iold : INTEGER;
        V    : Scalar;

      BEGIN { ParseDelTime }
        SkipSpaces;
        Iold := I;
        CheckLength;
        SkipDigits;
        IF I = Iold THEN
          Error('Bad delta time');
{$IFOPT R+}
  {$R-} Val(Copy(Line, Iold, I-Iold), V, code); {$R+}
{$ELSE}
        Val(Copy(Line, Iold, I-Iold), V, code);
{$ENDIF}
        IF code <> 0 THEN
          Error('Bad number');
        SkipSpaces;
        CheckLength;
        CASE UpCase(Line[I]) OF
          'S' : x := ROUND(V*0.5);
          'M' : x := ROUND(V*0.5*60.0);
          'H' : x := ROUND(V*0.5*60.0*60.0);
          'D' : x := ROUND(V*0.5*60.0*60.0*24.0);
        ELSE
          Error('Bad delta time unit');
        END;
        SkipTail;
      END { ParseDelTime };

    PROCEDURE ParseDateTimeAbs(VAR x : LONGINT);

{ Parse a full absolute time/date value }

      VAR
        Iold   : INTEGER;
        DTtemp : DateTimeWH;

      BEGIN { ParseDateTimeAbs }
        SkipSpaces;
        Iold := I;
        SkipNotSpaces;
        SkipSpaces;
        SkipNotSpaces;
        ValDateTime(CDD, Copy(Line, Iold, I-Iold), DTtemp);
        IF NOT DateTimeValid(DTtemp) THEN
          Error('Bad absolute time');
        IF DTtemp.DT.Year >= 80 THEN
          INC(DTtemp.DT.Year, 1900)
        ELSE
          INC(DTtemp.DT.Year, 2000);
        x := SecInd2(DTtemp);
      END { ParseDateTimeAbs };

    PROCEDURE ParseString(VAR x : pSTRING65);

{ Parse a string with no embedded spaces }

      VAR
        Iold : INTEGER;

      BEGIN { ParseString }
        SkipSpaces;
        CheckLength;
        Iold := I;
        SkipNotSpaces;
        New(x);
        x^ := Copy(Line, Iold, I-Iold);
      END { ParseString };
 
    PROCEDURE ParseLineTail(VAR x : pSTRING);

{ Parse rest of line as a string with embedded spaces }

      BEGIN { ParseLineTail }
        SkipSpaces;
        CheckLength;
        New(x);
        x^ := Copy(Line, I, 255);
        I  := Length(Line)+1;
      END { ParseLineTail };

    BEGIN { GetCommands }
      Assign(f, FileName);
      Reset(f);
      IF IOResult <> 0 THEN
        Fatal('Error opening "'+FileName+'"');
      ErrorLine  := 0;
      TopCom     := 0;
      UseLogFile := FALSE;
      LogName    := NIL;
      WHILE NOT EOF(f) DO
        BEGIN
          INC(ErrorLine);
          ReadLn(f, Line);
          IF IOResult <> 0 THEN
            Fatal('Error reading "'+FileName+'"');
          I := 1;
          SkipSpaces;
          IF (I <= Length(Line)) THEN
            BEGIN
              INC(TopCom);
              WITH ComTable[TopCom] DO
                BEGIN
                  Code        := UpCase(Line[I]);
                  DateTimeAbs := MaxTime;
                  MscVal      := 0;
                  Directory   := NIL;
                  Command     := NIL;
                  SkipTail;
                  CASE Code OF
                    'D' : BEGIN { Execute a command at intervals }
                            DateTimeAbs := GetDateTime;
                            ParseDelTime(MscVal);
                            DateTimeAbs := (((DateTimeAbs-1) DIV MscVal)+1) *
                              MscVal;
                            ParseLineTail(Command);
                          END;
                    'F' : BEGIN { Execute a command when a file is found }
                            DateTimeAbs := GetDateTime;
                            ParseDelTime(MscVal);
                            DateTimeAbs := (((DateTimeAbs-1) DIV MscVal)+1) *
                              MscVal;
                            ParseString(Directory);
                            IF (Directory^[Length(Directory^)] <> ':') AND
                               (Directory^[Length(Directory^)] <> '\') THEN
                              Directory^ := Directory^ + '\';
                            ParseLineTail(Command);
                          END;
                    'K' : BEGIN { Execute a command when a key is pressed }
                            ParseNumber(MscVal);
                            ParseLineTail(Command);
                          END;
                    'L' : BEGIN { Specify a log file }
                            DEC(TopCom);
                            ParseString(LogName);
                            SkipSpaces;
                          END;
                    'M' : BEGIN { Execute a command at intervals on mark }
                            DateTimeAbs := GetDateTime;
                            ParseDateTimeAbs(DTtmp);
                            ParseDelTime(MscVal);
                            DateTimeAbs := (((DateTimeAbs-DTtmp-1) DIV MscVal)
                              +1) * MscVal + DTtmp;
                            ParseLineTail(Command);
                          END;
                    'Q' : BEGIN { Quit program at a given time }
                            ParseDateTimeAbs(DateTimeAbs);
                            ParseNumber(MscVal);
                            SkipSpaces;
                          END;
                    'R' : BEGIN { Skip a remark }
                            DEC(TopCom);
                            I := Length(Line)+1;
                          END;
                    'T' : BEGIN { Execute a comand at a given time }
                            ParseDateTimeAbs(DateTimeAbs);
                            ParseLineTail(Command);
                          END;
                  ELSE
                    Error('Bad command - expecting D,F,K,L,M,Q,R or T');
                  END;
                  IF I <= Length(Line) THEN
                    Error('End of line expected');
                END;
            END;
        END;
      Close(f);
      IF LogName <> NIL THEN
        BEGIN
          Assign(LogFile, LogName^);
          Append(LogFile);
          IF IOResult <> 0 THEN
            BEGIN
              ReWrite(LogFile);
              IF IOResult <> 0 THEN
                Fatal('Error opening log file "' + LogName^ + '"');
            END;
          UseLogFile := TRUE;
        END;
    END { GetCommands };

{----------------------------------------------------------------------------}

  PROCEDURE ExeCommands(VAR ComTable : ComTableType);

    VAR
      I,
      IndexFirst  : INTEGER;
      CurDateTime,
      TimeFirst   : LONGINT;
      Done        : BOOLEAN;


    PROCEDURE Execute(Command : STRING);

{ Execute a DOS command, either built in or available on the PATH }

      VAR
        Result : WORD;

      BEGIN { Execute }
        LogMessage('Executing "' + Command + '"');
{$IFNDEF VER40}
        SwapVectors;
{$ENDIF}
        Exec(GetEnv('COMSPEC'), '/C ' + Command);
{$IFNDEF VER40}
        SwapVectors; 
{$ENDIF}
        IF DOSError <> 0 THEN
          BEGIN
            LogMessage('Execute failed. DOS error = ' + StrF(DOSError));
            LogMessage(IOErrorMsg(DOSError));
          END
        ELSE
          BEGIN
            Result := DOSExitCode;
            LogMessage('Executed okay. Exit code = ' + StrF(Result));
            IF Result >= 256 THEN
              Fatal('Unexpected subprocess exit code');
          END;
      END { Execute };

    PROCEDURE DeleteEntry(I : INTEGER);

{ Delete a no longer active entry from the command table }

      VAR
        J : INTEGER;

      BEGIN { DeleteEntry }
        WITH ComTable[I] DO
          BEGIN
            IF Directory <> NIL THEN
              Dispose(Directory);
            IF Command   <> NIL THEN
              Dispose(Command  );
          END;
        FOR J := I+1 TO TopCom DO
          ComTable[J-1] := ComTable[J];
        DEC(TopCom);
        IF TopCom = 0 THEN
          Done := TRUE;
      END { DeleteEntry };

    PROCEDURE Beep;

{ Alert user to a problem }

      BEGIN { Beep }
        Sound(1000);
        Delay(25);
        NoSound;
      END { Beep };

    FUNCTION CmpDateTime(x, y : LONGINT) : INTEGER;

{ Return -1,0,1 depending whether times x and y are <,=,> }

      BEGIN { CmpDateTime }
        IF (x XOR $80000000) < (y XOR $80000000) THEN
          CmpDateTime := -1
        ELSE
        IF (x XOR $80000000) > (y XOR $80000000) THEN
          CmpDateTime :=  1
        ELSE
          CmpDateTime :=  0;
      END { CmpDateTime };

    PROCEDURE ProcessKeys;

{ Get a key from user and see if command table says what to do with it }

      VAR
        Key : LONGINT;
        I   : INTEGER;

      BEGIN { ProcessKeys }
        Key := ORD(ReadKey);
        IF Key = 0 THEN
          Key := 256 + ORD(ReadKey);
        IF (Key = 0) OR (Key = 3) THEN
          Fatal('Break key pressed');
        I := 0;
        REPEAT
          INC(I);
        UNTIL (I > TopCom) OR
          ((ComTable[I].Code = 'K') AND (ComTable[I].MscVal = Key));
        IF I > TopCom THEN
          Beep
        ELSE
          Execute(ComTable[I].Command^);
      END { ProcessKeys };


    PROCEDURE CheckDirectory(VAR Com : DOScommand);

{ Search a directory for files, executing a DOS command on them and deleting
  them in date/time order }

      VAR
        Done        : BOOLEAN;
        SearchTmp,
        SearchFirst : SearchRec;
        f           : FILE;

      BEGIN { CheckDirectory }
        WITH Com DO
          BEGIN
            REPEAT
              FindFirst(Directory^+'*.*', 0, SearchTmp);
              Done := DOSerror <> 0;
              IF NOT Done THEN
                BEGIN
                  SearchFirst := SearchTmp;
                  FindNext(SearchTmp);
                  WHILE DOSerror = 0 DO
                    BEGIN
                      IF CmpDateTime(SearchTmp.Time, SearchFirst.Time) < 0 THEN
                        SearchFirst := SearchTmp;
                      FindNext(SearchTmp);
                    END;
                  Execute(Command^+' '+Directory^+' '+SearchFirst.Name);
                END;
            UNTIL Done;
            INC(DateTimeAbs, MscVal);
          END;
      END { CheckDirectory };


    PROCEDURE WaitForDateTime(TimeFirst : LONGINT);

{ Delay till real time is past TimeFirst }

      BEGIN { WaitForDateTime }
        CurDateTime := GetDateTime;
        WHILE (CmpDateTime(TimeFirst, CurDateTime) > 0) AND (NOT KeyPressed) DO
          BEGIN
            Delay(1000);
            CurDateTime := GetDateTime; { *** BUSY WAIT *** }
          END;
      END { WaitForDateTime };


    BEGIN { ExeCommands }
      Done := FALSE;
      REPEAT
        TimeFirst  := ComTable[1].DateTimeAbs;
        IndexFirst := 1;
        FOR I := 2 TO TopCom DO
          IF CmpDateTime(TimeFirst, ComTable[I].DateTimeAbs) > 0 THEN
            BEGIN
              TimeFirst  := ComTable[I].DateTimeAbs;
              IndexFirst := I;
            END;
        WaitForDateTime(TimeFirst);
        IF KeyPressed THEN
          ProcessKeys
        ELSE
          WITH ComTable[IndexFirst] DO
            CASE Code OF
              'D' : BEGIN
                      Execute(Command^);
                      INC(DateTimeAbs, MscVal);
                    END;
              'F' : CheckDirectory(ComTable[IndexFirst]);
            { 'K' : } { Key interpreted by ProcessKeys }
              'M' : BEGIN
                      Execute(Command^);
                      INC(DateTimeAbs, MscVal);
                    END;
              'Q' : Done := TRUE;
            { 'R' : } { Remark filtered by GetCommand }
              'T' : BEGIN
                      Execute(Command^);
                      DeleteEntry(IndexFirst);
                    END;
            ELSE
              Fatal('System error');
            END;
      UNTIL Done;
    END { ExeCommands };


  BEGIN { PCDaemon }
    FSplit(ParamStr(0), MyDir, MyName, MyExt); 
    CheckBreak := FALSE;
    SetCBreak(FALSE);
    GetCountry(ISDcode, CDD);
    IF ParamCount > 1 THEN
      Fatal('Usage: '+MyName+' [CommandFile] ; Copyright 1991 Julian Byrne. All rights reserved.');
    IF ParamCount < 1 THEN
      CommandFile := GetEnv(MyName)
    ELSE
      CommandFile := ParamStr(1);
    FSplit(CommandFile, Dir, Name, Ext);
    IF Dir = '' THEN
      Dir := MyDir;
    IF Name = '' THEN
      Name := MyName;
    IF Ext = '' THEN
      Ext := ExtDefault;
    CommandFile := Dir+Name+Ext;
    GetCommands(CommandFile, ComTable);
    LogMessage(SiteName+': Started, press break key to stop');
    ExeCommands(ComTable);
  END { PCDaemon }.

