{$I Options }
UNIT NOS;

{
  Copyright (C) 1991 Julian Byrne. All rights reserved.

  Title:        Network Operating System Utilities
  File name:    NOS.PAS
  Version:      1.00
  Usage:        USES NOS;
  Description:  Provides subroutines to access Novell Netware (tm) facilities.
  Dependencies: See USES statement
  Author:       Julian Byrne
  Address:      Electrical and Computer Systems Engineering Department
                Monash University, Wellington Road, Clayton, Victoria, 3168
                Australia
  Internet:     julian.byrne@monash.edu.au.
  Other nets:   Quaterman & Hoskins "Notable Computer Networks"
                CACM Oct'86 pp932-971
  History:      91/ 5/ 1 Initial version
  Notes:        The procedures in this unit are straightforward implementations
                of some of the Netware v2.1 functions described in:

                Novell API Reference
                PRELIMINARY
                Copyright (c) 1988 by Novell, Inc.
                Revision 1.00

                Refer to this manual for details on record structures and
                error numbers. Points to note:
                - Includes support routines which may be useful elsewhere.
                - A number of values are stored in reverse byte order to that
                  of the IBM-PC, in particular network numbers and object
                  types. The functions Swap and Swap4 can be used to fix this.
                  The convention is to leave all values in H-L order,
                  only converting to L-H order when displaying on screen.
                - A number of the calls have no documented error return
                  and could cause mysterious errors if not used properly.
}

INTERFACE { NOS }

  USES
    DOS,
    Error,
    CheckIO;

  CONST
    WILD               =    $FFFF; { Object types in file server bindery }
    UNKNOWN            = Swap( 0); { Network byte order is opposite of PC's }
    USER               = Swap( 1); { See also ObjTypeName() in ERROR.PAS }
    USERGROUP          = Swap( 2);
    PRINTQUEUE         = Swap( 3);
    FILESERVER         = Swap( 4);
    JOBSERVER          = Swap( 5);
    GATEWAY            = Swap( 6);
    PRINTSERVER        = Swap( 7);
    ARCHIVEQUEUE       = Swap( 8);
    ARCHIVESERVER      = Swap( 9);
    JOBQUEUE           = Swap(10);
    ADMINISTRATION     = Swap(11);
    NASSNAGATEWAY      = Swap(33);
    REMOTEBRIDGESERVER = Swap(38);
    TCPIPGATEWAY       = Swap(39);

    Separator          = '\';  { What all good path specifications should use }
    SepOther           = '/';  { What they shouldn't }
    DriveUndefined     = 'a';  { An undefined drive letter }
    SeenNone           = $FFFF;
    Supervisor         = 'SUPERVISOR'; { So we don't mispell it elsewhere }

  TYPE
    STRING3       = STRING[  3];
    STRING12      = STRING[ 12];
    STRING15      = STRING[ 15];
    STRING21      = STRING[ 21];
    STRING27      = STRING[ 27];
    STRING47      = STRING[ 47];
    STRING55      = STRING[ 55];
    STRING59      = STRING[ 59];
    STRING80      = STRING[ 80];
    STRING100     = STRING[100];
    STRING127     = STRING[127];
    STRING255     = STRING[255];
    ASCIIZ14      = ARRAY[1..14] OF CHAR; { File name }
    ASCIIZ16      = ARRAY[1..16] OF CHAR; { Property name }
    ASCIIZ48      = ARRAY[1..48] OF CHAR; { File server name }
    GFSDATReplyType= ARRAY[0.. 6] OF BYTE; { Network date/time }
    DayOfWeekType = ARRAY[0.. 6] OF STRING3; { See CONST DayOfWeek }
    LongInt2      = ARRAY[0..5] OF BYTE;  { 6 byte integer }
    NetAddress    = ARRAY[1.. 6] OF BYTE; { Physical node address }
    FSNType       = LONGINT; { File server number (normally H-L order) }

    DriveHandleTable  = ARRAY[DriveType] OF BYTE;
    pDriveHandleTable = ^DriveHandleTable;
    DriveFlagTable    = ARRAY[DriveType] OF BYTE;
    pDriveFlagTable   = ^DriveFlagTable;
    DriveConIDTable   = ARRAY[DriveType] OF BYTE;
    pDriveConIDTable  = ^DriveConIDTable;
    CITEntry      = RECORD { (Get) Connection ID Table }
                      InUse         : BOOLEAN;
                      ConnectionID  : BYTE;
                      NetNumber     : FSNType;
                      ServerAddress : NetAddress;
                      Socket        : WORD;
                      RecTimeOut    : WORD;
                      RouterAddress : NetAddress;
                      Sequence      : BYTE;
                      ConnectionNum : BYTE;
                      ConnectionOk  : BOOLEAN;
                      MaxTimeOut    : WORD;
                      Filler        : ARRAY[1..5] OF BYTE;
                    END;
    ConIDTable    = ARRAY[1..8] OF CITEntry;
    pConIDTable   = ^ConIDTable;
    FSName        = ASCIIZ48;
    FSNameTable   = ARRAY[1..8] OF FSName;
    pFSNameTable  = ^FSNameTable;
    GCIReplyType  = RECORD { Get Connection Information }
                      BufLength : WORD;
                      ObjID     : LONGINT;
                      ObjType   : WORD;
                      ObjName   : ASCIIZ48;
                      LoginTime : GFSDATReplyType;
                      Dummy     : BYTE; { This avoids a network bug }
                    END;
    DOSDate       = WORD;
    DOSDateTime   = LONGINT;
    FileDataType  = RECORD
                      Attributes       : BYTE;
                      ExtAttributes    : BYTE;
                      FileSize         : LONGINT;
                      CreateDate       : DOSDate;
                      AccessDate       : DOSDate;
                      UpdateDateTime   : DOSDateTime;
                      ObjID            : LONGINT;
                      ArchiveDateTime  : DOSDateTime;
                      Reserved         : ARRAY[1..56] OF BYTE;
                    END;
    ScanDataType  = RECORD
                      DirHandle        : BYTE;
                      SearchAttributes : BYTE;
                      FilePath         : STRING255;
                    END;
    SFIReplyType  = RECORD { Scan File Information }
                      BufLength        : WORD;
                      SequenceNumber   : WORD;
                      Name             : ASCIIZ14;
                      FileData         : FileDataType;
                    END;
    SFIRequestType= RECORD { Set File Information }
                      BufLength        : WORD;
                      SubFunction      : BYTE;
                      FileData         : FileDataType;
                      ScanData         : ScanDataType;
                    END;
    GSIReplyType  = RECORD { Get Semaphore Information }
                      BufLength        : WORD;
                      NextRequest      : WORD;
                      OpenCount        : WORD;
                      SemaphoreValue   : SHORTINT;
                      Records          : BYTE;
                      List             : ARRAY[1..168] OF
                        RECORD
                          ConNum  : WORD;
                          TaskNum : BYTE;
                        END;
                      Dummy            : WORD;
                    END;
    GCSEntryType  = RECORD { Get Connection's Semaphores list entry }
                      OpenCount        : WORD;
                      SemaphoreValue   : SHORTINT;
                      TaskNum          : BYTE;
                      Name             : STRING127;
                    END;
    GCSReplyType  = RECORD { Get Connection's Semaphores }
                      BufLength        : WORD;
                      NextRequest      : WORD;
                      Records          : BYTE;
                      List             : GCSEntryType; { Warning - PACKED }
                      Dummy            : ARRAY[1..377] OF BYTE;
                    END;
    SORequestType = RECORD { Scan Object }
                      BufLength     : WORD;
                      SubFunction   : BYTE;
                      LastObjID     : LONGINT;
                      ObjType       : WORD;
                      ObjName       : STRING47;
                    END;
    SOReplyType   = RECORD { Scan Object }
                      BufLength     : WORD;
                      ObjID         : LONGINT;
                      ObjType       : WORD;
                      ObjName       : ASCIIZ48;
                      ObjFlag       : BOOLEAN;
                      ObjSecurity   : BYTE;
                      ObjProperties : BOOLEAN;
                    END;
    SPRequestType = RECORD { Scan Property }
                      BufLength     : WORD;
                      SubFunction   : BYTE;
                      ObjType       : WORD;
                      ObjName       : STRING47; { Warning - PACKED }
                      LastSeen      : LONGINT;
                      PropName      : STRING15; { Warning - PACKED }
                    END;
    SPReplyType   = RECORD { Scan Property }
                      BufLength     : WORD;
                      PropName      : ASCIIZ16;
                      PropFlags     : BYTE;
                      PropSecurity  : BYTE;
                      LastSeen      : LONGINT;
                      PropValue     : BOOLEAN;
                      PropMore      : BOOLEAN;
                    END;
    RPVRequestType= RECORD { Read Property Value }
                      BufLength     : WORD;
                      SubFunction   : BYTE;
                      ObjType       : WORD;
                      ObjName       : STRING47; { Warning - PACKED }
                      Segment       : BYTE; { 1 is first segment }
                      PropName      : STRING15;
                    END;
    ObjectListType= ARRAY[0..31] OF LONGINT;
    RPVReplyType  = RECORD { Read Property Value }
                      BufLength     : WORD;
                      PropValue     : ObjectListType;
                      MoreSegments  : BOOLEAN;
                      PropFlags     : BYTE;
                    END;
    GOCNRequestType = RECORD { Get Object Connection Numbers }
                      BufLength   : WORD;
                      SubFunction : BYTE;
                      ObjType     : WORD;
                      ObjName     : STRING47;
                    END;
    GOCNReplyType = RECORD { Get Object Connection Numbers }
                      BufLength   : WORD;
                      Connections : STRING100;
                    END;
    GCUSReplyType = RECORD { Get Connection's Usage Statistics }
                      BufLength      : WORD;
                      ElapsedTime    : LONGINT; { 1/18.2's of a second }
                      BytesRead      : LongInt2;
                      BytesWritten   : LongInt2;
                      RequestPackets : LONGINT;
                    END;
    GFSIReplyType = RECORD { Get File Server Information }
                      BufLength      : WORD;
                      ServerName     : ASCIIZ48;
                      VersionNet     : BYTE;
                      SubVersionNet  : BYTE;
                      ConnectMax     : WORD;
                      ConnectInUse   : WORD;
                      VolumeMax      : WORD;
                      RevisionOS     : BYTE;
                      LevelSFT       : BYTE;
                      LevelTTS       : BYTE;
                      ConnectPeak    : WORD;
                      VersionAccount : BYTE;
                      VersionVAP     : BYTE;
                      VersionQueue   : BYTE;
                      VersionPrint   : BYTE;
                      VersionConsole : BYTE;
                      LevelSecurity  : BYTE;
                      VersionBridge  : BYTE;
                      Reserved       : ARRAY[1..60] OF BYTE;
                    END;
  SBMRequestType  = RECORD { Send Broadcast Message }
                      BufLength   : WORD;
                      SubFunction : BYTE;
                      Connect     : STRING100; { Warning - PACKED }
                      Message     : STRING55;
                    END;
  SBMReplyType    = RECORD { Send Broadcast Message }
                      BufLength   : WORD;
                      Connect     : STRING100;
                    END;
  SCBRequestType  = RECORD { Send Console Broadcast }
                      BufLength   : WORD;
                      SubFunction : BYTE;
                      Connect     : STRING100; { Warning - PACKED }
                      Message     : STRING59;
                    END;

  CONST
    DayOfWeek  : DayOfWeekType = ('Sun','Mon','Tue','Wed','Thu','Fri','Sat');

  FUNCTION  Swap4(x : LONGINT) : LONGINT;
  FUNCTION  PtrToLong(P : POINTER) : LONGINT;
  FUNCTION  LongToPtr(L : LONGINT) : POINTER;
  FUNCTION  StrLongInt2(x : LongInt2; Width : BYTE) : STRING12;
  FUNCTION  StrTime(VAR x : GFSDATReplyType) : STRING21;
  FUNCTION  StrNode(VAR x : NetAddress) : STRING12;
  FUNCTION  StrName16(VAR x : ASCIIZ16) : STRING15;
  FUNCTION  StrName48(VAR x : ASCIIZ48) : STRING47;
  FUNCTION  UpCaseString(x : STRING) : STRING;
  FUNCTION  Trim(x : STRING) : STRING;
  FUNCTION  Normalize(x : STRING) : STRING;
  FUNCTION  NovellAPI : BOOLEAN;
  FUNCTION  GetDriveHandleTable       : pDriveHandleTable;
  FUNCTION  GetDriveFlagTable         : pDriveFlagTable;
  FUNCTION  GetDriveConnectionIDTable : pDriveConIDTable;
  FUNCTION  GetConnectionIDTable      : pConIDTable;
  FUNCTION  GetFileServerNameTable    : pFSNameTable;
  PROCEDURE SetPreferredConnectionID(x : BYTE);
  FUNCTION  GetPreferredConnectionID  : BYTE;
  FUNCTION  GetDefaultConnectionID    : BYTE;
  FUNCTION  GetConnectionID(FSName : STRING47) : BYTE;
  FUNCTION  GetFileServerNumber(FSName : STRING47) : FSNtype;
  FUNCTION  GetFileServerName(x : FSNType) : STRING47;
  PROCEDURE GetConnectionInformation(    ConnectionNum : BYTE;
                                     VAR GCIReply      : GCIReplyType);
  FUNCTION  GetDirectoryPath(Handle : BYTE) : STRING;
  FUNCTION  GetDirectoryHandle(x : DriveType) : WORD;
  PROCEDURE ScanFileInformation(    LastSeen    : WORD;
                                VAR ScanDataArg : ScanDataType;
                                VAR SFIReply    : SFIReplyType);
  PROCEDURE SetFileInformation(VAR SFIRequest : SFIRequestType);
  FUNCTION  AllocTemporaryDirectoryHandle(Drive : DriveType;
              Handle : BYTE; Path : STRING) : WORD;
  PROCEDURE DeallocateDirectoryHandle(DirHandle : BYTE);
  PROCEDURE GetSemaphoreInformation(LastSeen : WORD; Name : STRING127;
              VAR GSIReply : GSIReplyType);
  PROCEDURE GetConnectionSemaphores(    ConnectionNum : BYTE;
                                        LastSeen      : WORD;
                                    VAR GCSReply      : GCSReplyType);
  PROCEDURE OpenSemaphore(Name : STRING127; Value : SHORTINT;
              VAR OpenCount : BYTE; VAR Handle : LONGINT); 
  PROCEDURE CloseSemaphore(Handle : LONGINT);
  PROCEDURE LogNetworkMessage(MessageArg : STRING80);
  PROCEDURE ScanObject(VAR SORequest : SORequestType; VAR SOReply:SOReplyType);
  PROCEDURE ScanProperty(VAR SPRequest:SPRequestType; VAR SPReply:SPReplyType);
  PROCEDURE ReadPropertyValue(VAR RPVRequest : RPVRequestType;
              VAR RPVReply : RPVReplyType);
  FUNCTION  GetObjectName(x : LONGINT; VAR ObjectType : WORD) : STRING47;
  FUNCTION  GetBaseStatus(Drive : DriveType) : WORD;
  PROCEDURE MapFakeRootDirectory(Drive : DriveType; Path : STRING);
  PROCEDURE DeleteFakeRootDirectory(Drive : DriveType);
  FUNCTION  GetRelativeDriveDepth(Drive : DriveType) : BYTE;
  PROCEDURE AttachToFileServer  (ConnectionID : BYTE);
  PROCEDURE DetachFromFileServer(ConnectionID : BYTE);
  PROCEDURE LoginToFileServer(ObjectType  : WORD;
                              ObjectName  : STRING47;
                              PasswordArg : STRING27);
  PROCEDURE Logout;
  PROCEDURE LogoutFromFileServer(ConnectionID : BYTE);
  PROCEDURE GetObjectConnectionNumbers(VAR GOCNRequest : GOCNRequestType;
    VAR GOCNReply : GOCNReplyType);
  PROCEDURE GetConnectionUsageStatistics(ConnectionNum : BYTE;
    VAR GCUSReply : GCUSReplyType);
  PROCEDURE GetFileServerInformation(VAR GFSIReply : GFSIReplyType);
  PROCEDURE GetFileServerDateAndTime(VAR GFSDATReply : GFSDATReplyType);
  FUNCTION  CheckConsolePrivileges : BOOLEAN;
  FUNCTION  GetBroadcastMode : BYTE;
  FUNCTION  SetBroadcastMode(Mode : BYTE) : BYTE;
  FUNCTION  GetBroadcastMessage : STRING55;
  PROCEDURE SendBroadcastMessage(VAR SBMRequest : SBMRequestType;
    VAR SBMReply : SBMReplyType);
  PROCEDURE SendConsoleBroadcast(VAR SCBRequest : SCBRequestType);
  PROCEDURE NetFExpand(
        PathOld  : PathStr;
    VAR PathNew  : PathStr;
    VAR FSName   : STRING47;
    VAR FSNumber : FSNType);

IMPLEMENTATION { NOS }

FUNCTION NetCommand(Func : BYTE; VAR Request, Reply) : BYTE;

  VAR
    R : Registers;

  BEGIN { NetCommand }
    WITH R DO
      BEGIN
        AH         := Func;
        DS         := SEG(Request);
        SI         := OFS(Request);
        ES         := SEG(Reply);
        DI         := OFS(Reply);
        MSDOS(R);
        NetCommand := AL;
      END;
  END { NetCommand };
{
  Reverse L-H and H-L byte order in a 4 byte longint.
}
FUNCTION Swap4(x : LONGINT) : LONGINT;

  BEGIN  { Swap4 }
    Swap4 := (LONGINT(Swap(x AND $FFFF)) SHL 16) OR
      LONGINT(Swap((x SHR 16) AND $FFFF));
  END { Swap4 };
{
  Convert a SEG:OFS pointer to a linear address, taking into account
  wrap at the 1Mb mark.
}
FUNCTION PtrToLong(P : POINTER) : LONGINT;

  BEGIN { PtrToLong }
    PtrToLong := (LONGINT(SEG(P^)) * LONGINT($10) + LONGINT(OFS(P^)))
      AND $000FFFFF;
  END { PtrToLong };
{
  Convert a linear address to a normalized pointer.
}
FUNCTION LongToPtr(L : LONGINT) : POINTER;

  BEGIN { LongToPtr }
    LongToPtr := PTR(L SHR 4, L AND $F);
  END { LongToPtr };
{
  6 byte long integer to hex string
}
FUNCTION StrLongInt2(x : LongInt2; Width : BYTE) : STRING12;

  VAR
    I      : WORD;
    Result : STRING12;

  BEGIN { StrLongInt2 }
    Result := '';
    FOR I := 0 TO 5 DO
      Result := Result + Hex2(x[I]);
    StrLongInt2 := Result;
  END { StrLongInt2 };
{
  Network date/time:  Year/Month/Day Hour:Minute:Second
}
FUNCTION StrTime(VAR x : GFSDATReplyType) : STRING21;

  BEGIN { StrTime }
    StrTime := Str0(x[0])+'/'+Str0(x[1])+'/'+Str0(x[2])+' '+
      Str0(x[3])+':'+Str0(x[4])+':'+Str0(x[5]){+' '+DayOfWeek[x[6]]};
  END { StrTime };
{
  Network hardware address
}
FUNCTION StrNode(VAR x : NetAddress) : STRING12;

  VAR
    Result : STRING12;
    I      : WORD;

  BEGIN { StrNode }
    Result := '';
    FOR I := 1 TO 6 DO
      Result := Result + Hex2(x[I]);
    StrNode := Result;
  END { StrNode };


FUNCTION StrName16(VAR x : ASCIIZ16) : STRING15;

  BEGIN { StrName16 }
    StrName16 := StrName(x);
  END { StrName16 };


FUNCTION StrName48(VAR x : ASCIIZ48) : STRING47;

  BEGIN { StrName48 }
    StrName48 := StrName(x);
  END { StrName48 };
{
  Convert all characters in a STRING to uppercase.
}
FUNCTION UpCaseString(x : STRING) : STRING;

  VAR
    I : BYTE;

  BEGIN { UpCaseString }
    FOR I := 1 TO Length(x) DO
      x[I] := UpCase(x[I]);
    UpCaseString := x;
  END { UpCaseString };
{
  Remove leading and trailing spaces from a string.
}
FUNCTION Trim(x : STRING) : STRING;

  VAR
    I, J : BYTE;

  BEGIN { Trim }
    J := Length(x);
    WHILE (J >  0) AND (x[J] <= ' ') DO
      DEC(J);
    I := 1;
    WHILE (I <= J) AND (x[I] <= ' ') DO
      INC(I);
    Trim := Copy(x, I, J-I+1);
  END { Trim };
{
  Change "/"'s to "\"'s in a file name, remove leading and trailing spaces
  from it and convert it to uppercase.
}
FUNCTION Normalize(x : STRING) : STRING;

  VAR
    I : BYTE;

  BEGIN { Normalize }
    FOR I := 1 TO Length(x) DO
      IF x[I] = SepOther THEN
        x[I] := Separator;
    Normalize := UpCaseString(Trim(x));
  END { Normalize };
{
  Determine whether Novell Netware (tm) is present using multiplex interrupt.
}
  FUNCTION NovellAPI : BOOLEAN;

    VAR
      R : Registers;

    BEGIN { NovellAPI }
      WITH R DO
        BEGIN
          AX := $7A00; { Novell Netware low level API (IPX) installation check }
          Intr($2F, R);
          NovellAPI := (AL = $FF); { TRUE -> Novell present }
        END;
    END { NovellAPI };


  FUNCTION GetDriveHandleTable : pDriveHandleTable;

    VAR
      R : Registers;

    BEGIN { GetDriveHandleTable }
      WITH R DO
        BEGIN
          AX := $EF00;
          MSDOS(R);
          GetDriveHandleTable := PTR(ES, SI);
        END;
    END { GetDriveHandleTable };


  FUNCTION GetDriveFlagTable : pDriveFlagTable;

    VAR
      R : Registers;

    BEGIN { GetDriveFlagTable }
      WITH R DO
        BEGIN
          AX := $EF01;
          MSDOS(R);
          GetDriveFlagTable := PTR(ES, SI);
        END;
    END { GetDriveFlagTable };


  FUNCTION GetDriveConnectionIDTable : pDriveConIDTable;

    VAR
      R : Registers;

    BEGIN { GetDriveConnectionIDTable }
      WITH R DO
        BEGIN
          AX := $EF02;
          MSDOS(R);
          GetDriveConnectionIDTable := PTR(ES, SI);
        END;
    END { GetDriveConnectionIDTable };


  FUNCTION GetConnectionIDTable : pConIDTable;

    VAR
      R : Registers;

    BEGIN { GetConnectionIDTable }
      WITH R DO
        BEGIN
          AX := $EF03;
          MSDOS(R);
          GetConnectionIDTable := Ptr(ES, SI);
        END;
    END { GetConnectionIDTable };


  FUNCTION GetFileServerNameTable : pFSNameTable;

    VAR
      R : Registers;

    BEGIN { GetFileServerNameTable }
      WITH R DO
        BEGIN
          AX := $EF04;
          MSDOS(R);
          GetFileServerNameTable := PTR(ES, SI);
        END;
    END { GetFileServerNameTable };


  PROCEDURE SetPreferredConnectionID(x : BYTE);

    VAR
      R : Registers;

    BEGIN { SetPreferredConnectionID }
      WITH R DO
        BEGIN
          IF x > 8 THEN
            FatalError(0, 'Set Preferred Connection ID', StrBYTE(x,0));
          AX := $F000;
          DL := x;
          MSDOS(R);
        END;
    END { SetPreferredConnectionID };


  FUNCTION GetPreferredConnectionID : BYTE;

    VAR
      R : Registers;

    BEGIN { GetPreferredConnectionID }
      WITH R DO
        BEGIN
          AX := $F001;
          MSDOS(R);
          GetPreferredConnectionID := AL;
        END;
    END { GetPreferredConnectionID };


  FUNCTION GetDefaultConnectionID : BYTE;

    VAR
      R : Registers;

    BEGIN { GetDefaultConnectionID }
      WITH R DO
        BEGIN
          AX := $F002;
          MSDOS(R);
          GetDefaultConnectionID := AL;
        END;
    END { GetDefaultConnectionID };


  FUNCTION GetConnectionID(FSName : STRING47) : BYTE;

    VAR
      pFSNT : pFSNameTable;
      I     : BYTE;

    BEGIN { GetConnectionID }
      pFSNT  := GetFileServerNameTable;
      I      := 1;
      WHILE (I <= 8) AND (StrName48(pFSNT^[I]) <> FSName) DO
        INC(I);
      IF I > 8 THEN
        I := 0;
      GetConnectionID := I;
    END { GetConnectionID };


  FUNCTION GetFileServerNumber(FSName : STRING47) : FSNtype;

    VAR
      pSegment   : ^BYTE;
      pPropName  : ^STRING15;
      SORequest  : SORequestType;
      SOReply    : SOReplyType;
      RPVRequest : RPVRequestType;
      RPVReply   : RPVReplyType;

    BEGIN { GetFileServerNumber }
      GetFileServerNumber := 0;
      SORequest.LastObjID := -1;
      SORequest.ObjType   := FILESERVER;
      SORequest.ObjName   := FSName;
      ScanObject(SORequest, SOReply);
      IF (SOReply.BufLength <> 0) THEN { File server exists }
        WITH RPVRequest DO
          BEGIN
            ObjType    := FILESERVER;
            ObjName    := FSName;
            pSegment   := PTR(SEG(ObjName[1]), OFS(ObjName[1])+Length(ObjName));
            pSegment^  := 1;
            pPropName  := PTR(SEG(pSegment^), OFS(pSegment^)+SIZEOF(BYTE));
            pPropName^ := 'NET_ADDRESS';
            ReadPropertyValue(RPVRequest, RPVReply);
            GetFileServerNumber := RPVReply.PropValue[0];
          END;
    END { GetFileServerNumber };


  FUNCTION GetFileServerName(x : FSNtype) : STRING47;

    VAR
      pSegment   : ^BYTE;
      pPropName  : ^STRING15;
      SORequest  : SORequestType;
      SOReply    : SOReplyType;
      RPVRequest : RPVRequestType;
      RPVReply   : RPVReplyType;

    BEGIN { GetFileServerName }
      GetFileServerName   := '';
      SORequest.LastObjID := -1;
      SORequest.ObjType   := FILESERVER;
      SORequest.ObjName   := '*';
      REPEAT
        ScanObject(SORequest, SOReply);
        IF (SOReply.BufLength <> 0) THEN
          WITH RPVRequest DO
            BEGIN
              ObjType  := SOReply.ObjType;
              ObjName  := StrName48(SOReply.ObjName);
              pSegment := PTR(SEG(ObjName[1]),OFS(ObjName[1])+Length(ObjName));
              pSegment^  := 1;
              pPropName  := PTR(SEG(pSegment^),OFS(pSegment^)+SIZEOF(BYTE));
              pPropName^ := 'NET_ADDRESS';
              ReadPropertyValue(RPVRequest, RPVReply);
              IF (RPVReply.BufLength <> 0) AND (RPVReply.PropValue[0] = x) THEN
                BEGIN
                  GetFileServerName := ObjName; 
                  SOReply.BufLength := 0;
                END;
              SORequest.LastObjID := SOReply.ObjID;
          END;
      UNTIL SOReply.BufLength = 0;
    END { GetFileServerName };


  PROCEDURE GetConnectionInformation(    ConnectionNum : BYTE;
                                     VAR GCIReply      : GCIReplyType);

    TYPE
      GCIRequestType = RECORD
                         BufLength   : WORD;
                         SubFunction : BYTE;
                         Connection  : BYTE;
                       END;

    VAR
      Result     : BYTE;
      GCIRequest : GCIRequestType;

    BEGIN { GetConnectionInformation }
      GCIRequest.BufLength   := 2;
      GCIRequest.SubFunction := $16;
      GCIRequest.Connection  := ConnectionNum;
      GCIReply.BufLength     := 62;
      Result := NetCommand($E3, GCIRequest, GCIReply);
      IF Result <> 0 THEN
        FatalError(Result, 'Get Connection Information',
          'Connection number '+StrBYTE(ConnectionNum,0));
    END { GetConnectionInformation };


  FUNCTION GetDirectoryPath(Handle : BYTE) : STRING;

    TYPE
      GDPRequestType = RECORD
                         BufLength   : WORD;
                         Subfunction : BYTE;
                         DirHandle   : BYTE;
                       END;
      GDPReplyType   = RECORD
                         BufLength   : WORD;
                         DrivePath   : STRING;
                       END;
    VAR
      Result     : BYTE;
      GDPRequest : GDPRequestType;
      GDPReply   : GDPReplyType;

    BEGIN { GetDirectoryPath }
      GDPRequest.BufLength   := 2;
      GDPRequest.SubFunction := $01;
      GDPRequest.DirHandle   := Handle;
      GDPReply.BufLength     := 256;
      Result := NetCommand($E2, GDPRequest, GDPReply);
      IF Result <> 0 THEN
        FatalError(Result, 'Get Directory Path', 'Handle '+
          StrBYTE(Handle,0));
      GetDirectoryPath       := GDPReply.DrivePath;
    END { GetDirectoryPath };


FUNCTION GetDirectoryHandle(x : DriveType) : WORD;

  VAR
    R : Registers;

  BEGIN { GetDirectoryHandle }
    WITH R DO
      BEGIN
        AX := $E900;
        DX := ORD(x)-ORD('A');
        MSDOS(R);
        IF AL = 0 THEN
          FatalError(0, 'Get Directory Handle', x);
        GetDirectoryHandle := AX;
      END;
  END { GetDirectoryHandle };


PROCEDURE ScanFileInformation(    LastSeen    : WORD;
                              VAR ScanDataArg : ScanDataType;
                              VAR SFIReply    : SFIReplyType);
  TYPE
    SFIRequestType= RECORD
                      BufLength        : WORD;
                      SubFunction      : BYTE;
                      SequenceNumber   : WORD;
                      ScanData         : ScanDataType;
                    END;
  VAR
    Result     : BYTE;
    SFIRequest : SFIRequestType;

  BEGIN { ScanFileInformation }
    SFIRequest.BufLength      := 6 + Length(ScanDataArg.FilePath);
    SFIRequest.SubFunction    := $0F;
    SFIRequest.SequenceNumber := LastSeen;
    SFIRequest.ScanData       := ScanDataArg;
    SFIReply.BufLength        := 94;
    Result := NetCommand($E3, SFIRequest, SFIReply);
    IF Result <> 0 THEN
      WITH ScanDataArg DO
        FatalError(Result, 'Scan File Information',
          'Last Seen '+StrWORD(LastSeen,0)+
          ', Dir Handle '+StrBYTE(DirHandle,0)+
          ', File Path "'+FilePath+'"');
  END { ScanFileInformation };


PROCEDURE SetFileInformation(VAR SFIRequest : SFIRequestType);

  TYPE
    SFIReplyType = RECORD
                     BufLength : WORD;
                   END;

  VAR
    Result   : BYTE;
    SFIReply : SFIReplyType;

  BEGIN { SetFileInformation }
    SFIRequest.BufLength   := 82+Length(SFIRequest.ScanData.FilePath);
    SFIRequest.SubFunction := $10;
    FillChar(SFIRequest.FileData.Reserved,
      SIZEOF(SFIRequest.FileData.Reserved), CHR(0));
    SFIReply.BufLength     := 0;
    Result := NetCommand($E3, SFIRequest, SFIReply);
    IF Result <> 0 THEN
      WITH SFIRequest.ScanData DO
        FatalError(Result, 'Set File Information',
          'Dir Handle '+StrBYTE(DirHandle,0)+', File Path "'+FilePath+'"');
  END { SetFileInformation };


FUNCTION AllocTemporaryDirectoryHandle(Drive : DriveType;
  Handle : BYTE; Path : STRING) : WORD;

  TYPE
    ATDHRequestType = RECORD
                        BufLength   : WORD;
                        SubFunction : BYTE;
                        DirHandle   : BYTE;
                        DriveLetter : CHAR;
                        Path        : STRING;
                      END;
    ATDHReplyType   = RECORD
                        BufLength   : WORD;
                        DirHandle   : BYTE;
                        RightsMask  : BYTE;
                      END;
  VAR
    Result      : BYTE;
    ATDHRequest : ATDHRequestType;
    ATDHReply   : ATDHReplyType;

  BEGIN { AllocTemporaryDirectoryHandle }
    ATDHRequest.BufLength   := 4+Length(Path);
    ATDHRequest.SubFunction := 13;
    ATDHRequest.DirHandle   := Handle;
    ATDHRequest.DriveLetter := Drive;
    ATDHRequest.Path        := Path;
    ATDHReply.BufLength     := 2;
    Result := NetCommand($E2, ATDHRequest, ATDHReply);
    IF Result <> 0 THEN
      FatalError(Result, 'Alloc Temporary Directory Handle',
        'Drive '+Drive+', Handle '+StrBYTE(Handle,0)+', Path "'+Path+'"');
    AllocTemporaryDirectoryHandle := (ATDHReply.RightsMask SHL 8)
      OR (ATDHReply.DirHandle);
  END { AllocTemporaryDirectoryHandle };


PROCEDURE DeallocateDirectoryHandle(DirHandle : BYTE);

  TYPE
    DDHRequestType = RECORD
                       BufLength   : WORD;
                       SubFunction : BYTE;
                       DirHandle   : BYTE;
                     END;
    DDHReplyType   = RECORD
                       BufLength   : WORD;
                     END;

  VAR
    Result     : BYTE;
    DDHRequest : DDHRequestType;
    DDHReply   : DDHReplyType;

  BEGIN { DeallocateDirectoryHandle }
    DDHRequest.BufLength   := 2;
    DDHRequest.SubFunction := $14;
    DDHRequest.DirHandle   := DirHandle;
    DDHReply.BufLength     := 0;
    Result := NetCommand($E2, DDHRequest, DDHReply);
    IF Result <> 0 THEN
      FatalError(Result, 'Deallocate Directory Handle',
        'Dir Handle '+StrBYTE(DirHandle,0));
  END { DeallocateDirectoryHandle };


PROCEDURE GetSemaphoreInformation(LastSeen : WORD; Name : STRING127;
  VAR GSIReply : GSIReplyType);

  TYPE
    GSIRequestTYPE = RECORD
                       BufLength   : WORD;
                       SubFunction : BYTE;
                       LastRecord  : WORD;
                       SemName     : STRING127;
                     END;
  VAR
    Result     : BYTE;
    GSIRequest : GSIRequestType;

  BEGIN { GetSemaphoreInformation }
    GSIRequest.BufLength   := 5 + Length(Name);
    GSIRequest.SubFunction := $E2;
    GSIRequest.LastRecord  := LastSeen;
    GSIRequest.SemName     := Name;
    GSIRequest.SemName[Length(Name)+1] := CHR(0);
      { Just in case - net bug fix }
    GSIReply.BufLength     := 512;
    Result := NetCommand($E3, GSIRequest, GSIReply);
    IF Result <> 0 THEN
      FatalError(Result, 'Get Semaphore Information',
        'Last Seen '+StrWORD(LastSeen,0)+', Name "'+Name+'"');
  END { GetSemaphoreInformation };


PROCEDURE GetConnectionSemaphores(    ConnectionNum : BYTE;
                                      LastSeen      : WORD;
                                  VAR GCSReply      : GCSReplyType);

  TYPE
    GCSRequestType = RECORD
                       BufLength   : WORD;
                       SubFunction : BYTE;
                       Connection  : WORD;
                       LastRecord  : WORD;
                     END;

  VAR
    Result     : BYTE;
    GCSRequest : GCSRequestType;

  BEGIN { GetConnectionSemaphores }
    GCSRequest.BufLength   := 5;
    GCSRequest.SubFunction := $E1;
    GCSRequest.Connection  := Swap(ConnectionNum);
    GCSRequest.LastRecord  := LastSeen;
    GCSReply.BufLength     := 512;
    Result := NetCommand($E3, GCSRequest, GCSReply);
    IF Result <> 0 THEN
      FatalError(Result, 'Get Connection''s Semaphores',
        'Connection '+StrBYTE(ConnectionNum,0)+', Last Seen '+
        StrWORD(LastSeen,0));
  END { GetConnectionSemaphores };


PROCEDURE OpenSemaphore(Name : STRING127; Value : SHORTINT;
  VAR OpenCount : BYTE; VAR Handle : LONGINT); 

  VAR
    R : Registers;

  BEGIN { OpenSemaphore }
    WITH R DO
      BEGIN
        Name[Length(Name)+1] := CHR(0); { Just in case - net bug fix }
        AX        := $C500;
        DS        := SEG(Name); { Note: STRING not ASCIIZ - net bug fix }
        DX        := OFS(Name);
        CL        := BYTE(Value);
        MSDOS(R);
        IF AL <> 0 THEN
          FatalError(AL, 'Open Semaphore', 'Name "'+Name+'", Value '
          +StrBYTE(BYTE(Value),0));
        OpenCount := BL;
        Handle    := (DX SHL 16) OR CX;
      END;
  END { OpenSemaphore };

 
PROCEDURE CloseSemaphore(Handle : LONGINT);

  VAR
    R : Registers;

  BEGIN { CloseSemaphore }
    WITH R DO
      BEGIN
        AX := $C504;
        CX := Handle AND $FFFF;
        DX := (Handle SHR 16) AND $FFFF;
        MSDOS(R);
        IF AL <> 0 THEN
          FatalError(AL, 'Close Semaphore', Hex8(Handle));
      END;
  END { CloseSemaphore };


PROCEDURE LogNetworkMessage(MessageArg : STRING80);

  TYPE
    LNMRequestType = RECORD
                       BufLength   : WORD;
                       SubFunction : BYTE;
                       Message     : STRING80;
                     END;
    LNMReplyType   = RECORD
                       BufLength   : WORD;
                     END;

  VAR
    Result     : BYTE;
    LNMRequest : LNMRequestType;
    LNMReply   : LNMReplyType;

  BEGIN { LogNetworkMessage }
    LNMRequest.BufLength   := 2 + Length(MessageArg);
    LNMRequest.SubFunction := $0D;
    LNMRequest.Message     := MessageArg;
    LNMReply.BufLength     := 0;
    Result := NetCommand($E3, LNMRequest, LNMReply);
  END { LogNetworkMessage };


PROCEDURE ScanObject(VAR SORequest : SORequestType; VAR SOReply : SOReplyType);

  VAR
    Result : BYTE;

  BEGIN { ScanObject }
    SORequest.BufLength   := 8 + Length(SORequest.ObjName);
    SORequest.SubFunction := $37;
    SOReply.BufLength     := 57;
    Result := NetCommand($E3, SORequest, SOReply);
    IF Result = 252 THEN { No such object }
      SOReply.BufLength := 0
    ELSE
    IF Result <> 0 THEN
      FatalError(Result, 'Scan Object', 'Last '+
        StrLONGINT(Swap4(SORequest.LastObjID),0)+', '+
        ObjTypeName(SORequest.ObjType)+', Name "'+SORequest.ObjName+'"');
  END { ScanObject };


PROCEDURE ScanProperty(VAR SPRequest : SPRequestType; VAR SPReply:SPReplyType);

  VAR
    Result    : BYTE;
    pLastSeen : ^LONGINT;
    pPropName : ^STRING15;

  BEGIN { ScanProperty }
    WITH SPRequest DO
      BEGIN
        pLastSeen   := PTR(SEG(ObjName[1]), OFS(ObjName[1]) + Length(ObjName));
        pPropName   := PTR(SEG(pLastSeen^), OFS(pLastSeen^) + SIZEOF(LONGINT));
        BufLength   := 9 + Length(ObjName) + Length(pPropName^);
        SubFunction := $3C;
        SPReply.BufLength := 24;
        Result := NetCommand($E3, SPRequest, SPReply);
        IF Result = 251 THEN { No such property }
          SPReply.BufLength := 0
        ELSE
        IF Result <> 0 THEN
          FatalError(Result, 'Scan Property', 
            ObjTypeName(ObjType)+', Object Name "'+
            ObjName+'", Sequence # '+StrLONGINT(Swap4(pLastSeen^),0)+
            ', Property Name "'+pPropName^+'"');
      END;
  END { ScanProperty };


PROCEDURE ReadPropertyValue(VAR RPVRequest : RPVRequestType;
  VAR RPVReply : RPVReplyType);

  VAR
    Result    : BYTE;
    pSegment  : ^BYTE;
    pPropName : ^STRING15;

  BEGIN { ReadPropertyValue }
    WITH RPVRequest DO
      BEGIN
        pSegment    := PTR(SEG(ObjName[1]), OFS(ObjName[1]) + Length(ObjName));
        pPropName   := PTR(SEG(pSegment^), OFS(pSegment^) + SIZEOF(BYTE));
        BufLength   := 6 + Length(ObjName) + Length(pPropName^);
        SubFunction := $3D;
        RPVReply.BufLength := 130;
        Result := NetCommand($E3, RPVRequest, RPVReply);
        IF (Result = 236) OR (Result = 251) THEN { No such segment/property }
          RPVReply.BufLength := 0
        ELSE
        IF Result <> 0 THEN
          FatalError(Result, 'Read Property Value', 
            ObjTypeName(RPVRequest.ObjType)+', Object Name "'+
            RPVRequest.ObjName+'", Segment '+StrBYTE(pSegment^,0)+
            ', Property Name "'+pPropname^+'"');
      END;
  END { ReadPropertyValue };


FUNCTION GetObjectName(x : LONGINT; VAR ObjectType : WORD) : STRING47;

  TYPE
    GONRequestType = RECORD
                       BufLength   : WORD;
                       SubFunction : BYTE;
                       ObjID       : LONGINT;
                     END;
    GONReplyType   = RECORD
                       BufLength   : WORD;
                       ObjID       : LONGINT;
                       ObjType     : WORD;
                       ObjName     : ASCIIZ48;
                     END;
  VAR
    Result     : BYTE;
    GONRequest : GONRequestType;
    GONReply   : GONReplyType;

  BEGIN { GetObjectName }
    GONRequest.BufLength   := 5;
    GONRequest.SubFunction := $36;
    GONRequest.ObjID       := x;
    GONReply.BufLength     := 54;
    Result := NetCommand($E3, GONRequest, GONReply);
    IF Result <> 0 THEN
      FatalError(Result, 'Get Object Name', StrLONGINT(Swap4(x),0));
    ObjectType    := GONReply.ObjType;
    GetObjectName := StrName48(GONReply.ObjName);
  END { GetObjectName };
{
  On return MSB = network pathbase
            LSB = base flags
                  00h not mapped
                  01H permanent map
                  02H temporary map
                  03H local
}
FUNCTION GetBaseStatus(Drive : DriveType) : WORD;

  VAR
    R : Registers;

  BEGIN { GetBaseStatus }
    WITH R DO
      BEGIN
        AX := $E900;
        DX := ORD(Drive)-ORD('A');
        MSDOS(R);
        IF (Flags AND FCarry) <> 0 THEN
          FatalError(AL, 'Get Base Status', Drive);
        GetBaseStatus := AX;
      END;
  END { GetBaseStatus };


PROCEDURE MapFakeRootDirectory(Drive : DriveType; Path : STRING);

  VAR
    R : Registers;

  BEGIN { MapFakeRootDirectory }
    WITH R DO
      BEGIN
        Path := Path + CHR(0);
        AX   := $E905;
        BL   := ORD(Drive)-ORD('A')+1;
        DS   := SEG(Path[1]);
        DX   := OFS(Path[1]);
        MSDOS(R);
        IF (Flags AND FCarry) <> 0 THEN
          FatalError(AL, 'Map Fake Root Directory', 'Drive '+Drive+', Path '
            +Path);
      END;
  END { MapFakeRootDirectory };


PROCEDURE DeleteFakeRootDirectory(Drive : DriveType);

  VAR
    R : Registers;

  BEGIN { DeleteFakeRootDirectory }
    WITH R DO
      BEGIN
        AX := $E906;
        BL := ORD(Drive)-ORD('A')+1;
        MSDOS(R);
        IF (Flags AND FCarry) <> 0 THEN
          FatalError(AL, 'Delete Fake Root Directory', Drive);
      END;
  END { DeleteFakeRootDirectory };


FUNCTION GetRelativeDriveDepth(Drive : DriveType) : BYTE;

  VAR
    R : Registers;

  BEGIN { GetRelativeDriveDepth }
    WITH R DO
      BEGIN
        AX := $E907;
        BL := ORD(Drive)-ORD('A')+1;
        MSDOS(R);
        IF (Flags AND FCarry) <> 0 THEN
          FatalError(AL, 'Get Relative Drive Depth', Drive);
        GetRelativeDriveDepth := AL;
      END;
  END { GetRelativeDriveDepth };
{
  Note: when attaching with this function it is the calling program's
        responsibility to update the file server mapping table in the shell.
}
PROCEDURE AttachToFileServer(ConnectionID : BYTE);

  VAR
    R : Registers;

  BEGIN { AttachToFileServer }
    WITH R DO
      BEGIN
        AX := $F100;
        DL := ConnectionID;
        MSDOS(R);
        IF AL <> 0 THEN
          FatalError(AL, 'Attach To File Server', StrBYTE(ConnectionID,0));
      END;
  END { AttachToFileServer };
{
  Note: when detaching with this function it is the calling program's
        responsibility to update the file server mapping table in the shell.
}
PROCEDURE DetachFromFileServer(ConnectionID : BYTE);

  VAR
    R : Registers;

  BEGIN { DetachFromFileServer }
    WITH R DO
      BEGIN
        AX := $F101;
        DL := ConnectionID;
        MSDOS(R);
      END;
  END { DetachFromFileServer };


PROCEDURE LoginToFileServer(ObjectType  : WORD;
                            ObjectName  : STRING47;
                            PasswordArg : STRING27);
  TYPE
    LTFSRequestType = RECORD
                        BufLength   : WORD;
                        SubFunction : BYTE;
                        ObjType     : WORD;
                        ObjName     : STRING47; { Warning - PACKED }
                        Password    : STRING27; { Warning - PACKED }
                      END;
    LTFSReplyType   = RECORD
                        BufLength : WORD;
                      END;

  VAR
    Result      : BYTE;
    pPassword   : ^STRING27;
    LTFSRequest : LTFSRequestType;
    LTFSReply   : LTFSReplyType;

  BEGIN { LoginToFileServer }
    WITH LTFSRequest DO
      BEGIN
        BufLength   := 5 + Length(ObjectName) + Length(PasswordArg);
        SubFunction := $14;
        ObjType     := ObjectType;
        ObjName     := ObjectName;
        pPassword   := PTR(SEG(ObjName[1]), OFS(ObjName[1])+Length(ObjName));
        pPassword^  := PasswordArg;
        LTFSReply.BufLength := 0;
        Result := NetCommand($E3, LTFSRequest, LTFSReply);
        IF Result <> 0 THEN
          FatalError(Result, 'Login To File Server', 
            ObjTypeName(ObjectType)+', Name "'+ObjectName+
            '", Password '+pPassword^);
      END;
  END { LoginToFileServer };


PROCEDURE Logout;

  VAR
    R : Registers;

  BEGIN { Logout }
    WITH R DO
      BEGIN
        AH := $D7;
        MSDOS(R);
      END;
  END { Logout };


PROCEDURE LogoutFromFileServer(ConnectionID : BYTE);

  VAR
    R : Registers;

  BEGIN { LogoutFromFileServer }
    WITH R DO
      BEGIN
        AX := $F102;
        DL := ConnectionID;
        MSDOS(R);
      END;
  END { LogoutFromFileServer };


PROCEDURE GetObjectConnectionNumbers(VAR GOCNRequest : GOCNRequestType;
  VAR GOCNReply : GOCNReplyType);

  VAR
    Result : BYTE;

  BEGIN { GetObjectConnectionNumbers }
    GOCNRequest.BufLength   := 4 + Length(GOCNRequest.ObjName);
    GOCNRequest.SubFunction := $15;
    GOCNReply.BufLength     := 101;
    Result := NetCommand($E3, GOCNRequest, GOCNReply);
    IF Result <> 0 THEN
      FatalError(Result, 'Get Object Connection Numbers',
        ObjTypeName(GOCNRequest.ObjType)+', Name "'+GOCNRequest.ObjName+'"');
  END { GetObjectConnectionNumbers };


PROCEDURE GetConnectionUsageStatistics(ConnectionNum : BYTE;
  VAR GCUSReply : GCUSReplyType);

  TYPE
    GCUSRequestType = RECORD
                        BufLength   : WORD;
                        SubFunction : BYTE;
                        Connection  : WORD;
                      END;
  VAR
    Result      : BYTE;
    GCUSRequest : GCUSRequestType;

  BEGIN { GetConnectionUsageStatistics }
    GCUSRequest.BufLength   := 3;
    GCUSRequest.SubFunction := $E5;
    GCUSRequest.Connection  := Swap(WORD(ConnectionNum));
    GCUSReply.BufLength     := 20;
    Result := NetCommand($E3, GCUSRequest, GCUSReply);
    IF Result <> 0 THEN
      FatalError(Result, 'Get Connection Usage Statistics',
        StrBYTE(ConnectionNum,0));
  END { GetConnectionUsageStatistics };


PROCEDURE GetFileServerInformation(VAR GFSIReply : GFSIReplyType);

  TYPE
    GFSIRequestType = RECORD
                        BufLength   : WORD;
                        SubFunction : BYTE;
                      END;
  VAR
    Result      : BYTE;
    GFSIRequest : GFSIRequestType;

  BEGIN { GetFileServerInformation }
    GFSIRequest.BufLength   := 1;
    GFSIRequest.SubFunction := $11;
    GFSIReply.BufLength     := 128;
    Result := NetCommand($E3, GFSIRequest, GFSIReply);
    IF Result <> 0 THEN
      FatalError(Result, 'Get File Server Information', '');
  END { GetFileServerInformation };


PROCEDURE GetFileServerDateAndTime(VAR GFSDATReply : GFSDATReplyType);

  VAR
    R : Registers;

  BEGIN { GetFileServerDateAndTime }
    WITH R DO
      BEGIN
        AH := $E7;
        DS := SEG(GFSDATReply);
        DX := OFS(GFSDATReply);
        MSDOS(R);
      END;
  END { GetFileServerDateAndTime };


FUNCTION CheckConsolePrivileges : BOOLEAN;

  TYPE
    CCPRequestType = RECORD
                       BufLength   : WORD;
                       SubFunction : BYTE;
                     END;
    CCPReplyType   = RECORD
                       BufLength   : WORD;
                     END;
  VAR
    Result     : BYTE;
    CCPRequest : CCPRequestType;
    CCPReply   : CCPReplyType;

  BEGIN { CheckConsolePrivileges }
    CCPRequest.BufLength   := 1;
    CCPRequest.SubFunction := $C8;
    CCPReply.BufLength     := 0;
    Result := NetCommand($E3, CCPRequest, CCPReply);
    CheckConsolePrivileges := (Result = 0);
  END { CheckConsolePrivileges };

{
  Broadcast mode  Server action  Shell action
        0         Store          Retrieve/Display
        1         Reject         Retrieve/Display
        2         Reject         Ignore
        3         Store          Ignore (Use GetBroadcastMessage)
}

FUNCTION  GetBroadcastMode : BYTE;

  VAR
    R : Registers;

  BEGIN { GetBroadcastMode }
    WITH R DO
      BEGIN
        AH := $DE;
        DL := $04;
        MSDOS(R);
        GetBroadcastMode := AL;
      END;
  END { GetBroadcastMode };


FUNCTION SetBroadcastMode(Mode : BYTE) : BYTE;

  VAR
    R : Registers;

  BEGIN { SetBroadcastMode }
    WITH R DO
      BEGIN
        AH               := $DE;
        DL               := Mode;
        MSDOS(R);
        SetBroadcastMode := AL;
      END;
  END { SetBroadcastMode };


FUNCTION  GetBroadcastMessage : STRING55;

  TYPE
    GBMRequestType = RECORD
                       BufLength   : WORD;
                       SubFunction : BYTE;
                     END;
    GBMReplyType   = RECORD
                       BufLength   : WORD;
                       Message     : STRING55;
                     END;

  VAR
    Result     : BYTE;
    GBMRequest : GBMRequestType;
    GBMReply   : GBMReplyType;

  BEGIN { GetBroadcastMessage }
    GBMRequest.BufLength   := 1;
    GBMRequest.SubFunction := $01;
    GBMReply.BufLength     := 56;
    GBMReply.Message[0]    := CHR(55);
    Result := NetCommand($E1, GBMRequest, GBMReply);
    IF Result <> 0 THEN
      FatalError(Result, 'Get Broadcast Message', '');
    GetBroadcastMessage    := GBMReply.Message;
  END { GetBroadcastMessage };


PROCEDURE SendBroadcastMessage(VAR SBMRequest : SBMRequestType;
  VAR SBMReply : SBMReplyType);

  VAR
    Result   : BYTE;
    pMessage : ^STRING55;

  BEGIN { SendBroadcastMessage }
    pMessage               := PTR(SEG(SBMRequest.Connect[1]),
      OFS(SBMRequest.Connect[1])+Length(SBMRequest.Connect));
    SBMRequest.BufLength   := 3 + Length(SBMRequest.Connect) +
      Length(pMessage^);
    SBMRequest.SubFunction := $00;
    SBMReply.BufLength     := 101;
    SBMReply.Connect[0]    := CHR(100);
    Result := NetCommand($E1, SBMRequest, SBMReply);
    IF Result <> 0 THEN
      FatalError(Result, 'Send Broadcast Message', 'Connect length '+
        StrBYTE(Length(SBMRequest.Connect),0)+', Message "'+pMessage^+'"');
  END { SendBroadcastMessage };


PROCEDURE SendConsoleBroadcast(VAR SCBRequest : SCBRequestType);

  TYPE
    SCBReplyType = RECORD
                     BufLength : WORD;
                   END;

  VAR
    Result   : BYTE;
    pMessage : ^STRING59;
    SCBReply : SCBReplyType;

  BEGIN { SendConsoleBroadcast }
    pMessage               := PTR(SEG(SCBRequest.Connect[1]),
      OFS(SCBRequest.Connect[1])+Length(SCBRequest.Connect));
    SCBRequest.BufLength   := 3 + Length(SCBRequest.Connect) +
      Length(pMessage^);
    SCBRequest.SubFunction := $D1;
    SCBReply.BufLength     := 0;
    Result := NetCommand($E3, SCBRequest, SCBReply);
    IF Result <> 0 THEN
      FatalError(Result, 'Send Console Broadcast', 'Connect length '+
        StrBYTE(Length(SCBRequest.Connect),0)+', Message "'+pMessage^+'"');
  END { SendConsoleBroadcast };
{
  Converts a path specification to a form which specifies the true disk
  and directory on a file server eg. SYS:PUBLIC\SYSCON.EXE.
  Also returns the name and number of the file server the path is on.
  If file is not on a file server it is FExpand'ed and the file server
  name returned is ''. Possible inputs might include:
    T, Z:T, Z:\T, Z:D\T, Z:..\T, SYS:T, SYS:D\T, BASIL/SYS:D\T, Z:SYS:D\T ...
  It is complicated by the fact that Z:\ could refer to an arbitrary network
  directory.
}
PROCEDURE NetFExpand(
      PathOld  : PathStr;
  VAR PathNew  : PathStr;
  VAR FSName   : STRING47;
  VAR FSNumber : FSNType);

  VAR
    I, 
    PreferredID,
    PosSep,
    PosColon1,
    PosColon2  : BYTE;
    pDHT       : pDriveHandleTable;
    pDCIT      : pDriveConIDTable;
    PathTmp1,
    PathTmp2   : PathStr;

  PROCEDURE GetFS;

    VAR
      pCIT         : pConIDTable;
      pFSNT        : pFSNameTable;
      ConnectionID : BYTE;

    BEGIN { GetFS }
      ConnectionID := GetDefaultConnectionID;
      pFSNT        := GetFileServerNameTable;
      FSName       := StrName48(pFSNT^[ConnectionID]);
      pCIT         := GetConnectionIDTable;
      FSNumber     := pCIT^[ConnectionID].NetNumber;
    END { GetFS };

  BEGIN { NetFExpand }
    PathOld   := Normalize(PathOld);
    PosColon1 := Pos(':', Copy(PathOld, 2, 255));
    IF PosColon1 <> 0 THEN
      BEGIN
        PosColon1 := PosColon1 + 1;
        PosColon2 := Pos(':', Copy(PathOld, PosColon1+1, 255));
        IF PosColon2 <> 0 THEN
          PosColon2 := PosColon2 + PosColon1;
      END
    ELSE
      PosColon2 := 0;

    IF (PosColon1 = 0) OR ((PosColon1 = 2) AND (PosColon2 = 0)) THEN
      BEGIN { MSDOS style path, expand it }
        PathOld   := Normalize(FExpand(PathOld));
        PosColon1 := 2;
        PosColon2 := 0;
        IF (Length(PathOld) < 2) OR (PathOld[2] <> ':') THEN
          FatalError(0, 'Net Filename Expand',
            'Sorry, internal error. FExpand failed giving "'+PathOld+'"');
      END;

    IF (PosColon1 = 2) AND (PosColon2 = 0) THEN { MSDOS style path, convert }
      BEGIN
        I := 0;
        IF NovellAPI THEN
          BEGIN
            pDHT := GetDriveHandleTable;
            IF (PathOld[1] < 'A') OR (PathOld[1] >= 'a') THEN
              I := 0
            ELSE
              I := pDHT^[PathOld[1]];
          END;
        IF I = 0 THEN { Mapped on to local disk }
          BEGIN
            PathNew  := PathOld;
            FSName   := '';
            FSNumber := 0;
          END
        ELSE { Mapped on to network disk }
          BEGIN { Get true volume:directory of network drive }
            PreferredID := GetPreferredConnectionID;
            pDCIT       := GetDriveConnectionIDTable;
            SetPreferredConnectionID(pDCIT^[PathOld[1]]);
            PathNew  := Normalize(GetDirectoryPath(I));   { Get net def dir}
            GetDir(ORD(PathOld[1])-ORD('A')+1, PathTmp1); { Get DOS def dir}
            PathTmp1 := Copy(Normalize(PathTmp1), 4, 255);{ Strip DOS drive}
            PathTmp2 := Copy(PathNew,Length(PathNew)-Length(PathTmp1)+1,255);
            IF PathTmp1 <> PathTmp2 THEN
              FatalError(0, 'Net Filename Expand',
                'Sorry, internal error. DOS/net dir mismatch "'+PathTmp1+
                '" <> "'+PathTmp2+'"');
            PathNew  := Copy(PathNew, 1, Length(PathNew)-Length(PathTmp1))
              + Separator + Copy(PathOld, 4, 255); { True network directory }
            GetFS;
            SetPreferredConnectionID(PreferredID);
          END;
      END
    ELSE { Network style path }
      BEGIN
        IF NOT NovellAPI THEN
          FatalError(0, 'Net Filename Expand',
            'Need Novell Netware (tm) to interpret "'+PathOld+'"');
        IF (PosColon1 = 2) AND (PosColon2 <> 0) THEN { Strip MSDOS drive }
          BEGIN
            PathOld   := Copy(PathOld, 3, 255);
            PosColon1 := PosColon2-2;
            PosColon2 := 0;
          END;
        PosSep := Pos(Separator, PathOld);
        IF (PosSep = 0) OR (PosSep > PosColon1) THEN
          BEGIN { No file server in file specification, get default }
            PathNew  := PathOld;
            GetFS;
          END
        ELSE { File server name in file specification, get number }
          BEGIN
            PathNew  := Copy(PathOld, PosSep+1, 255);
            FSName   := Copy(PathOld, 1, PosSep-1);
            FSNumber := GetFileServerNumber(FSName);
          END;
      END;
  END { NetFExpand };


END { NOS }.

