PROGRAM NetLock(INPUT, OUTPUT);
{$I Options}

{
  Copyright (C) 1991,1992 Julian Byrne. All rights reserved.

  Title:	NETLOCK
  File name:	NETLOCK.PAS
  Version:	See PROCEDURE Usage;
  Usage:	See PROCEDURE Usage;
  Description:	Modifies an application binary to add security options
		on a Novell Netware (tm) network.
  Dependencies: See USES statement
  Author:	Julian Byrne
  Address:	Electrical and Computer Systems Engineering Department
		Monash University, Wellington Road, Clayton, Victoria, 3168
		Australia
  Internet:	julian.byrne@eng.monash.edu.au.
  Other nets:	Quaterman & Hoskins "Notable Computer Networks"
		CACM Oct'86 pp932-971
  History:	91/ 5/ 1 Initial version
		91/ 7/21 Replaced IsUser/GetUser with IsSupervisor/GetSupervisor
			 Modified code to handle multiple lock versions
		92/10/10 Added /W option
		93/ 1/26 Fixed bug reading v1.0/1.1 locked programs.
  Notes:
}

  USES
    CRT,
    DOS,
    ASCII,	{ ASCII constants }
    Error,	{ Error handling routines }
    CheckIO,	{ Checked I/O handling routines }
    NOS;	{ Network Operating System subroutine library }

  CONST
    TicksPerSec   = 18.2;     { PC clock interrupts per second		      }
    EOFMark	  = CHR($00); { Used to mark end-of-file in command parsing   }
    CTRLZ	  = CHR($1A);
		    { These must match the definitions in APPLOCK.ASM	      }
    LockLog10	  = 'Lock log v1.0'+CR+LF+CTRLZ; { Lock log header records    }
    LockLog11	  = 'Lock log v1.1'+CR+LF+CTRLZ;
    Signature10   = 'Application Lock  v1.0'+CTRLZ+NUL; { Lock code signatures}
    Signature11   = 'Application Lock  v1.1'+CTRLZ+NUL;
    SignatureSize = Length(Signature10);

    JMPDWORD	  = $EA;  { Used to initialize lock data		      }
    LockEndMark   = $AA;  { Used to check assembly language program linked ok }
    NumMess	  = 12;   { Number of messages in lock data		      }
    GetCharLimit  = 8;	  { Maximum number of nested command response files   }
    IndirectChar  = '@';  { Marks a response file in a command line           }
    CommentChar   = ';';  { Marks a comment in a command line                 }
    StackSize	  = $200; { Stack space to set aside for application lock     }
    MessLimit	  = NumMess*255+15; { Total length limit of all lock messages }
			    { Default file extensions (When no '.' present)   }
    ExtLog	  = '.LOG'; { Log                                             }
    ExtTmp	  = '.$$$'; { Temporary                                       }
    ExtExe	  = '.EXE'; { Application                                     }
    ExtRes	  = '.RES'; { Response                                        }
    ExtUnl	  = '.UNL'; { Original, unlocked application (when locking)   }
    ExtLok	  = '.LOK'; { Original, locked application (when unlocking)   }
{
  Default switch values. See NETLOCK.DOC for a description.
}
    Xdef	  = FALSE;	{ Execute only	  }
    Udef	  = 0;		{ User limit	  }
    Wdef	  = 0;		{ No wait time	  }
    Ldef	  = '';         { Log file name   }
    Ndef	  = '';         { Notify address  }
    Ndef2	  = Supervisor; { If /N specified but no arg }
    Pdef	  = '';         { Password }
    FSdef	  = '';         { File server }
    FVdef	  = 0;		{ File server }
    Sdef	  = '';         { Semaphore name }
    MAdef	  = 'Sorry, error accessing lock. Contact supervisor.';
    MFdef1	  = 'Sorry, error accessing file server "';
    MFdef2	  = '". Contact supervisor.';
    MLdef	  = 'Sorry, error accessing log file. Contact supervisor.';
    MNdef	  = ' started'; { Notify message  }
    MPdef	  = 'Sorry, incorrect password.';
    MQdef	  = 'Password: ';
    MSdef	  = '';         { Success message }
    MUdef1	  = 'Sorry, user limit of ';
    MUdef2	  = ' exceeded. Try again later.';

  TYPE
{
  Binary log file record. See NETLOCK.DOC for a description of these fields.
}
    LogRecord	  = RECORD
		      LogNum,
		      LogLim,
		      LogCon	    : BYTE;
		      FSDAT	    : GFSDATReplyType;
		      IALen	    : WORD;
		      NetNum	    : FSNType;
		      NodeAddr	    : NetAddress;
		      Socket,
		      CILen	    : WORD;
		      ObjID	    : LONGINT;
		      ObjType	    : WORD;
		      ObjName	    : ASCIIZ48;
		      LogTime	    : GFSDATReplyType;
		      PassOk	    : BYTE;
		    END;
{
  The data record filled in by this program and appended to the modified
  application.
  It contains data from the EXE header necessary to start the application
  as usual once the lock is finished, and data specifying which actions the
  lock is to perform. This record must match the definition in APPLOCK.ASM.
}
    LockDataType  =
    RECORD
      DataSize	   : WORD;    { LockDataType size		       }
      Filler	   : BYTE;
      JmpByte	   : BYTE;    { JMP [DWORD PTR Chain]		       }
      Chain,		      { Application EXE Header StartCS/StartIP }
      UStack	   : POINTER; {      "       "    "    StartSS/StartSP }
      SizeMin,		      {      "       "    "    MinSize         }
      SizeMax,		      {      "       "    "    MaxSize         }
      SizeLP,		      {      "       "    "    LastPageSize    }
      SizePF,		      {      "       "    "    PageFileSize    }
      SizeLSW,		      {      "     Size Least Significant Word }
      SizeMSW	   : WORD;    {      "     Size Most  Significant Word }
      Limit	   : BYTE;    { Maximum number of users      }
      LogMode	   : BYTE;    { Log file share access flags  }
      LogAttr	   : WORD;    { Log file creation attributes }
      SemServer    : FSNType; { File server number locked to }
      WaitTime	   : WORD;    { Number of clock ticks to wait }
      pMess	   : ARRAY[1..NumMess] OF WORD; { Pointers into 'Messages' }
      Messages	   : ARRAY[0..MessLimit] OF CHAR;
    END;

  VAR
    Header	   : EXEHeader; { Memory image of application EXE header   }
    ExeBak,			{ Name of application backup		   }
    ExeInp,			{ Application being locked/unlocked	   }
    ExeOut,			{ Temporary file or result file 	   }
    ExeRes	   : PathStr;	{ Modified application result file	   }
    sMA,			{ Lock access error message - switch value }
    sMF,			{ File server access error message	   }
    sML,			{ Log file access error message 	   }
    sMN,			{ Notification message			   }
    sMP,			{ Password error message		   }
    sMQ,			{ Password prompt message		   }
    sMS,			{ Successful application start message	   }
    sMU,			{ User limit exceeded error message	   }
    sP, 			{ Password. '' = no password               }
    sS, 			{ Lock semaphore name			   }
    sL, 			{ Log file name. '' = no log file          }
    sN		   : STRING;	{ User to notify. '' = no notification     }
    sX		   : BOOLEAN;	{ Execute-only. FALSE = not execute-only   }
    sU, 			{ User limit. 0 = no user limit 	   }
    sW		   : WORD;	{ Clock ticks to wait when displaying	   }
    sFV 	   : FSNType;	{ File server to lock to - LONGINT	   }
    sFS 	   : STRING47;	{ File server to lock to - STRING	   }
    ExeCount	   : WORD;	{ Number of arguments - 1 or 2		   }
    OutOnServer,		{ Whether output file should be shared	   }
    LogOnServer    : BOOLEAN;	{ Whether log file should be shared	   }
    fBak,			{ Application backup			   }
    fInp,			{ Original application			   }
    fOut	   : FILE;	{ Modified application			   }
    ParFill,			{ 0-15 - bytes to fill app. to para. bound }
    CodeSizeFile,		{ Size of lock code found in file	   }
    DataSizeFile,		{ Size of lock data found in file	   }
    CodeSizeMem    : WORD;	{ Size of lock code to be appended	   }
    LockData	   : LockDataType; { Where lock data is prepared	   }
    GetCharTable   : ARRAY[0..GetCharLimit] OF FILE;{ Nested response files}
    GetCharCount,		{ Number of currently open response files  }
    GetCharPoint   : INTEGER;	{ Position in command line arguments	   }
    SwitchChar	   : CHAR;	{ Switch character, normally '/'           }
    Switches	   : BOOLEAN;	{ TRUE if there is at least one switch	   }
    ExitSave	   : POINTER;	{ Next exit handler in chain		   }
    SaveDrive	   : DriveType; { Recover default drive on error exit	   }
    SaveDir	   : PathStr;	{ Recover default directory on error exit  }
    SaveFile	   : PathStr;	{ Delete unfinished output on error exit   }
    MyDir	   : DirStr;	{ Paramstr(0) split up			   }
    MyName	   : NameStr;
    MyExt	   : ExtStr;
    LogID,
    PreferredID    : BYTE;	{ Preferred connection ID		   }

  PROCEDURE LockBegin; EXTERNAL; { Lock to attach to application }

    {$L LockBegin}

  PROCEDURE LockEnd; EXTERNAL; { Pointer to end of lock }

    {$L LockEnd}

  PROCEDURE DocBegin; EXTERNAL; { Program documentation - see DocumentList }

    {$L DocBegin}

  PROCEDURE DocEnd; EXTERNAL; { End of program documentation }

    {$L DocEnd }
{
  If there is a fatal error this exit handler will restore things to normal.
  Default drive restored, default directory restored and deletion of an
  uncompleted output file if it exists.
}
{$IFOPT F-}	 { Save $F state }
{$DEFINE FMINUS}
{$F+}
{$ENDIF}
  PROCEDURE ExitHandler;

    VAR
      f     : FILE;

    BEGIN { ExitHandler }
      ExitProc := ExitSave;
      IF SaveDir <> '' THEN
	BEGIN
	  ChDirCheck(SaveDir);
	  SaveDir := '';
	END;
      IF SaveDrive <> DriveUndefined THEN
	BEGIN
	  SelectDisk(SaveDrive);
	  SaveDrive := DriveUndefined;
	END;
      IF SaveFile <> '' THEN
	BEGIN
	  AssignCheck(f, SaveFile);
	  EraseIfPresent(f);
	  SaveFile := '';
	END;
    END { ExitHandler };
{$IFDEF FMINUS} { Restore $F state }
{$UNDEF FMINUS}
{$F-}
{$ENDIF}
{
  Convert a paragraph size to a byte size.
}
  FUNCTION ParToByte(x : WORD) : LONGINT;

    BEGIN { ParToByte }
      ParToByte := LONGINT(x) SHL 4;
    END { ParToByte };
{
  Convert a byte size to a paragraph size.
}
  FUNCTION ByteToPar(x : LONGINT) : WORD;

    BEGIN { ByteToPar }
      ByteToPar := WORD(((x+$F) SHR 4) AND $FFFF);
    END { ByteToPar };
{
  Get number of bytes to round a value to the next paragraph (16 byte)
  boundary.
}
  FUNCTION GetParFill(x : WORD) : WORD;

    BEGIN { GetParFill }
      GetParFill := ((x+$000F) AND $FFF0)-x;
    END { GetParFill };
{
  Generates a random 8 character password from among 243101250 possibilities.
}
  FUNCTION GenPassword : STRING8;

    CONST
      Consonants : ARRAY[0..20] OF CHAR = 'BCDFGHJKLMNPQRSTVWXYZ';
      Vowels	 : ARRAY[0.. 4] OF CHAR = 'AEIOU';
      Digits	 : ARRAY[0.. 9] OF CHAR = '0123456789';

    BEGIN { GenPassword }
      GenPassword := Consonants[Random(SIZEOF(Consonants))]
	+ Vowels[Random(SIZEOF(Vowels))]
	+ Consonants[Random(SIZEOF(Consonants))]
	+ Vowels[Random(SIZEOF(Vowels))]
	+ Consonants[Random(SIZEOF(Consonants))]
	+ Vowels[Random(SIZEOF(Vowels))]
	+ Consonants[Random(SIZEOF(Consonants))]
	+ Digits[Random(SIZEOF(Digits))];
    END { GenPassword };
{
  Called to output a message.
  FN (defined in UNIT Error) is used to allow the exit handler to give a
  meaningful error message if this (!) routine fails.
}
  PROCEDURE DisplayMessage(Message : STRING);

    VAR
      f : TEXT;

    BEGIN { DisplayMessage }
      FN := pSTRING(@OutName);
      AssignCRT(f);
      ReWrite(f);
      WriteLn(f, MyName, ': ', Message, '.');
      Close(f);
      FN := NIL;
    END { DisplayMessage };
{
  Called for a warning error condition.
}
  PROCEDURE Warn(Message : STRING);

    BEGIN { Warn }
      DisplayMessage('Warning: '+Message);
    END { Warn };
{
  Called for a fatal error condition.
}
  PROCEDURE Fatal(Message : STRING);

    BEGIN { Fatal }
      DisplayMessage('Fatal error: '+Message);
      Halt;
    END { Fatal };
{
  Return if SUPERVISOR or SUPERVISOR equivalent on the nominated file server.
  Only a user SUPERVISOR equivalent can unlock a file.
  We do this by determining if we can read the SUPERVISOR's login control
  information.
}
  FUNCTION IsSUPERVISOR(FileServerNumber : FSNType) : BOOLEAN;

    VAR
      pCIT	  : pConIDTable;
      Found	  : BOOLEAN;
      I 	  : BYTE;
      PreferredID : BYTE;
      pSegment	  : ^BYTE;
      pPropName   : ^STRING15;
      RPVRequest  : RPVRequestType;
      RPVReply	  : RPVReplyType;

    BEGIN { IsSUPERVISOR }
      PreferredID := GetPreferredConnectionID;
      pCIT	  := GetConnectionIDTable;
      Found	  := FALSE;
      I 	  := 1;
      REPEAT
	WITH pCIT^[I] DO
	  IF InUse AND (ConnectionNum <> $FF) AND ConnectionOk AND
	    (NetNumber = FileServerNumber) THEN
	    BEGIN
	      SetPreferredConnectionID(I);
	      RPVRequest.ObjType := USER;
	      RPVRequest.ObjName := Supervisor;
	      pSegment	 := PTR(SEG(RPVRequest.ObjName[1]),
		OFS(RPVRequest.ObjName[1])+Length(RPVRequest.ObjName));
	      pSegment^  := 1;
	      pPropName  := PTR(SEG(pSegment^), OFS(pSegment^)+SIZEOF(BYTE));
	      pPropName^ := 'LOGIN_CONTROL';
	      ReadPropertyValue(RPVRequest, RPVReply);
	      Found	 := RPVReply.BufLength <> 0;
	    END;
	INC(I);
      UNTIL Found OR (I > 8);
      SetPreferredConnectionID(PreferredID);
      IsSUPERVISOR := Found;
    END { IsSUPERVISOR };
{
  Return a server number we are currently a SUPERVISOR on.
  Used if asked to unlock an application which is not tied to a
  particular file server. Returns 0 if no server number exists.
}
  FUNCTION GetSupervisor : FSNType;

    VAR
      pCIT	  : pConIDTable;
      Found	  : FSNType;
      I 	  : BYTE;
      PreferredID : BYTE;

    BEGIN { GetSupervisor }
      PreferredID := GetPreferredConnectionID;
      pCIT	  := GetConnectionIDTable;
      Found	  := 0;
      I 	  := 1;
      REPEAT
	WITH pCIT^[I] DO
	  IF InUse AND (ConnectionNum <> $FF) AND ConnectionOk THEN
	    IF IsSupervisor(NetNumber) THEN
	      Found := NetNumber;
	INC(I);
      UNTIL (Found <> 0) OR (I > 8);
      GetSupervisor := Found;
    END { GetSupervisor };
{
  Get a mapped MSDOS drive we can use to access a particular file server's
  volumes. Needed to get a directory handle which is on the file server.
}
  FUNCTION GetDrive(ServerNumber : FSNType) : DriveType;

    VAR
      I     : BYTE;
      Result,
      Drive : DriveType;
      pCIT  : pConIDTable;
      pDCIT : pDriveConIDTable;

    BEGIN { GetDrive }
      pCIT   := GetConnectionIDTable;
      pDCIT  := GetDriveConnectionIDTable;
      Result := DriveUndefined;
      Drive  := 'A';
      REPEAT
	I := pDCIT^[Drive];
	IF I <> 0 THEN
	  WITH pCIT^[I] DO
	    IF InUse AND (ConnectionNum <> $FF) AND ConnectionOk AND
	      (NetNumber = ServerNumber) THEN
	      Result := Drive;
	INC(Drive);
      UNTIL (Result <> DriveUndefined) OR (Drive > 'Z');
      GetDrive := Result;
    END { GetDrive };
{
  Make a file execute only. Does this by scanning the file attributes, setting
  the execute-only bit and then setting the file attributes. Complicated by
  the fact that there may be multiple file servers (we need to be talking
  to the right one), and a directory handle is required to access the file.
}
  PROCEDURE ExeOnly(Path : PathStr);

    CONST
      ExecuteOnlyBit = $08; { = VolumeID - can't use MSDOS attribute calls }

    VAR
      Drive	   : DriveType;
      PreferredID,
      ExeID,
      PosColon,
      Handle	   : BYTE;
      ServerName   : STRING47;
      ServerNumber : FSNType;
      SFIRequest   : SFIRequestType;
      SFIReply	   : SFIReplyType;

    BEGIN { ExeOnly }
      NetFExpand(Path, Path, ServerName, ServerNumber);
      ExeID	  := GetConnectionID(ServerName);
      IF ExeID = 0 THEN
	Fatal('Can''t make "'+Path+'" execute-only. No connection to "'+
	  ServerName+'"');
      PreferredID := GetPreferredConnectionID;
      SetPreferredConnectionID(ExeID);
      PosColon := Pos(':', Path);
      Drive    := GetDrive(ServerNumber);
      IF Drive = DriveUndefined THEN
	Fatal('Sorry, can''t make "'+ServerName+Separator+Path+
	  '" execute-only. No mapped drive available');
      SaveDrive := GetCurrentDisk;
      SelectDisk(Drive);
      Handle	:= GetDirectoryHandle(Drive) AND $FF;
      SaveDir	:= GetDirectoryPath(Handle);
      ChDirCheck(Copy(Path, 1, PosColon));
      SFIRequest.ScanData.DirHandle	   := Handle;
      SFIRequest.ScanData.SearchAttributes := 0; { Normal files only }
      SFIRequest.ScanData.FilePath	   := Copy(Path, PosColon+1, 255);
      ScanFileInformation(SeenNone, SFIRequest.ScanData, SFIReply);
      SFIRequest.FileData		   := SFIReply.FileData;
      SFIRequest.FileData.Attributes	   := SFIRequest.FileData.Attributes
	OR ExecuteOnlyBit;
      SetFileInformation(SFIRequest);
      ChDirCheck(SaveDir);
      SaveDir	:= '';
      SelectDisk(SaveDrive);
      SaveDrive := DriveUndefined;
      SetPreferredConnectionID(PreferredID); { Also set ok on error exit }
    END { ExeOnly };
{
  Return if a file is a locked application. It determines this by looking
  to see if there is a signature next to the start address of the
  application. Impossible if it is an execute-only application.
}
  FUNCTION LockedFile(VAR f : FILE; VAR Header : EXEHeader) : BOOLEAN;

    VAR
      Pos      : LONGINT;
      SigTmp   : PACKED ARRAY[1..SignatureSize] OF CHAR;

    BEGIN { LockedFile }
      WITH Header DO
	IF StartIP = (SIZEOF(WORD) + SignatureSize) THEN
	  BEGIN
	    Pos := FilePosCheck(f);
	    SeekCheck(f, LONGINT(HeadSize)*LONGINT($10)
	      +PtrToLong(PTR(StartCS, SIZEOF(WORD))));
	    BlockReadCheck(f, SigTmp, SIZEOF(SigTmp));
	    SeekCheck(f, Pos);
	    LockedFile := (Signature10 = SigTmp) OR
			  (Signature11 = SigTmp);
	  END
	ELSE
	  LockedFile := FALSE;
    END { LockedFile };
{
  Read the locking data from a locked application.
  Impossible if the locked application is execute-only.
}
  PROCEDURE ReadLockData(VAR f : FILE; VAR Header : EXEHeader;
    VAR LockData : LockDataType;
    VAR CodeSizeFile : WORD; VAR DataSizeFile : WORD);

    VAR
      Pos : LONGINT;

    BEGIN { ReadLockData }
      WITH Header, LockData DO
	BEGIN
	  Pos := FilePosCheck(f);
	  SeekCheck(f, LONGINT(Header.HeadSize)*LONGINT($10)
	    +PtrToLong(PTR(Header.StartCS, 0)));
	  BlockReadCheck(f, CodeSizeFile, SIZEOF(CodeSizeFile));
	  SeekCheck(f, FilePosCheck(f)+CodeSizeFile-SIZEOF(CodeSizeFile));
	  BlockReadCheck(f, DataSizeFile, SIZEOF(DataSizeFile));
	  DataSize := DataSizeFile;
	  BlockReadCheck(f, LockData.Filler, DataSize-SIZEOF(DataSize));
	  SeekCheck(f, Pos);
	  CASE CodeSizeFile OF
	    1732, 1847 : { v1.0, v1.1 }
	      BEGIN
		Move(WaitTime, pMess,
		  DataSize-(OFS(WaitTime)-OFS(LockData)));
		INC(DataSize, SIZEOF(WaitTime));
		WaitTime := Wdef;
	      END;
	    1901 : { v1.2, v1.3 }
	      BEGIN
		{ Do nothing }
	      END;
	  ELSE
	    Warn('Unexpected lock code size. Not v1.0-1.3');
	  END;
	END;
    END { ReadLockData };
{
  Convert a string to quoted form. eg. '"Hello" world' -> '"""Hello"" world"'
}
  FUNCTION StringQ(x : STRING) : STRING;

    VAR
      I      : WORD;
      Result : STRING;

    BEGIN { StringQ }
      Result := '"';
      FOR I := 1 TO Length(x) DO
	BEGIN
	  Result := Result + x[I];
	  IF x[I] = '"' THEN
	    Result := Result + '"';
	END;
      Result := Result + '"';
      StringQ := Result;
    END { StringQ };
{
  Display locking data in a form which can be used as a response file
  to relock an application in the same way.
}
  PROCEDURE DumpLockData(VAR LockData : LockDataType);

    VAR
      I,
      Base : WORD;


    FUNCTION String0(Index : WORD) : STRING;

      BEGIN { String0 }
	WITH LockData DO
	  IF pMess[Index] = 0 THEN
	    String0 := ''
	  ELSE
	    String0 := StrName(Messages[pMess[Index]-Base]);
      END { String0 };


    FUNCTION StringL(Index : WORD) : STRING;

      VAR
	I      : WORD;
	Result : STRING;

      BEGIN { StringL }
	WITH LockData DO
	  IF pMess[Index] = 0 THEN
	    StringL := ''
	  ELSE
	    BEGIN
	      I := pMess[Index]-Base;
	      Move(Messages[I], Result, ORD(Messages[I])+1);
	      StringL := Result;
	    END;
      END { StringL };


    BEGIN { DumpLockData }
      WITH LockData DO
	BEGIN
	  FN   := pSTRING(@OutName);
	  I    := 0;
	  REPEAT
	    INC(I);
	    Base := pMess[I];
	  UNTIL (Base <> 0) OR (I >= NumMess);
	  WriteLn(CommentChar, ' Unlocked by:           ', MyName);
	  WriteLn(CommentChar, ' Application:           ', ExeInp);
	  WriteLn(CommentChar, ' Program start address: ', Hex9(Chain));
	  WriteLn(CommentChar, ' Program stack:         ', Hex9(UStack));
	  WriteLn(CommentChar, ' Program size:          ',
	    (SizeMSW SHL 16) OR SizeLSW);
	  IF NOT Xdef THEN
	    Write(CommentChar, ' ');
	  WriteLn(SwitchChar, 'X');
	  IF Limit = UDef THEN
	    Write(CommentChar, ' ');
	  WriteLn(SwitchChar, 'U  ', Limit);
	  IF pMess[2] = 0 THEN
	    Write(CommentChar, ' ');
	  WriteLn(SwitchChar, 'L  ', StringQ(String0(2)));
	  IF pMess[5] = 0 THEN
	    Write(CommentChar, ' ');
	  WriteLn(SwitchChar, 'N  ', StringQ(StringL(5)));
	  IF pMess[3] = 0 THEN
	    Write(CommentChar, ' ');
	  WriteLn(SwitchChar, 'P  ', StringQ(DecodeSTRING(StringL(3))));
	  IF Swap4(SemServer) = FVdef THEN
	    Write(CommentChar, ' ');
	  WriteLn(SwitchChar, 'F  0', Hex8(Swap4(SemServer)),
	    ' ; ', GetFileServerName(SemServer));
	  IF pMess[1] = 0 THEN
	    Write(CommentChar, ' ');
	  WriteLn(SwitchChar, 'S  ', StringQ(StringL(1)));
	  IF WaitTime = WDef THEN
	    Write(CommentChar, ' ');
	  Write(SwitchChar, 'W ');
	  IF WaitTime <> $FFFF THEN
	    Write(WaitTime/TicksPerSec:3:1);
	  WriteLn;
	  WriteLn(SwitchChar, 'MF ', StringQ(StringL(7)));
	  WriteLn(SwitchChar, 'ML ', StringQ(StringL(9)));
	  WriteLn(SwitchChar, 'MA ', StringQ(StringL(8)));
	  WriteLn(SwitchChar, 'MU ', StringQ(StringL(10)));
	  WriteLn(SwitchChar, 'MP ', StringQ(StringL(11)));
	  WriteLn(SwitchChar, 'MQ ', StringQ(StringL(4)));
	  WriteLn(SwitchChar, 'MS ', StringQ(StringL(12)));
	  WriteLn(SwitchChar, 'MN ', StringQ(StringL(6)));
	  FN := NIL;
	END;
    END { DumpLockData };
{
  Display what was done when locking an application.
}
  PROCEDURE Report;

    BEGIN { Report }
      FN := pSTRING(@OutName);
      Write('Original application is:   ');
      IF ExeCount = 1 THEN
	WriteLn(ExeBak)
      ELSE
	WriteLn(ExeInp);
      Write('Modified application is:   ', ExeRes);
      IF sX THEN
	Write(' (Execute-only)');
      WriteLn;
      IF (sFV <> 0) OR (sP <> '') THEN
	BEGIN
	  WriteLn('Restrictions:');
	  IF (sFV <> 0) THEN
	    WriteLn('User must be logged in to: ', Hex8(Swap4(sFV)),
	      ' (', sFS, ')');
	  IF sS <> '' THEN
	    WriteLn('User limit semaphore name: ', sS);
	  IF sU <> 0 THEN
	    WriteLn('Simultaneous user limit:   ', sU);
	  IF sW <> 0 THEN
	    BEGIN
	      Write('Message display wait:      Key press ');
	      IF sW <> $FFFF THEN
		Write('or ', sW/TicksPerSec:4:1, ' seconds')
	      ELSE
		Write('only');
	      WriteLn;
	    END;
	  IF sL <> '' THEN
	    BEGIN
	      Write('File logged to:            ');
	      WriteLn(sL);
	    END;
	  IF sN <> '' THEN
	    WriteLn('User notified:             ', sN);
	  IF sP <> '' THEN
	    WriteLn('Password required:         ', sP);
	END
      ELSE
	WriteLn('No restrictions!');
      FN := NIL;
    END { Report };

{ The sayings come from the Unix fortune cookie program database
  and the Penguin Dictionary of [Modern] Quotations. }

  PROCEDURE ShowSaying;

    CONST
      L = CR+LF;

    BEGIN { ShowSaying }
      CASE Random(15) OF
	0 : WriteLn(
'You can only live once, but if you do it right, once is enough.');
	1 : WriteLn(
'A billion here, a couple of billion there -- first thing you know it', L,
'adds up to be real money.', L,
'':16, '-- Senator Everett McKinley Dirksen');
	2 : WriteLn(
'A classic is something that everybody wants to have read and nobody', L,
'wants to read.  -- Mark Twain');
	3 : WriteLn(
'Any clod can have the facts, but having opinions is an art.', L,
'':16, '-- Charles McCabe');
	4 : WriteLn(
'Anyone who who uses the phrase "easy as taking candy from a baby" has never',
L, 'tried taking candy from a baby.  -- Robin Hood');
	5 : WriteLn(
'Art is anything you can get away with.  -- Marshall McLuhan');
	6 : WriteLn(
'As far as the laws of mathematics refer to reality, they are not certain,',
L, 'and as far as they are certain, they do not refer to reality.',
L, '':16, '-- Albert Einstein');
	7 : WriteLn(
'At Group L, Stoffel oversees six first-rate programmers, a managerial',
L, 'challenge roughly comparable to herding cats.', L,
'':16, '-- The Washington Post Magazine, June 9, 1985');
	8 : WriteLn(
'A science is any discipline in which the fool of this generation can go', L,
'beyond the point reached by the genius of the last generation.', L,
'':16, '-- Max Gluckman [Politics, Law and Ritual]');
	9 : WriteLn(
'Bad as our urban conditions often are, there is not a slum in the country',L,
'which has a third of the infantile death-rate of the royal family in the',L,
'middle ages.    -- J.B.S.Haldane [Daedalus or Science and the Future]');
	10 : WriteLn(
'This is not the age of pamphleteers. It is the age of engineers. The spark',L,
'gap is mightier than the pen. Democracy will not be salvaged by men who talk',
L, 'fluently, debate forcefully and quote aptly.', L,
'':16, '-- Lancelot Hogben [Science for the Citizen]');
	11 : WriteLn(
'A first rate theory predicts; a second rate theory forbids; and a third-rate',
L, 'theory explains after the event.  -- A. I. Kitaigorodskii');
	12 : WriteLn(
'Darling, only one more instalment and baby will be *ours*.  -- Punch magazine');
	13 : WriteLn(
'A true gentleman is a man who knows how to play the bagpipes - but doesn''t.',
L, '-- Wall Street Journal');
	14 : WriteLn(
'In Italy for thirty years under the Borgias they had warfare, terror, murder,',
L,
'bloodshed - they produced Michelangelo, Leonardo da Vinci and the Renaissance.',
L,
'In Switzerland they had brotherly love, five hundred years of democracy and',
L,
'peace, and what did they produce ...? The cuckoo clock. -- Orson Welles');
      END;
    END { ShowSaying };
{
  List program documentation to standard output.
}
  PROCEDURE DocumentList;

    VAR
      CON : FILE;

    BEGIN { DocumentList }
      FN := pSTRING(@OutName);
      AssignCheck(CON, '');
      ReWriteCheck(CON, 1);
      BlockWriteCheck(CON, Mem[SEG(DocBegin):OFS(DocBegin)],
	OFS(DocEnd)-OFS(DocBegin));
      CloseCheck(CON);
      FN := NIL;
      Halt;
    END { DocumentList };
{
  Display program usage information with whatever the defaults are.
}
  PROCEDURE Usage;

    VAR
      c : CHAR;
      f : TEXT;

    BEGIN { Usage }
      FN := pSTRING(@OutName);
      ClrScr;
      WriteLn('Modify application to restrict and log usage on a Novell Netware (tm) network.');
      WriteLn('Usage:   ', MyName, ' <ExeFile|LogFile|LockFile> [OutFile] [Switches]');
      WriteLn('Example: ', MyName, ' TEST/U5/X  (Limit TEST.EXE to 5 users and make execute-only)');
      WriteLn('Switch  Arg.   Default (Explanation)');
      WriteLn(SwitchChar, '?             (This help message)');
      WriteLn(SwitchChar, 'D             (List built in program documentation)');
      WriteLn(SwitchChar, 'X             (Make application execute-only)');
      WriteLn(SwitchChar, 'U [number]    ', Udef, ' (Maximum number of simultaneous users. 0=no limit)');
      WriteLn(SwitchChar, 'L [logfile]   "', Ldef, '" (Where to log software accesses)');
      WriteLn(SwitchChar, 'N [address]   "', Ndef, '" (User to broadcast start message to)');
      WriteLn(SwitchChar, 'P [password]  "', Pdef, '" (Password protect)');
      WriteLn(SwitchChar, 'F  name|hex   ExeOut server (File server to lock to)');
      WriteLn(SwitchChar, 'S  name       Random (Name of user limit semaphore)');
      Write  (SwitchChar, 'W [seconds]   ');
      IF WDef = $FFFF THEN
	Write('For ever')
      ELSE
      IF WDef = $0000 THEN
	Write('Don''t wait')
      ELSE
	Write(Wdef/TicksPerSec:3:1);
      WriteLn(' (Waiting time after messages. Blank=wait for key)');
      WriteLn(SwitchChar, 'MF message    "', MFdef1, '?', MFdef2, '"');
      WriteLn(SwitchChar, 'ML message    "', MLdef, '"');
      WriteLn(SwitchChar, 'MA message    "', MAdef, '"');
      WriteLn(SwitchChar, 'MU message    "', MUdef1, '?', MUdef2, '"');
      WriteLn(SwitchChar, 'MP message    "', MPdef, '"');
      WriteLn(SwitchChar, 'MQ message    "', MQdef, '"');
      WriteLn(SwitchChar, 'MS message    "', MSdef, '" (Successful software access message)');
      WriteLn(SwitchChar, 'MN message    "', '?', MNdef, '" (Broadcast message)');
      WriteLn('@responsefile  none (For long command lines, can be nested)');
      AssignCRT(f);
      ReWrite(f);
      Write(f, '--More--');
      Close(f);
      FN := pSTRING(@InpName);
      IF ReadKey = CHR(0) THEN
	c := ReadKey;
      FN := pSTRING(@OutName);
      WriteLn;
      ClrScr;
      WriteLn('        Version 1.4, Copyright (c) 1st January 1996 Julian Byrne');
      WriteLn;
      WriteLn('This program is freeware. The FSF/GNU copyleft license applies.');
      WriteLn;
      WriteLn('If you use this package regularly and you feel so inclined please send');
      WriteLn('a donation to your favourite third-world charity and/or send me a');
      WriteLn('postcard or an email.');
      WriteLn;
      WriteLn('mailto:Julian.Byrne@eng.monash.edu.au');
      WriteLn('http://kryten.eng.monash.edu.au/~jcb');
      WriteLn;
      WriteLn('Airmail: Julian Byrne, Howitt Hall, Monash Uni., Clayton, 3168, Australia');
      WriteLn;
      WriteLn('           Thanks. Have a nice day! ;-)');
      WriteLn;
      ShowSaying;
      FN := NIL;
      Halt;
    END { Usage };
{
  Set things up to read the command line and any response files referred to.
}
  PROCEDURE InitGetChar;

    BEGIN { InitGetChar }
      GetCharCount := 0;
      GetCharPoint := 1;
      SwitchChar   := GetSwitchChar;
    END { InitGetChar };
{
  Get one character from the command line or a response file.
  Returns EOF if at end of command line or response file.
}
  FUNCTION GetChar1 : CHAR;

    VAR
      c : BYTE;

    BEGIN { GetChar1 }
      IF GetCharCount = 0 THEN
	IF GetCharPoint > Mem[PrefixSEG:128] THEN
	  GetChar1 := EOFMark
	ELSE
	  BEGIN
	    GetChar1 := CHR(Mem[PrefixSEG:128+GetCharPoint]);
	    INC(GetCharPoint);
	  END
      ELSE
	IF EOFCheck(GetCharTable[GetCharCount]) THEN
	  GetChar1 := EOFMark
	ELSE
	  BEGIN
	    BlockReadCheck(GetCharTable[GetCharCount], c, SIZEOF(c));
	    GetChar1 := CHR(c);
	  END;
    END { GetChar1 };
{
  Get a non-space character.
}
  FUNCTION SkipSpaces1 : CHAR;

    VAR
      c : CHAR;

    BEGIN { SkipSpaces1 }
      REPEAT
	c := GetChar1;
      UNTIL (c = EOFMark) OR (c > ' ');
      SkipSpaces1 := c;
    END { SkipSpaces1 };
{
  Opposite of GetChar1. Used if we go too far while parsing the command line.
}
  PROCEDURE UnGetChar1;

    BEGIN { UnGetChar1 }
      IF GetCharCount = 0 THEN
	  DEC(GetCharPoint)
      ELSE
	SeekCheck(GetCharTable[GetCharCount],
	  FilePosCheck(GetCharTable[GetCharCount])-1);
    END { UnGetChar1 };
{
  Get a string from the command line or a response file. A string may be
  null (terminated by end of response file or command line),
  a word delimited by spaces or a normal " delimited string with "" to mark
  embedded "'s. In a response file a string may cross line boundaries but
  not end-of-file boundaries.
}
  FUNCTION GetString1 : STRING;

    VAR
      c : CHAR;
      s : STRING;

    BEGIN { GetString1 }
      s := '';
      c := SkipSpaces1;
      IF (c = ':') OR (c = '=') THEN
	c := SkipSpaces1;
      IF c = EOFMark THEN
	{ Nothing }
      ELSE
      IF c = '"' THEN { "...""...""..." type string }
	BEGIN
	  REPEAT
	    c := GetChar1;
	    IF (c = EOFMark) THEN
	      Fatal('Unterminated string: <'+s+'>')
	    ELSE
	    IF c = '"' THEN
	      BEGIN
		c := GetChar1;
		IF c <> '"' THEN { Terminating " }
		  BEGIN
		    IF c <> EOFMark THEN
		      UnGetChar1;
		    c := EOFMark;
		  END;
	      END;
	    IF c <> EOFMark THEN
	      BEGIN
		IF Length(s) = 255 THEN
		  Fatal('String too long <...'+Copy(s, Length(s)-16, 255)+'>');
		s := s + c;
	      END;
	  UNTIL (c = EOFMark);
	END
      ELSE { Space terminated string }
	BEGIN
	  WHILE (c > ' ') AND (c <> EOFMark) AND (c <> SwitchChar) AND
	    (c <> IndirectChar) AND (c <> CommentChar) DO
	    BEGIN
	      s := s + c;
	      c := GetChar1;
	    END;
	  IF (c <> EOFMark) THEN
	    UnGetChar1;
	END;
      GetString1 := s;
    END { GetString1 };
{
  Get a character, opening and closing nested response files as indicated.
  Will return end-of-file only if at the end of the command line.
}
  FUNCTION GetChar : CHAR;

    VAR
      c    : CHAR;
      s    : PathStr;
      Dir  : DirStr;
      Name : NameStr;
      Ext  : ExtStr;

    BEGIN { GetChar }
      WHILE (GetCharCount > 0) AND EOFCheck(GetCharTable[GetCharCount]) DO
	BEGIN
	  CloseCheck(GetCharTable[GetCharCount]);
	  DEC(GetCharCount);
	END;
      c := GetChar1;
      IF (c = IndirectChar) THEN
	IF (GetCharCount < GetCharLimit) THEN
	  BEGIN
	    s := Normalize(GetString1);
	    FSplit(s, Dir, Name, Ext);
	    IF Ext = '' THEN
	      Ext := ExtRes;
	    INC(GetCharCount);
	    AssignCheck(GetCharTable[GetCharCount], Dir+Name+Ext);
	    ResetCheckShare(GetCharTable[GetCharCount], 1);
	    c := GetChar; { Recursive call }
	  END
	ELSE
	  Fatal('Response file nest limit of '+StrBYTE(GetCharLimit,0)+
	    ' exceeded');
      GetChar := c;
    END { GetChar };
{
  Get a non-space character, opening/closing nested response files as
  indicated.
}
  FUNCTION SkipSpaces : CHAR;

    VAR
      c : CHAR;

    BEGIN { SkipSpaces }
      REPEAT
	c := GetChar;
      UNTIL (c = EOFMark) OR (c > ' ');
      SkipSpaces := c;
    END { SkipSpaces };
{
  Parse command line into switch values (stored in variables prefixed by 's')
  and application names (stored in 'ExeInp' and 'ExeOut'). Any switch with
  an "EOFMark" initial value has a non-constant default which will be filled
  in later if necessary.
}
  PROCEDURE CommandParse;

    VAR
      c, c2 : CHAR;
      Code  : INTEGER;
      vReal : REAL;
      s     : STRING;

    BEGIN { CommandParse }
      Switches := FALSE;
      ExeCount := 0;
      sMA      := MAdef;
      sMF      := EOFMark;
      sML      := MLdef;
      sMN      := EOFMark;
      sMP      := MPdef;
      sMQ      := MQdef;
      sMS      := MSdef;
      sMU      := EOFMark;
      sFS      := FSdef;
      sFV      := FVdef;
      sL       := EOFMark;
      sN       := Ndef;
      sP       := Pdef;
      sS       := Sdef;
      sU       := Udef;
      sW       := Wdef;
      sX       := Xdef;
      REPEAT
	c := SkipSpaces;
	IF c = EOFMark THEN
	  { Do nothing }
	ELSE
	IF c = CommentChar THEN { Skip line comment }
	  BEGIN
	    WHILE (c <> EOFMark) AND (c <> LF) DO
	      c := GetChar1;
	    c := LF;
	  END
	ELSE
	IF c = SwitchChar THEN
	  BEGIN
	    Switches := TRUE;
	    c	     := GetChar;
	    CASE UpCase(c) OF
	      '?' : Usage;
	      'D' : BEGIN
		      ClrScr;
		      DocumentList;
		    END;
	      'F' : BEGIN
		      s := Trim(UpCaseString(GetString1));
		      IF s = '' THEN
			Fatal('"'+SwitchChar+'F " missing value');
		      IF NOT NovellAPI THEN
			Fatal('Sorry, Novell Netware (tm) networking software required');
		      IF (s[1] >= '0') AND (s[1] <= '9') THEN
			BEGIN
			  ValHex(s, sFV, code);
			  IF code <> 0 THEN
			    Fatal('Illegal hex value "'+s+'"');
			  sFV := Swap4(sFV);
			  sFS := GetFileServerName(sFV);
			  IF sFS = '' THEN
			    Fatal('Unknown file server '+Hex8(Swap4(sFV)));
			END
		      ELSE
			BEGIN
			  sFS := s;
			  sFV := GetFileServerNumber(sFS);
			  IF sFV = 0 THEN
			    Fatal('Unknown file server "'+sFS+'"');
			END;
		    END;
	      'L' : sL := Normalize(GetString1);
	      'M' : BEGIN
		      c2 := GetChar;
		      CASE UpCase(c2) OF
			'A' : sMA := GetString1;
			'F' : sMF := GetString1;
			'L' : sML := GetString1;
			'N' : BEGIN
				sMN := GetString1;
				IF Length(sMN) >= 40 THEN
				  Fatal('"'+sMn+'" too long. '+
				  ' Must be less than 40 characters');
			      END;
			'P' : sMP := GetString1;
			'Q' : sMQ := GetString1;
			'S' : sMS := GetString1;
			'U' : sMU := GetString1;
		      ELSE
			BEGIN
			  IF c2 = EOFMark THEN
			    Fatal('Missing message type "'+
			      SwitchChar+c+' "')
			  ELSE
			    Fatal('Unknown message type "'+
			      SwitchChar+c+c2+'"');
			END;
		      END;
		    END;
	      'N' : BEGIN
		      sN := UpCaseString(Trim((GetString1)));
		      IF sN = '' THEN
			sN := Ndef2;
		    END;
	      'P' : BEGIN
		      sP := UpCaseString(GetString1);
		      IF sP = '' THEN
			sP := GenPassword;
		    END;
	      'S' : BEGIN
		      sS := UpCaseString(Trim(GetString1));
		      IF sS = '' THEN
			Fatal('"'+SwitchChar+'S " missing value');
		    END;
	      'U' : BEGIN
		      s := Trim(GetString1);
		      IF s = '' THEN
			sU := 1
		      ELSE
			BEGIN
			  Val(s, sU, Code);
			  IF (sU > 255) OR (Code <> 0) THEN
			    Fatal('"'+SwitchChar+
			      'U " bad value (Must be 0 to 255)');
			END;
		    END;
	      'W' : BEGIN
		      s := Trim(GetString1);
		      IF s = '' THEN
			sW := $FFFF
		      ELSE
			BEGIN
			  Val(s, vReal, Code);
			  IF (vReal < 0.0) OR (vReal > 1800.0) OR (Code <> 0) THEN
			    Fatal('"'+SwitchChar+
			      'W " bad value (Must be 0.0 to 1800.0)');
			  sW := ROUND(vReal*TicksPerSec);
			END;
		    END;
	      'X' : sX := NOT sX;
	    ELSE
	      BEGIN
		IF c = EOFMark THEN
		  Fatal('Missing switch "'+SwitchChar+' "')
		ELSE
		  Fatal('Unknown switch "'+SwitchChar+c+'"');
	      END;
	    END;
	  END
	ELSE
	  BEGIN
	    UnGetChar1;
	    INC(ExeCount);
	    CASE ExeCount OF
	      1 : ExeInp := Normalize(GetString1);
	      2 : ExeOut := Normalize(GetString1);
	    ELSE
	      BEGIN
		Fatal('More than 2 file names');
	      END;
	    END;
	  END;
      UNTIL c = EOFMark;
      IF ExeCount = 0 THEN
	IF Switches THEN
	  Fatal('Missing file name')
	ELSE
	  BEGIN
	    FN := pSTRING(@OutName);
	    WriteLn('Display usage:      ', MyName, ' /?');
	    WriteLn('Save documentation: ', MyName, ' /d > ', MyName, '.DOC');
	    FN := NIL;
	    Halt;
	  END;
    END { CommandParse };
{
  Install default values for switches which were not set by the command line.
  Some default values depend on other switch values and so can't be set
  until after the command line is parsed.
}
  PROCEDURE CommandDefaults;

    VAR
      tVO, tVL: FSNType;
      tSO, tSL: STRING47;
      S       : STRING8;
      PathO,
      PathL   : PathStr;
      DirL,
      DirI,
      DirO    : DirStr;
      NameL,
      NameI,
      NameO   : NameStr;
      ExtL,
      ExtI,
      ExtO    : ExtStr;

    BEGIN { CommandDefaults }
      ExeInp := Normalize(ExeInp);
      FSplit(ExeInp, DirI, NameI, ExtI);
      IF ExtI = '' THEN
	ExtI := ExtExe;
      ExeInp := DirI+NameI+ExtI;

      ExeBak := DirI+NameI; { Extension added when we know whether lock/unlock }

      IF ExeCount > 1 THEN
	BEGIN
	  ExeOut := Normalize(ExeOut);
	  FSplit(ExeOut, DirO, NameO, ExtO);
	  IF ExtO = '' THEN
	    ExtO := ExtExe;
	  ExeOut := DirO+NameO+ExtO;
	  IF ExeInp = ExeOut THEN
	    ExeCount := 1;
	END;
      IF ExeCount = 1 THEN
	BEGIN
	  DirO	 := DirI;
	  NameO  := NameI;
	  ExtO	 := ExtTmp;
	  ExeOut := DirO+NameO+ExtO;
	  IF ExeInp = ExeOut THEN
	    Fatal('Temporary and output files are identical: "'+ExeOut+'"');
	END;
      NetFExpand(ExeOut, PathO, tSO, tVO);
      OutOnServer := (tVO <> 0);

      IF ExeCount = 1 THEN
	ExeRes := ExeInp
      ELSE
	ExeRes := ExeOut;

      IF sL = EOFMark THEN
	BEGIN
	  DirL	:= '';
	  NameL := '';
	  ExtL	:= '';
	END
      ELSE
      IF sL <> '' THEN
	BEGIN
	  sL := Normalize(sL);
	  FSplit(sL, DirL, NameL, ExtL);
	  IF ExtL = '' THEN
	    ExtL := ExtLog;
	END
      ELSE
	BEGIN
	  DirL	:= DirO;
	  NameL := NameO;
	  ExtL	:= ExtLog;
	END;
      sL := DirL+NameL+ExtL;
      IF sL <> '' THEN
	NetFExpand(sL, PathL, tSL, tVL)
      ELSE
	BEGIN
	  PathL := '';
	  tVL	:= 0;
	  tSL	:= '';
	END;
      LogOnServer := (tVL <> 0);

      IF (sU <> 0) AND (sS = '') THEN { Require semaphore name }
	sS := NameO+'-'+GenPassword;

      IF (sS <> '') OR (sN <> '') OR (sL <> '') THEN { Require file server }
	IF sFV = 0 THEN
	  BEGIN
	    sFV := tVO;
	    sFS := tSO;
	    IF sFV = 0 THEN
	      BEGIN
		sFV := tVL;
		sFS := tSL;
		IF sFV = 0 THEN
		  Fatal(SwitchChar+'F switch needed');
	      END;
	  END;
{
  Perform various consistency checks
}
      IF sX AND (NOT OutOnServer) THEN
	Fatal('"'+ExeRes+'" can''t be made execute-only off a file server');
      IF sX AND (sFV <> 0) AND (tVO <> 0) AND (sFV <> tVO) THEN
	Fatal('"'+ExeRes+'" not on file server "'+sFS+'" and '+SwitchChar+
	  'X given');
      IF (sL <> '') AND (tVL = 0) THEN
	Warn('"'+sL+'" not on a file server');
      IF (sFV <> 0) AND (tVL <> 0) AND (sFV <> tVL) THEN
	Fatal('"'+sL+'" not on file server "'+sFS+'"');
{
  Do remaining switch defaults
}
      sL := PathL;

      IF sMF = EOFMark THEN
	IF sFV <> 0 THEN
	  sMF := MFdef1 + sFS + MFdef2
	ELSE
	  sMF := '';
      IF sMN = EOFMark THEN
	IF sN <> '' THEN
	  sMN := NameO+MNdef
	ELSE
	  sMN := '';
      IF sMU = EOFMark THEN
	IF sU <> 0 THEN
	  BEGIN
	    Str(sU, S);
	    sMU := MUdef1 + S + MUdef2;
	  END
	ELSE
	  sMU := '';
    END { CommandDefaults };


  PROCEDURE CopyFile(Count : LONGINT; VAR fInp, fOut : FILE);

    CONST
      BufMax  = 32768;

    VAR
      BufSize : WORD;
      Buffer  : ARRAY[1..BufMax] OF BYTE;

    BEGIN { CopyFile }
      WHILE Count <> 0 DO
	BEGIN
	  IF Count > BufMax THEN
	    BufSize := BufMax
	  ELSE
	    BufSize := Count;
	  BlockReadCheck(fInp, Buffer, BufSize);
	  BlockWriteCheck(fOut, Buffer, BufSize);
	  Count := Count - BufSize;
	END;
    END { CopyFile };
{
  Create a locked application from the modified header, the input application,
  filler bytes, the lock code and the lock data.
}
  PROCEDURE CopyFileLock(VAR Header   : EXEHeader;
			 VAR LockData : LockDataType;
			     ParFill  : WORD;
			     CodeSize : WORD;
			 VAR fInp,
			     fOut     : FILE);

    VAR
      OverlaySize,
      MainSize : LONGINT;
      Zeroes   : ARRAY[0..15] OF CHAR;

    BEGIN { CopyFileLock }
      MainSize	  := HeaderFileSize(Header) -
		     LONGINT(ParFill + CodeSize + LockData.DataSize);
      OverlaySize := FileSizeCheck(fInp)-MainSize;
      BlockWriteCheck(fOut, Header, SIZEOF(Header));
      MainSize	  := MainSize - SIZEOF(Header);
      CopyFile(MainSize, fInp, fOut);
      IF ParFill <> 0 THEN
	BEGIN
	  FillChar(Zeroes, ParFill, CHR(0));
	  BlockWriteCheck(fOut, Zeroes, ParFill);
	END;
      BlockWriteCheck(fOut, Mem[SEG(LockBegin):OFS(LockBegin)], CodeSize);
      BlockWriteCheck(fOut, LockData, LockData.DataSize);
      CopyFile(OverlaySize, fInp, fOut);
    END { CopyFileLock };
{
  Created an unlocked application from the modified header and the
  locked application.
}
  PROCEDURE CopyFileUnlock(VAR Header	    : EXEHeader;
			   VAR LockData     : LockDataType;
			       ParFill	    : WORD;
			       CodeSizeFile : WORD;
			       DataSizeFile : WORD;
			   VAR fInp,
			       fOut	    : FILE);

    VAR
      OverlaySize,
      MainSize	 : LONGINT;

    BEGIN { CopyFileUnlock }
      MainSize	  := HeaderFileSize(Header);
      OverlaySize := FileSizeCheck(fInp) - (MainSize +
		     LONGINT(ParFill + CodeSizeFile + DataSizeFile));
      BlockWriteCheck(fOut, Header, SIZEOF(Header));
      MainSize := MainSize - SIZEOF(Header);
      CopyFile(MainSize, fInp, fOut);
      SeekCheck(fInp, FileSizeCheck(fInp)-OverlaySize);
      CopyFile(OverlaySize, fInp, fOut);
    END { CopyFileUnlock };
{
  Create the lock data record from the program switch values and the
  application to be locked's header.
}
  PROCEDURE InitLockData(VAR Header   : EXEHeader;
			 VAR LockData : LockDataType;
			     CodeSize : WORD);

    VAR
      I,
      Base,
      Offset,
      Entry : WORD;
      SizeL : LONGINT;

    PROCEDURE FillMessage(Skip : BOOLEAN; Message : STRING);

      BEGIN { FillMessage }
	WITH LockData DO
	  BEGIN
	    INC(Entry);
	    IF Skip THEN
	      pMess[Entry] := 0
	    ELSE
	    BEGIN
	      pMess[Entry]  := Base + Offset; { Semaphore name }
	      Move(Message[1], Messages[Offset], Length(Message));
	      Offset := Offset + Length(Message);
	    END;
	  END;
      END { FillMessage };


    BEGIN { InitLockData }
      WITH Header, LockData DO
	BEGIN
	  Filler    := 0;		      { Save original header }
	  JmpByte   := JMPDWORD;
	  Chain     := PTR(StartCS, StartIP);
	  UStack    := PTR(StartSS, StartSP);
	  SizeMin   := MinSize;
	  SizeMax   := MaxSize;
	  SizeLP    := LastPageSize;
	  SizePF    := PageFileSize;
	  SizeL     := HeaderFileSize(Header)-ParToByte(HeadSize);
	  SizeLSW   := SizeL AND $FFFF;
	  SizeMSW   := SizeL SHR 16;

	  Limit     := sU;		      { Insert lock data }
	  IF LogOnServer THEN
	    BEGIN
	      LogMode := ShareDenyWrite OR AccessWrite;
	      LogAttr := Shareable;
	    END
	  ELSE
	    BEGIN
	      LogMode := FileMode;
	      LogAttr := FileAttr;
	    END;
	  SemServer := sFV;
	  WaitTime  := sW;
	  Base	    := CodeSize + OFS(Messages) - OFS(LockData);
	  Offset    := 0;
	  Entry     := 0;
	  FillMessage( sS  = '', CHR(Length(sS))+sS+CHR(0));
	  FillMessage( sL  = '', sL+CHR(0));
	  FillMessage( sP  = '', CHR(Length(sP))+EncodeSTRING(sP));
	  FillMessage((sP  = '') OR (sMQ = ''), CHR(Length(sMQ))+sMQ);
	  FillMessage( sN  = '', CHR(Length(sN))+sN);
	  FillMessage((sN  = '') OR (sMN = ''), CHR(Length(sMN))+sMN);
	  FillMessage((sFS = '') OR (sMF = ''), CHR(Length(sMF))+sMF);
	  FillMessage((sS  = '') OR (sMA = ''), CHR(Length(sMA))+sMA);
	  FillMessage((sL  = '') OR (sML = ''), CHR(Length(sML))+sML);
	  FillMessage((sU  = 0 ) OR (sMU = ''), CHR(Length(sMU))+sMU);
	  FillMessage((sP  = '') OR (sMP = ''), CHR(Length(sMP))+sMP);
	  FillMessage(sMS  = ''               , CHR(Length(sMS))+sMS);
{
  Round lock code+data size to next paragraph boundary
}
	  I	   := GetParFill(Base + Offset);
	  DataSize := OFS(Messages) - OFS(LockData) + Offset + I;
	  IF I <> 0 THEN
	    FillChar(Messages[Offset], I, CHR(0));
	END;
    END { InitLockData };
{
  Modify application to be locked's header based on the lock data.
}
  PROCEDURE FixHeaderLock(VAR LockData : LockDataType;
			  VAR Header   : EXEHeader;
			  VAR ParFill  : WORD;
			      CodeSize : WORD);

    VAR
      I,
      AddSize : WORD;

    BEGIN { FixHeaderLock }
      WITH Header, LockData DO
	BEGIN
	  ParFill := GetParFill(LastPageSize);
	  INC(LastPageSize, ParFill);
	  StartCS := ByteToPar(HeaderFileSize(Header)) - HeadSize;
	  StartIP := SIZEOF(WORD) + SignatureSize;
	  AddSize := CodeSize + DataSize;
	  INC(LastPageSize, AddSize);
	  StartSS := StartCS;
	  StartSP := AddSize + StackSize;
	  WHILE (LastPageSize > PageSize) DO
	    BEGIN
	      DEC(LastPageSize, PageSize);
	      INC(PageFileSize);
	    END;
	  IF MinSize <> 65535 THEN
	    BEGIN
	      I := ByteToPar(ParFill + AddSize);
	      IF MinSize > I THEN
		MinSize := MinSize - I
	      ELSE
		MinSize := 0;
	      I := ByteToPar(StackSize);
	      IF MinSize < I THEN
		MinSize := I;
	    END;
	  IF MaxSize <> 65535 THEN
	    BEGIN
	      I := ByteToPar(ParFill + AddSize);
	      IF MaxSize > I THEN
		MaxSize := MaxSize - I
	      ELSE
		MaxSize := 0;
	      IF MaxSize < MinSize THEN
		MaxSize := MinSize;
	    END;
	END;
    END { FixHeaderLock };
{
  Recover a locked application's original header from the lock data.
}
  PROCEDURE FixHeaderUnLock(VAR LockData : LockDataType;
			    VAR Header	 : EXEHeader;
			    VAR ParFill  : WORD);

    BEGIN { FixHeaderUnlock }
      WITH Header, LockData DO
	BEGIN
	  LastPageSize := SizeLP;
	  PageFileSize := SizePF;
	  MinSize      := SizeMin;
	  MaxSize      := SizeMax;
	  StartSS      := SEG(UStack^);
	  StartSP      := OFS(UStack^);
	  StartCS      := SEG(Chain^);
	  StartIP      := OFS(Chain^);
	  ParFill      := GetParFill(LastPageSize);
	END;
    END { FixHeaderUnlock };
{
  Determine if a log file exists. If not then create it.
  Need to be careful as there is a potential race condition with any other
  program that might want to access the same log file.
}
  PROCEDURE ProbeLog(Path : PathStr; Mode : BYTE; Attr : WORD);

    VAR
      f       : FILE;
      Header  : ARRAY[1..16] OF CHAR;
      FAsave  : WORD;
      FMsave  : BYTE;
      Created : BOOLEAN;

    BEGIN { ProbeLog }
      Header   := LockLog11;
      AssignCheck(f, Path);
      FMsave   := FileMode;
      FAsave   := FileAttr;
      FileMode := Mode;
      FileAttr := Attr;
      Created  := CreateNewFile(f, 1);
      FileMode := FMsave;
      FileAttr := FAsave;
      IF Created THEN
	BEGIN
	  BlockWriteCheck(f, Header, SIZEOF(Header));
	  CloseCheck(f);
	END;
    END { ProbeLog };
{
  Convert a binary log file to text form. See NETLOCK.DOC for file format.
}
  PROCEDURE ListLog(VAR fInp : FILE);

    VAR
      Version10 : BOOLEAN;
      I 	: WORD;
      Header	: PACKED ARRAY[1..16] OF CHAR;
      Buffer	: LogRecord;

    BEGIN { ListLog }
      FN	:= pSTRING(@OutName);
      SeekCheck(fInp, 0);
      BlockReadCheck(fInp, Header, SIZEOF(Header));
      IF (Header <> LockLog10) AND (Header <> LockLog11) THEN
	Fatal('"'+FileName(fInp)+'" not a lock log file v1.0 or v1.1 or v1.2');
      Version10 := (Header = LockLog10);
      WITH Buffer DO
	WHILE NOT EOFCheck(fInp) DO
	  BEGIN
	    BlockReadCheck(fInp, Buffer, SIZEOF(Buffer));
	    IF (IALen <> 12) OR (CILen <> 62) THEN
	      Fatal('"'+FileName(fInp)+
		'" has a corrupted record at file position '+
		StrLONGINT(FilePosCheck(fInp)-SIZEOF(Buffer),0));
	    Write(LogNum:2, LogLim:3);
	    IF NOT Version10 THEN
	      Write((PassOk AND 1):2);
	    WriteLn(LogCon:3, ' ', StrTime(FSDAT),
	      ' ', Hex8(Swap4(NetNum)), ':', StrNode(NodeAddr),
	      {' ', Swap(Socket), ' ', Swap4(ObjID),
	       ' ', ObjNameStr[Swap(ObjType)],}
	      ' ', StrTime(LogTime), ' ', StrName(ObjName));
	  END;
      CloseCheck(fInp);
      FN := NIL;
    END { ListLog };
{
  After a successful lock or unlock of an application, where the user did
  not specify an output file, rename the original as a backup and the
  temporary file as the same name as the original.
}
  PROCEDURE DoBackup;

    BEGIN { DoBackup }
      IF ExeBak = FileName(fInp) THEN
	Fatal('Backup and input files are identical: "'+ExeBak+'"');
      IF ExeBak = FileName(fOut) THEN
	Fatal('Backup and output files are identical: "'+ExeBak+'"');
      AssignCheck(fBak, ExeBak);
      EraseIfPresent(fBak);
      RenameCheck(fInp, ExeBak);
      RenameCheck(fOut, ExeInp);
    END { DoBackup };

{ ---------- MAIN PROGRAM ---------- }

  BEGIN { NetLock }
    FSplit(ParamStr(0), MyDir, MyName, MyExt);
    CodeSizeMem := OFS(LockEnd)-OFS(LockBegin);

    SaveFile  := ''; { Set up for exit handler }
    SaveDir   := '';
    SaveDrive := DriveUndefined;
    ExitSave  := ExitProc;
    ExitProc  := @ExitHandler; { Just in case there is a fatal error }

    FN := pSTRING(@OutName); { So output and input can be redirected }
    Assign(OUTPUT, '');
    ReWrite(OUTPUT);
    FN := pSTRING(@InpName);
    Assign(INPUT, '');
    Reset(INPUT);
    FN := NIL;

    IF SIZEOF(LogRecord) <> 88 THEN { Check if NETLOCK object link was okay }
      Fatal('Sorry, internal error - records not packed');
    IF Mem[SEG(LockEnd):OFS(LockEnd)-1] <> LockEndMark THEN
      Fatal('Sorry, internal error - lock file end mark wrong');
    IF Mem[SEG(DocEnd):OFS(DocEnd)-1] <> ORD(LF) THEN
      Fatal('Sorry, internal error - doc file end not a linefeed');
    IF Lo(DOSVersion) < 2 THEN
      Fatal('Sorry, this program requires MSDOS 2.0 or above');

    Randomize; { For password and semaphore name generators }
    RandSeed := RandSeed + PrefixSeg;
      { So generated password can't be deduced from locked file's timestamp }

    InitGetChar;     { Parse arguments }
    CommandParse;
    CommandDefaults;

    IF (sFV <> 0) AND (NOT NovellAPI) THEN
      Fatal('Sorry, Novell Netware (tm) networking software required');

    AssignCheck(fInp, ExeInp); { Determine type of input file }
    ResetCheckShare(fInp, 1);
    Header.Magic := $0000; { Bad - will cause failure }
    IF FileSizeCheck(fInp) >= SIZEOF(Header.Magic) THEN
      BlockReadCheck(fInp, Header.Magic, SIZEOF(Header.Magic)); { 2 bytes }
    IF (Header.Magic = $5A4D) OR (Header.Magic = $4D5A) THEN { Application }
      BEGIN
	BlockReadCheck(fInp, Header.LastPageSize,
	  SIZEOF(Header)-SIZEOF(Header.Magic)); { Read rest of header }
	AssignCheck(fOut, ExeOut);
	IF LockedFile(fInp, Header) THEN
	  IF Switches THEN
	    Fatal('Invalid switches with unlock of "'+ExeInp+'"')
	  ELSE
	    BEGIN
	      ReadLockData(fInp, Header, LockData, CodeSizeFile, DataSizeFile);
	      IF ((LockData.SemServer = 0) AND (GetSupervisor <> 0)) OR
		IsSupervisor(LockData.SemServer) THEN
		BEGIN
		  FixHeaderUnLock(LockData, Header, ParFill);
		  SaveFile := ExeOut;
		  ReWriteCheck(fOut, 1);
		  CopyFileUnlock(Header, LockData, ParFill, CodeSizeFile,
		    DataSizeFile, fInp, fOut);
		  CloseCheck(fOut);
		  CloseCheck(fInp);
		  ExeBak := Normalize(ExeBak + ExtLok);
		  IF ExeCount = 1 THEN
		    DoBackup;
		  SaveFile := ''; { Exit handler should not delete result }
		  DumpLockData(LockData);
		END
	      ELSE
		Fatal('Sorry, you''re not '+Supervisor+
		  ' equivalent. Can''t unlock locked "'+ExeInp+'"');
	    END
	ELSE { File not locked, lock it }
	  BEGIN
	    IF NOT OutOnServer THEN
	      Warn('"'+ExeRes+'" not on a file server');
	    IF FileSizeCheck(fInp) <> HeaderFileSize(Header) THEN
	      Warn('"'+ExeInp+
		'" is overlayed - locked version may malfunction');
	    InitLockData(Header, LockData, CodeSizeMem);
	    FixHeaderLock(LockData, Header, ParFill, CodeSizeMem);
	    IF sL <> '' THEN { Make sure log file is accessible }
	      IF LogOnServer THEN
		BEGIN
		  LogID       := GetConnectionID(sFS);
		  IF LogID = 0 THEN
		    Fatal('Can''t create log file. No connection to "'+sFS+'"');
		  PreferredID := GetPreferredConnectionID;
		  SetPreferredConnectionID(LogID);
		  ProbeLog(sL, ShareDenyWrite+AccessWrite, Shareable);
		  SetPreferredConnectionID(PreferredID);
		END
	      ELSE
		ProbeLog(sL, FileMode, FileAttr);
	    SaveFile := ExeOut;
	    IF OutOnServer THEN
	      ReWriteCheckShare(fOut, 1)
	    ELSE
	      ReWriteCheck(fOut, 1);
	    CopyFileLock(Header, LockData, ParFill, CodeSizeMem, fInp, fOut);
	    CloseCheck(fOut);
	    CloseCheck(fInp);
	    ExeBak := Normalize(ExeBak + ExtUnl);
	    IF ExeCount = 1 THEN
	      DoBackup;
	    SaveFile := ''; { Exit handler should not delete result }
	    IF sX THEN
	      ExeOnly(ExeRes);
	    Report;
	  END;
      END
    ELSE
    IF Header.Magic = $6F4C THEN { = 'Lo'. Lock log. }
      IF Switches OR (ExeCount > 1) THEN
	Fatal('Invalid switches with dump lock log of "'+ExeInp+'"')
      ELSE
	ListLog(fInp)
    ELSE
      Fatal('"'+ExeInp+'" is not an application EXE or lock log file');
    FN := NIL;
    Halt;
  END { NetLock }.
