UNIT Country;
{$I Options}
  
{
  Copyright (C) 1991 Julian Byrne. All rights reserved.

  Title:        Country dependent information
  File name:    COUNTRY.PAS
  Version:      1.00
  Usage:        USES Country;
  Description:  Provides access to MSDOS country dependent information.
  Dependencies: See USES statement
  Author:       Julian Byrne
  Address:      Electrical and Computer Systems Engineering Department
                Monash University, Wellington Road, Clayton, Victoria, 3168
                Australia
  Internet:     julian.byrne@monash.edu.au.
  Other nets:   Quaterman & Hoskins "Notable Computer Networks"
                CACM Oct'86 pp932-971
  History:      90/ 5/ 1 Initial version
  Notes: A unit for setting and listing the country dependent  information
    available in MSDOS 2.01 or higher.  It is written in Turbo Pascal v5.0.
    Contains routines for displaying currency, displaying time, and changing
    case given this information. 
}

INTERFACE { Country }

  USES
    CRT,
    DOS;

  TYPE
    DateTimeWH   = RECORD
                     DT        : DateTime;
                     Hundredths,
                     DayOfWeek : WORD;
                   END;

  CONST
    DateTimeNil   : DateTime   = (Year  : 0;
                                  Month : 0;
                                  Day   : 0;
                                  Hour  : 0;
                                  Min   : 0;
                                  Sec   : 0);
    DateTimeWHnil : DateTimeWH = (DT         : (Year  : 0;
                                                Month : 0;
                                                Day   : 0;
                                                Hour  : 0;
                                                Min   : 0;
                                                Sec   : 0);
                                  Hundredths : 0;
                                  DayOfWeek  : 0);
    MonName   : ARRAY[1..12] OF STRING[9] = (
      'January' , 'February', 'March' , 'April'    , 'May'    ,
      'June'    , 'July'    , 'August', 'September', 'October',
      'November', 'December');
    DayName   : ARRAY[0..6] OF STRING[9] = (
      'Sunday   ', 'Monday   ', 'Tuesday  ', 'Wednesday', 'Thursday ',
      'Friday   ', 'Saturday ');

  TYPE
    CDI           = PACKED RECORD { Country Dependent Information  }
                      DateTimeFormat : INTEGER;

{ 0 - USA standard    h:m:s m/d/y
  1 - Europe standard h:m:s d/m/y
  2 - Japan standard  y/m/d h:m:s

  All strings are NUL terminated }

                      CurrencySymbol : ARRAY[1..5] OF CHAR;
                      ThousandsSep   : ARRAY[1..2] OF CHAR;
                      DecimalSep     : ARRAY[1..2] OF CHAR;
                      DateSep        : ARRAY[1..2] OF CHAR;
                      TimeSep        : ARRAY[1..2] OF CHAR;
                      BitField       : SET OF 0..7;

{ Bit 0 = 0 If currency symbol precedes the currency amount.
        = 1 If currency symbol comes after the currency amount.
  Bit 1 = 0 If currency symbol immediately precedes the currency amount.
        = 1 If there is a space between the currency symbol and the amount. }

                      CurrencyPlaces : BYTE;
                      TimeFormat     : BYTE;

{ 0 - 12 hour time
  1 - 24 hour time }

                      CaseMapCall    : POINTER;

{ A FAR procedure address which will perform country specific lower-to-upper
  case mapping on character values from 80H to FFH in AL. Returned in AL with
  no other registers modified except flags. }

                      DataListSep    : ARRAY[1..2] OF CHAR;
                    END;
    String2       = STRING[  2];
    String32      = STRING[ 32];

  FUNCTION  ToUpper         (VAR CDD : CDI; x : CHAR) : CHAR;
  FUNCTION  ToLower         (VAR CDD : CDI; x : CHAR) : CHAR;
  PROCEDURE GetDate         (VAR x : DateTimeWH);
  PROCEDURE GetTime         (VAR x : DateTimeWH);
  PROCEDURE GetDateTime     (VAR x : DateTimeWH);
  PROCEDURE SetCountry      (    ISDcode : BYTE);
  PROCEDURE GetCountry      (VAR ISDcode : BYTE; VAR CDD : CDI);
  FUNCTION  StrZ            (x : String32) : String32;
  FUNCTION  Str0            (x : WORD) : String2;
  FUNCTION  StrDate         (VAR CDD : CDI; VAR x : DateTimeWH) : String32;
  FUNCTION  StrTime         (VAR CDD : CDI; VAR x : DateTimeWH) : String32;
  FUNCTION  StrDateTime     (VAR CDD : CDI; VAR x : DateTimeWH) : String32;
  FUNCTION  LeapYear        (Year : INTEGER) : BOOLEAN;
  FUNCTION  MonLen          (VAR x : DateTimeWH) : BYTE;
  FUNCTION  DayInd          (VAR x : DateTimeWH) : INTEGER;
  FUNCTION  DayInd2         (VAR x : DateTimeWH) : LONGINT;
  FUNCTION  SecInd          (VAR x : DateTimeWH) : LONGINT;
  FUNCTION  SecInd2         (VAR x : DateTimeWH) : LONGINT;
  PROCEDURE WeekDay         (VAR x : DateTimeWH);
  FUNCTION  DateValid       (VAR x : DateTimeWH) : BOOLEAN;
  FUNCTION  TimeValid       (VAR x : DateTimeWH) : BOOLEAN;
  FUNCTION  DateTimeValid   (VAR x : DateTimeWH) : BOOLEAN;
  PROCEDURE ValDate         (VAR CDD : CDI; DTS : String32; VAR x : DateTimeWH);
  PROCEDURE ValTime         (VAR CDD : CDI; DTS : String32; VAR x : DateTimeWH);
  PROCEDURE ValDateTime     (VAR CDD : CDI; DTS : String32; VAR x : DateTimeWH);
  FUNCTION  ReadNumStr      (    x : String32) : String32;
  PROCEDURE WriteDateDel    (VAR CDD : CDI; VAR x : DateTimeWH);
  PROCEDURE WriteTimeDel    (VAR CDD : CDI; VAR x : DateTimeWH);
  PROCEDURE WriteDateTimeDel(VAR CDD : CDI; VAR x : DateTimeWH);
  PROCEDURE WriteDateAbs    (VAR CDD : CDI; VAR x : DateTimeWH);
  PROCEDURE WriteTimeAbs    (VAR CDD : CDI; VAR x : DateTimeWH);
  PROCEDURE WriteDateTimeAbs(VAR CDD : CDI; VAR x : DateTimeWH);
  PROCEDURE ReadDateDel     (VAR CDD : CDI; VAR x : DateTimeWH);
  PROCEDURE ReadTimeDel     (VAR CDD : CDI; VAR x : DateTimeWH);
  PROCEDURE ReadDateTimeDel (VAR CDD : CDI; VAR x : DateTimeWH);
  PROCEDURE ReadDateAbs     (VAR CDD : CDI; VAR x : DateTimeWH);
  PROCEDURE ReadTimeAbs     (VAR CDD : CDI; VAR x : DateTimeWH);
  PROCEDURE ReadDateTimeAbs (VAR CDD : CDI; VAR x : DateTimeWH);
  FUNCTION  DateCmp         (VAR a, b : DateTimeWH) : INTEGER;
  FUNCTION  TimeCmp         (VAR a, b : DateTimeWH) : INTEGER;
  FUNCTION  DateTimeCmp     (VAR a, b : DateTimeWH) : INTEGER;
  FUNCTION  StrCurrency     (VAR CDD : CDI; CS, FS  : String32) : String32;

IMPLEMENTATION { Country }

  CONST
    NUL           =  #0; 
    ETX           =  #3;
    BEL           =  #7;
    BS            =  #8;
    CR            = #13;
    LeapSince80   : PACKED ARRAY[0..255] OF BYTE = (
       0,  1,  1,  1,  1,  2,  2,  2,  2,  3,  3,  3,  3,  4,  4,  4,
       4,  5,  5,  5,  5,  6,  6,  6,  6,  7,  7,  7,  7,  8,  8,  8, 
       8,  9,  9,  9,  9, 10, 10, 10, 10, 11, 11, 11, 11, 12, 12, 12, 
      12, 13, 13, 13, 13, 14, 14, 14, 14, 15, 15, 15, 15, 16, 16, 16, 
      16, 17, 17, 17, 17, 18, 18, 18, 18, 19, 19, 19, 19, 20, 20, 20, 
      20, 21, 21, 21, 21, 22, 22, 22, 22, 23, 23, 23, 23, 24, 24, 24, 
      24, 25, 25, 25, 25, 26, 26, 26, 26, 27, 27, 27, 27, 28, 28, 28, 
      28, 29, 29, 29, 29, 30, 30, 30, 30, 30, 30, 30, 30, 31, 31, 31, 
      31, 32, 32, 32, 32, 33, 33, 33, 33, 34, 34, 34, 34, 35, 35, 35, 
      35, 36, 36, 36, 36, 37, 37, 37, 37, 38, 38, 38, 38, 39, 39, 39, 
      39, 40, 40, 40, 40, 41, 41, 41, 41, 42, 42, 42, 42, 43, 43, 43, 
      43, 44, 44, 44, 44, 45, 45, 45, 45, 46, 46, 46, 46, 47, 47, 47, 
      47, 48, 48, 48, 48, 49, 49, 49, 49, 50, 50, 50, 50, 51, 51, 51, 
      51, 52, 52, 52, 52, 53, 53, 53, 53, 54, 54, 54, 54, 54, 54, 54, 
      54, 55, 55, 55, 55, 56, 56, 56, 56, 57, 57, 57, 57, 58, 58, 58, 
      58, 59, 59, 59, 59, 60, 60, 60, 60, 61, 61, 61, 61, 62, 62, 62);
    MonLenb       : PACKED ARRAY[1..12] OF BYTE    =
                      (31,28,31,30,31,30,31,31,30,31,30,31);
    MonSta        : PACKED ARRAY[1..12] OF INTEGER =
                      (0,31,59,90,120,151,181,212,243,273,304,334);

  VAR
    CaseMapCall   : POINTER;


  FUNCTION ToUpper(VAR CDD : CDI; x : CHAR) : CHAR;

    BEGIN { ToUpper }
      IF x >= #$80 THEN
        BEGIN
          CaseMapCall := CDD.CaseMapCall;
          INLINE(
             $8A/$46/<x          { MOV     AL, [BP+x]        }
            /$FF/$1E/CaseMapCall { CALL    FAR [CaseMapCall] }
            /$88/$46/<-1);       { MOV     [BP-1], AL        }
        END
      ELSE
      IF (x >= 'a') AND (x <= 'z') THEN
        ToUpper := CHR(ORD(x)-ORD('a')+ORD('A'))
      ELSE
        ToUpper := x;
    END { ToUpper };


  VAR
    CaseMapCallLower  : POINTER;
    LowerMap          : ARRAY[#$00..#$FF] OF CHAR;

  FUNCTION ToLower(VAR CDD : CDI; x : CHAR) : CHAR;

    VAR
      c : CHAR;

    BEGIN { ToLower }
      IF CaseMapCallLower <> CDD.CaseMapCall THEN
        BEGIN
          CaseMapCallLower := CDD.CaseMapCall;
          FOR c := #$00 TO #$FF DO
            LowerMap[c] := c;
          FOR c := #$80 TO #$FF DO
            LowerMap[ToUpper(CDD, c)] := c;
          FOR c := 'A' TO 'Z' DO
            LowerMap[c] := CHR(ORD(c)-ORD('A')+ORD('a'));
        END;
      ToLower := LowerMap[x]
    END { ToLower };


  PROCEDURE GetDate(VAR x : DateTimeWH);

    BEGIN { GetDate }
      DOS.GetDate(x.DT.Year, x.DT.Month, x.DT.Day, x.DayOfWeek);
    END { GetDate };


  PROCEDURE GetTime(VAR x : DateTimeWH);

    BEGIN { GetTime }
      DOS.GetTime(x.DT.Hour, x.DT.Min, x.DT.Sec, x.Hundredths);
    END { GetTime };


  PROCEDURE GetDateTime(VAR x : DateTimeWH);

    BEGIN { GetDateTime }
      GetDate(x);
      GetTime(x);
    END { GetDateTime };


  PROCEDURE SetCountry(ISDcode : BYTE);

    VAR
      R : Registers;

    BEGIN { SetCountry }
      WITH R DO
        BEGIN
          AX    := $3800 + ISDcode;
          DX    := $FFFF;
          DS    := $FFFF;
          Flags := $0000;
          Intr($21, R);
        END;
    END { SetCountry };


  PROCEDURE GetCountry(VAR ISDcode : BYTE; VAR CDD : CDI);

    VAR
      R : Registers;

    BEGIN { GetCountry }
      WITH R DO
        BEGIN
          AX      := $3800;
          DX      := Ofs(CDD);
          DS      := Seg(CDD);
          Flags   := $0000;
          Intr($21, R);
          ISDcode := Lo(AX);
        END;
    END { GetCountry };


  FUNCTION StrZ(x : String32) : String32;

    VAR
      I : INTEGER;

    BEGIN { StrZ }
      x := x + CHR(0);
      I := 1;
      WHILE x[I] <> CHR(0) DO
        INC(I);
      StrZ := Copy(x, 1, I-1);
    END { StrZ };


  FUNCTION Str0(x : WORD) : String2;

    BEGIN { Str0 }
      Str0 := CHR(((x DIV 10) MOD 10) + ORD('0')) + CHR((x MOD 10) + ORD('0'));
    END { Str0 };


  FUNCTION StrDate(VAR CDD : CDI; VAR x : DateTimeWH) : String32;

    VAR
      DTS : String32;

    BEGIN { StrDate }
      WITH CDD DO
        BEGIN
          CASE DateTimeFormat OF
            0 : DTS := Str0(x.DT.Month) + StrZ(DateSep)
                     + Str0(x.DT.Day  ) + StrZ(DateSep)
                     + Str0(x.DT.Year );
            1 : DTS := Str0(x.DT.Day  ) + StrZ(DateSep)
                     + Str0(x.DT.Month) + StrZ(DateSep)
                     + Str0(x.DT.Year );
            2 : DTS := Str0(x.DT.Year ) + StrZ(DateSep)
                     + Str0(x.DT.Month) + StrZ(DateSep)
                     + Str0(x.DT.Day  );
          ELSE
            BEGIN
              Str(DateTimeFormat, DTS);
              DTS := 'Bad date/time format: ' + DTS;
            END;
          END;
        END;
      StrDate := DTS;
    END { StrDate };


  FUNCTION StrTime(VAR CDD : CDI; VAR x : DateTimeWH) : String32;

    VAR
      DTS : String32;

    BEGIN { StrTime }
      WITH CDD DO
        BEGIN
          DTS := StrZ(TimeSep) + Str0(x.DT.Min)
               + StrZ(TimeSep) + Str0(x.DT.Sec)
               + StrZ(DecimalSep) + Str0(x.Hundredths);
          CASE TimeFormat OF
            0 : IF x.DT.Hour < 12 THEN
                  IF x.DT.Hour = 0 THEN
                    DTS := '12' + DTS + 'a'
                  ELSE
                    DTS := Str0(x.DT.Hour) + DTS + 'a'
                ELSE
                  IF x.DT.Hour = 12 THEN
                    DTS := '12' + DTS + 'p'
                  ELSE
                    DTS := Str0(x.DT.Hour-12) + DTS + 'p';
            1 : DTS := Str0(x.DT.Hour) + DTS;
          ELSE
            BEGIN
              Str(TimeFormat, DTS);
              DTS := 'Bad time format: ' + DTS;
            END;
          END;
        END;
      StrTime := DTS;
    END { StrTime };


  FUNCTION StrDateTime(VAR CDD : CDI; VAR x : DateTimeWH) : String32;

    VAR
      D, T : String32;

    BEGIN { StrDateTime }
      D := StrDate(CDD, x);
      T := StrTime(CDD, x);
      CASE CDD.DateTimeFormat OF
        0 : StrDateTime := T + ' ' + D;
        1 : StrDateTime := T + ' ' + D;
        2 : StrDateTime := D + ' ' + T;
      ELSE
        StrDateTime := T + ' ' + D;
      END;
    END { StrDateTime };


  FUNCTION LeapYear(Year : INTEGER) : BOOLEAN;

    BEGIN { LeapYear }
      LeapYear := ((Year MOD   4)  = 0) AND
                  ((Year MOD 100) <> 0) OR
                  ((Year MOD 400)  = 0);
    END { LeapYear };


  FUNCTION MonLen(VAR x : DateTimeWH) : BYTE;
  
    BEGIN { MonLen }
      IF (x.DT.Year < 1980) OR (x.DT.Year > (1980+255)) THEN
        MonLen := MonLenb[x.DT.Month] + ORD(x.DT.Month = 2)
      ELSE
        MonLen := MonLenb[x.DT.Month] +
                  ORD((x.DT.Month = 2) AND LeapYear(x.DT.Year));
    END { MonLen };


  FUNCTION DayInd(VAR x : DateTimeWH) : INTEGER;

    BEGIN { DayInd }
      DayInd := MonSta[x.DT.Month] + 
                ORD((x.DT.Month > 2) AND LeapYear(x.DT.Year)) +
                INTEGER(x.DT.Day) - 1;
    END { DayInd };


  FUNCTION DayInd2(VAR x : DateTimeWH) : LONGINT;

    BEGIN { DayInd2 }
      DayInd2 := 365*(LONGINT(x.DT.Year)-1980)+
        LeapSince80[INTEGER(x.DT.Year)-1980]+DayInd(x);
    END { DayInd2 };


  FUNCTION SecInd(VAR x : DateTimeWH) : LONGINT;

    BEGIN { SecInd }
      SecInd := ((LONGINT(x.DT.Hour)*60)+x.DT.Min)*60+x.DT.Sec; 
    END { SecInd };


  FUNCTION SecInd2(VAR x : DateTimeWH) : LONGINT;

    BEGIN { SecInd2 }
      SecInd2 := DayInd2(x)*(12*60*60)+(SecInd(x) DIV 2);
    END { SecInd2 };


  PROCEDURE WeekDay(VAR x : DateTimeWH);

    BEGIN { WeekDay }
      x.DayOfWeek := (2+DayInd2(x)) MOD 7; { 2 -> 1/1/80 was a Tuesday }
    END { WeekDay };


  FUNCTION DateValid(VAR x : DateTimeWH) : BOOLEAN;

    BEGIN { DateValid }
      IF (x.DT.Month < 1) OR (x.DT.Month > 12) THEN
        DateValid := FALSE
      ELSE
        DateValid := (x.DT.Day >= 1) AND (x.DT.Day <= MonLen(x))
    END { DateValid };


  FUNCTION TimeValid(VAR x : DateTimeWH) : BOOLEAN;

    BEGIN { TimeValid }
      TimeValid := (x.DT.Hour    >= 0) AND (x.DT.Hour    <  24)
               AND (x.DT.Min     >= 0) AND (x.DT.Min     <  60)
               AND (x.DT.Sec     >= 0) AND (x.DT.Sec     <  60)
               AND (x.Hundredths >= 0) AND (x.Hundredths < 100);
    END { TimeValid };


  FUNCTION DateTimeValid(VAR x : DateTimeWH) : BOOLEAN;

    BEGIN { DateTimeValid }
      DateTimeValid := DateValid(x) AND TimeValid(x);
    END { DateTimeValid };


  PROCEDURE ValDate(VAR CDD : CDI; DTS : String32; VAR x : DateTimeWH);

    VAR
      LenSep : BYTE;

    FUNCTION ValZ(x : INTEGER) : BYTE;

      BEGIN { ValZ }
        IF (DTS[x  ] < '0') OR (DTS[x  ] > '9') OR
           (DTS[x+1] < '0') OR (DTS[x+1] > '9') THEN
          ValZ := 255
        ELSE
          ValZ := (ORD(DTS[x]) - ORD('0'))*10 + ORD(DTS[x+1]) - ORD('0');
      END { ValZ };

    BEGIN { ValDate }
      WITH CDD DO
        BEGIN
          LenSep := Length(StrZ(DateSep));
          CASE DateTimeFormat OF
            0 : BEGIN
                  x.DT.Year  := ValZ(5+LenSep+LenSep);
                  x.DT.Month := ValZ(1);
                  x.DT.Day   := ValZ(3+LenSep);
                END;
            1 : BEGIN
                  x.DT.Year  := ValZ(5+LenSep+LenSep);
                  x.DT.Month := ValZ(3+LenSep);
                  x.DT.Day   := ValZ(1);
                END;
            2 : BEGIN
                  x.DT.Year  := ValZ(1);
                  x.DT.Month := ValZ(3+LenSep);
                  x.DT.Day   := ValZ(5+LenSep+LenSep);
                END;
          ELSE
            BEGIN
              x.DT.Year   := 65535;
              x.DT.Month  := 65535;
              x.DT.Day    := 65535;
            END;
          END;
          x.DayOfWeek := 65535;
        END;
    END { ValDate };


  PROCEDURE ValTime(VAR CDD : CDI; DTS : String32; VAR x : DateTimeWH);

    VAR
      LenSep : BYTE;

    FUNCTION ValZ(x : INTEGER) : BYTE;

      BEGIN { ValZ }
        IF (DTS[x  ] < '0') OR (DTS[x  ] > '9') OR
           (DTS[x+1] < '0') OR (DTS[x+1] > '9') THEN
          ValZ := 255
        ELSE
          ValZ := (ORD(DTS[x]) - ORD('0'))*10 + ORD(DTS[x+1]) - ORD('0');
      END { ValZ };

    BEGIN { ValTime }
      WITH CDD DO
        BEGIN
          LenSep := Length(StrZ(TimeSep));
          CASE TimeFormat OF
            0 : BEGIN
                  x.DT.Hour    := ValZ(1);
                  x.DT.Min     := ValZ(3+LenSep);
                  x.DT.Sec     := ValZ(5+LenSep+LenSep);
                  x.Hundredths := ValZ(7+LenSep+LenSep
                                      + Length(StrZ(DecimalSep)));
                  IF (DTS[9+LenSep+LenSep+Length(StrZ(DecimalSep))] = 'a') THEN
                    IF x.DT.Hour = 12 THEN
                      x.DT.Hour := 0
                    ELSE
                  ELSE
                    IF x.DT.Hour < 12 THEN
                      INC(x.DT.Hour,12)
                    ELSE
                      ;
                END;
            1 : BEGIN
                  x.DT.Hour    := ValZ(1);
                  x.DT.Min     := ValZ(3+LenSep);
                  x.DT.Sec     := ValZ(5+LenSep+LenSep);
                  x.Hundredths := ValZ(7+LenSep+LenSep
                                      + Length(StrZ(DecimalSep)));
                END;
          ELSE
            BEGIN
              x.DT.Hour    := 65535;
              x.DT.Min     := 65535;
              x.DT.Sec     := 65535;
              x.Hundredths := 65535;
            END;
          END;
        END;
    END { ValTime };


  PROCEDURE ValDateTime(VAR CDD : CDI; DTS : String32; VAR x : DateTimeWH);

    VAR
      LenDate,
      LenTime : BYTE;

    BEGIN { ValDateTime }
      WITH CDD DO
        BEGIN
          LenDate := Length(StrZ(DateSep))*2+6;
          LenTime := Length(StrZ(TimeSep))*2+Length(StrZ(DecimalSep))+8;
          IF TimeFormat = 0 THEN
            INC(LenTime);
          CASE DateTimeFormat OF
            0,
            1 : BEGIN
                  ValDate(CDD, Copy(DTS, Length(DTS)-LenDate+1, LenDate), x);
                  ValTime(CDD, Copy(DTS, 1                    , LenTime), x);
                END;
            2 : BEGIN
                  ValDate(CDD, Copy(DTS, 1                    , LenDate), x);
                  ValTime(CDD, Copy(DTS, Length(DTS)-LenTime+1, LenTime), x);
                END;
          ELSE
            x.DT.Year    := 65535;
            x.DT.Month   := 65535;
            x.DT.Day     := 65535;
            x.DayOfWeek  := 65535;
            x.DT.Hour    := 65535;
            x.DT.Min     := 65535;
            x.DT.Sec     := 65535;
            x.Hundredths := 65535;
          END;
        END;
    END { ValDateTime };


  FUNCTION ReadNumStr(x : String32) : String32;

    VAR
      p,
      q,
      b,
      e    : INTEGER;
      c1,
      c2   : CHAR;

    FUNCTION IsDigit(c : CHAR) : BOOLEAN;

      BEGIN { IsDigit }
        IsDigit := ((c >= '0') AND (c <= '9')) OR (c = 'a') OR (c = 'p');
      END { IsDigit };

    BEGIN { ReadNumStr }
      b := 1;
      WHILE NOT IsDigit(x[b]) DO
        INC(b);
      e := Length(x);
      Write(BS);
      WHILE NOT IsDigit(x[e]) DO
        BEGIN
          Write(BS);
          DEC(e);
        END;
      p := e;
      WHILE p > b DO
        BEGIN
          Write(BS);
          DEC(p);
        END;
      REPEAT
        c1 := ReadKey;
        CASE c1 OF
          ETX : BEGIN
                  WriteLn('^C');
                  Halt;
                END;
          CR  : ;
          '0'..'9', 'a', 'p', 'A', 'P' :
                IF ((x[p] >= '0') AND (x[p] <= '9')) <>
                   ((c1   >= '0') AND (c1   <= '9')) THEN
                  Write(BEL)
                ELSE
                  BEGIN
                    IF (c1 >= 'A') AND (c1 <= 'Z') THEN
                      c1 := CHR(ORD(c1)-ORD('A')+ORD('a'));
                    x[p] := c1;
                    q    := p+1;
                    WHILE (q <= e) AND (NOT IsDigit(x[q])) DO
                      INC(q);
                    IF q > e THEN
                      Write(x[p], BS)
                    ELSE
                      WHILE p < q DO
                        BEGIN
                          Write(x[p]);
                          INC(p);
                        END;
                  END;
          NUL : BEGIN
                  c2 := ReadKey;
                  CASE c2 OF
                    'P' : BEGIN { Down arrow }
                            CASE x[p] OF
                              'a' : x[p] := 'p';
                              'p' : x[p] := 'a';
                              '0' : x[p] := '9';
                            ELSE
                              DEC(x[p]);
                            END;
                            Write(x[p], BS);
                          END;
                    'H' : BEGIN { Up arrow }
                            CASE x[p] OF
                              'a' : x[p] := 'p';
                              'p' : x[p] := 'a';
                              '9' : x[p] := '0';
                            ELSE
                              INC(x[p]);
                            END;
                            Write(x[p], BS);
                          END;
                    'K' : IF p > b THEN
                            REPEAT
                              Write(BS);
                              DEC(p);
                            UNTIL IsDigit(x[p]);
                    'M' : IF p < e THEN
                            REPEAT
                              Write(x[p]);
                              INC(p);
                            UNTIL IsDigit(x[p]);
                    'G' : WHILE p > b DO { Home }
                            BEGIN
                              Write(BS);
                              DEC(p);
                            END;
                    'O' : WHILE p < e DO { End }
                            BEGIN
                              Write(x[p]);
                              INC(p);
                            END;
                  ELSE
                    Write(BEL);
                  END;
                END;
        ELSE
          Write(BEL);
        END;
      UNTIL c1 = CR;
      Write(Copy(x, p, 255));
      ReadNumStr := x;
    END { ReadNumStr };


  PROCEDURE WriteDateDel(VAR CDD : CDI; VAR x : DateTimeWH);

    VAR
      TimeFormatAbs : BYTE;

    BEGIN { WriteDateDel }
      TimeFormatAbs  := CDD.TimeFormat;
      CDD.TimeFormat := 1;
      Write(StrDate(CDD, x));
      CDD.TimeFormat := TimeFormatAbs;
    END { WriteDateDel };


  PROCEDURE WriteTimeDel(VAR CDD : CDI; VAR x : DateTimeWH);

    VAR
      TimeFormatAbs : BYTE;

    BEGIN { WriteTimeDel }
      TimeFormatAbs  := CDD.TimeFormat;
      CDD.TimeFormat := 1;
      Write(StrTime(CDD, x));
      CDD.TimeFormat := TimeFormatAbs;
    END { WriteTimeDel };


  PROCEDURE WriteDateTimeDel(VAR CDD : CDI; VAR x : DateTimeWH);

    VAR
      TimeFormatAbs : BYTE;

    BEGIN { WriteDateTimeDel }
      TimeFormatAbs  := CDD.TimeFormat;
      CDD.TimeFormat := 1;
      Write(StrDateTime(CDD, x));
      CDD.TimeFormat := TimeFormatAbs;
    END { WriteDateTimeDel };


  PROCEDURE WriteDateAbs(VAR CDD : CDI; VAR x : DateTimeWH);

    BEGIN { WriteDateAbs }
      Write(StrDate(CDD, x));
    END { WriteDateAbs };


  PROCEDURE WriteTimeAbs(VAR CDD : CDI; VAR x : DateTimeWH);

    BEGIN { WriteTimeAbs }
      Write(StrTime(CDD, x));
    END { WriteTimeAbs };


  PROCEDURE WriteDateTimeAbs(VAR CDD : CDI; VAR x : DateTimeWH);

    BEGIN { WriteDateTimeAbs }
      Write(StrDateTime(CDD, x));
    END { WriteDateTimeAbs };


  PROCEDURE ReadDateDel(VAR CDD : CDI; VAR x : DateTimeWH);

    VAR
      TimeFormatAbs : BYTE;
      DTS           : String32;

    BEGIN { ReadDateDel }
      TimeFormatAbs  := CDD.TimeFormat;
      CDD.TimeFormat := 1;
      DTS            := StrDate(CDD, DateTimeWHnil);
      Write(DTS);
      ValDate(CDD, ReadNumStr(DTS), x);
      CDD.TimeFormat := TimeFormatAbs;
      x.DayOfWeek    := 0;
    END { ReadDateDel };


  PROCEDURE ReadTimeDel(VAR CDD : CDI; VAR x : DateTimeWH);

    VAR
      TimeFormatAbs : BYTE;
      DTS           : String32;

    BEGIN { ReadTimeDel }
      TimeFormatAbs  := CDD.TimeFormat;
      CDD.TimeFormat := 1;
      DTS            := StrTime(CDD, DateTimeWHnil);
      Write(DTS);
      ValTime(CDD, ReadNumStr(DTS), x);
      CDD.TimeFormat := TimeFormatAbs;
      x.DayOfWeek    := 0;
    END { ReadTimeDel };


  PROCEDURE ReadDateTimeDel(VAR CDD : CDI; VAR x : DateTimeWH);

    VAR
      TimeFormatAbs : BYTE;
      DTS           : String32;

    BEGIN { ReadDateTimeDel }
      TimeFormatAbs  := CDD.TimeFormat;
      CDD.TimeFormat := 1;
      DTS            := StrDateTime(CDD, DateTimeWHnil);
      Write(DTS);
      ValDateTime(CDD, ReadNumStr(DTS), x);
      CDD.TimeFormat := TimeFormatAbs;
      x.DayOfWeek    := 0;
    END { ReadDateTimeDel };


  PROCEDURE ReadDateAbs(VAR CDD : CDI; VAR x : DateTimeWH);

    VAR
      DTS : String32;
      Ok  : BOOLEAN;

    BEGIN { ReadDateAbs }
      IF NOT DateValid(x) THEN
        BEGIN
          GetDate(x);
          x.DT.Sec     := 0;
          x.Hundredths := 0;
        END;
      DTS := StrDate(CDD, x);
      Write(DTS);
      REPEAT
        DTS := ReadNumStr(DTS);
        ValDate(CDD, DTS, x);
        IF x.DT.Year >= 80 THEN
          INC(x.DT.Year, 1900)
        ELSE
          INC(x.DT.Year, 2000);
        Ok := DateValid(x);
        IF NOT Ok THEN
          Write(BEL);
      UNTIL Ok;
      WeekDay(x);
    END { ReadDateAbs };


  PROCEDURE ReadTimeAbs(VAR CDD : CDI; VAR x : DateTimeWH);

    VAR
      DTS : String32;
      Ok  : BOOLEAN;

    BEGIN { ReadTimeAbs }
      IF NOT TimeValid(x) THEN
        BEGIN
          GetTime(x);
          x.DT.Sec     := 0;
          x.Hundredths := 0;
        END;
      DTS := StrTime(CDD, x);
      Write(DTS);
      REPEAT
        DTS := ReadNumStr(DTS);
        ValTime(CDD, DTS, x);
        Ok  := TimeValid(x);
        IF NOT Ok THEN
          Write(BEL);
      UNTIL Ok;
    END { ReadTimeAbs };
                      

  PROCEDURE ReadDateTimeAbs(VAR CDD : CDI; VAR x : DateTimeWH);

    VAR
      DTS : String32;
      Ok  : BOOLEAN;

    BEGIN { ReadDateTimeAbs }
      IF NOT DateTimeValid(x) THEN
        BEGIN
          GetDateTime(x);
          x.DT.Sec     := 0;
          x.Hundredths := 0;
        END;
      DTS := StrDateTime(CDD, x);
      Write(DTS);
      REPEAT
        DTS := ReadNumStr(DTS);
        ValDateTime(CDD, DTS, x);
        IF x.DT.Year >= 80 THEN
          INC(x.DT.Year, 1900)
        ELSE
          INC(x.DT.Year, 2000);
        Ok := DateTimeValid(x);
        IF NOT Ok THEN
          Write(BEL);
      UNTIL Ok;
      WeekDay(x);
    END { ReadDateTimeAbs };


  FUNCTION DateCmp(VAR a, b : DateTimeWH) : INTEGER;

    VAR
      i : INTEGER;

    BEGIN { DateCmp }
      i := INTEGER(a.DT.Year)-INTEGER(b.DT.Year);
      IF i = 0 THEN BEGIN i := INTEGER(a.DT.Month)-INTEGER(b.DT.Month);
        IF i = 0 THEN BEGIN i := INTEGER(a.DT.Day)-INTEGER(b.DT.Day);
        END;
      END;
      DateCmp := i;
    END { DateCmp };


  FUNCTION TimeCmp(VAR a, b : DateTimeWH) : INTEGER;

    VAR
      i : INTEGER;

    BEGIN { TimeCmp }
      i := INTEGER(a.DT.Hour)-INTEGER(b.DT.Hour);
      IF i = 0 THEN BEGIN i := INTEGER(a.DT.Min)-INTEGER(b.DT.Min);
        IF i = 0 THEN BEGIN i := INTEGER(a.DT.Sec)-INTEGER(b.DT.Sec);
          IF i = 0 THEN BEGIN i := INTEGER(a.Hundredths)-INTEGER(b.Hundredths);
          END;
        END;
      END;
      TimeCmp := i;
    END { TimeCmp };


  FUNCTION DateTimeCmp(VAR a, b : DateTimeWH) : INTEGER;

    VAR
      i : INTEGER;

    BEGIN { DateTimeCmp }
      i := DateCmp(a, b);
      IF i = 0 THEN
        i := TimeCmp(a, b); 
      DateTimeCmp := i;
    END { DateTimeCmp };


  FUNCTION StrCurrency(VAR CDD : CDI; CS, FS  : String32) : String32;

    VAR
      I  : INTEGER;

    BEGIN { StrCurrency }
      WITH CDD DO
        BEGIN
          I := Length(CS)-3;
          WHILE I > 0 DO
            BEGIN
              CS := Copy(CS, 1, I) + StrZ(ThousandsSep) + Copy(CS, I+1,255);
              I  := I - 3;
            END;
          IF CurrencyPlaces > 0 THEN
            CS := CS + StrZ(DecimalSep) +
                  Copy(FS, Length(FS) - CurrencyPlaces + 1, CurrencyPlaces);
          IF 0 IN BitField THEN
            IF 1 IN BitField THEN
              CS := CS + ' ' + StrZ(CurrencySymbol)
            ELSE
              CS := CS + StrZ(CurrencySymbol)
          ELSE
            IF 1 IN BitField THEN
              CS := StrZ(CurrencySymbol) + ' ' + CS
            ELSE
              CS := StrZ(CurrencySymbol) + CS;
        END;
      StrCurrency := CS;
    END { StrCurrency };


  BEGIN { Country }
    CaseMapCallLower := NIL;
  END { Country }.
